divert(-1)

# adit_lotos.m4	(C) K. J. Turner (kjt@cs.stir.ac.uk)	23/10/08
#
# This "m4" macro file generates Clinical Guidance Tree specifications in
# LOTOS format.

# All internal macros start with "adit_". Visible macros are as follows:
#
#  chance	probabilistic, non-deterministic choice
#  comment	comment transcribed to the output
#  decision	deterministic choice
#  question	request for user input
#  terminal	leaf node
#  tree		entire tree
#  value	value (text or expression)

# The following macros may be defined on the command line:
#
#  adit_debug	Error level reporting: 0 all, 1 notes (default), 2 errors,
#		3 panics
#
#  adit_gate	The level of hiding gates (e.g. "-D adit_gate=2") as an "or" of
#		the following:
#		  1 = chance prefix is "i" (default is hidden gate)
#		  2 = user can choose "back" (default is choice of "back")
#
#  adit_indent	The indentation per level is normally 2 spaces, but can be
#		overridden from the command line (e.g. "-D adit_indent=1").
#
#  adit_prefix	As a macro name must not appear in ordinary text, it is
#		desirable to prefix the visible macros with a distinguishing
#		string. Normally this is "adit_", but can be overridden on the
#		command line (e.g. "-D adit_prefix=ADIT_", or
#		"-D adit_prefix=" to use no prefix).

# ADIT expressions are translated into LOTOS as follows. The note operator
# priorities cannot be used so sub-expressions should be parenthesised.
#
#   ADIT				LOTOS
#   if C then E1 else E2 fi	([C]-> E1 [] [not(C)] -> E2)
#   == != < <= > >=		== /= < <= > >=
#   and or not			and or not
#   + - * /			+ - * /
#   true false			1 0
#   defined undefined		defined undefined
#   expression			get(variable,map) or number(sign,whole,fract)
#   variable = expression	Exit(set(variable,expression,map))
#
# The following differences exist in the handling of ADIT expressions:
#
#   + variable names		may contain underscores after the initial letter
#   - no operator priorities	parenthesise as required (left-to-right default)
#   - not			use with parentheses "not(x)" rather than prefix
#   - no combined assignments	no assignments such as "x = y = 3"
#   - discarded variables	payoff variables (starting "pay") and
#				probability variables (starting "prob") are not
#				used; assignments to them are discarded, and an
#				error is issued if they appear in expressions
#   - fixed case		variables, functions and format must have the
#				exact case understood by the code (e.g. "and",
//				"Radio(Yes,No)")

# A number of characters require special consideration:
#
#  //		Use this for comments that should be removed during macro
#		translation. This allows use of hash, which is the usual M4
#		comment symbol.
#
#  ( )		Parentheses are used to delimit macro parameters. As a result,
#		they must be balanced in ordinary text.
#
#  ,		Comma is used to separate macro parameters. As a result, it must
#		not appear unprotected in ordinary text. A comma can be
#		protected by appearing within balanced parentheses or within
#		[[ ]].
#
#  $		Dollar is used for macro parameters. As a result, it must not
#		appear followed by digits. It is suggested that $ be used if
#		this is the case.
#
#  ` '		Single quotes are normally used to quote macro parameters. They
#		are changed to [[ ]] to avoid interference with ordinary text.
#		Since a single quote may appear in the text, it should not be
#		used by any ADIT macros. Instead, it is suggested that
#		McChangeQuote with and  be used for quotes.
#
#  [[ ]]	Double brackets are used to quote macro parameters. As a result,
#		they must not appear in ordinary text.
#
#  		During macro translation, logical not is used temporarily as a
#		separator. As a result, it must not appear in ordinary text. If
#		required, use &not; as the HTML equivalent.
#
#  		Since it is not possible in M4 to match any character including
#		a newline, this is expressed as "not ". As a result, this
#		must not appear in ordinary text.

# ============================== Definition Macros =============================

# ---------------------------------- changecom ---------------------------------
# allow use of hash by changing to // ... newline for comments
# ------------------------------------------------------------------------------

changecom(//)

// -------------------------------- changequote --------------------------------
// allow use of single quotes by changing to [[ ... ]] instead for quoting
// -----------------------------------------------------------------------------

changequote([[,]])

// -------------------------------- adit_debug ---------------------------------
// returns 1/0 for diagnostic output or not: 0 all, 1 notes (default), 2 errors,
// 3 panics; may be overriden from the command line ("-D adit_debug=2")
// -----------------------------------------------------------------------------

define(adit_debug,
 [[1]])

//---------------------------------- adit_gate ---------------------------------
// returns the level of hiding gates: 0 = hidden chance gates/user can go
// "back", 1 = internal choices/user can go "back", 2 = hidden chance gates/
// no "back", 3 = internal choices/no "back" ("-D adit_gate=3")
// -----------------------------------------------------------------------------

ifdef([[adit_gate]],
  ,
  [[define([[adit_gate]],
    [[0]])]])

//-------------------------------- adit_indent ---------------------------------
// returns the number of spaces per indent (e.g. "2"); may be overriden from the
// command line ("-D adit_indent=4")
// -----------------------------------------------------------------------------

ifdef([[adit_indent]],
  ,
  [[define([[adit_indent]],
    [[2]])]])

// ------------------------------- adit_prefix ---------------------------------
// returns the prefix for "external" ADIT macros (default "adit_");
// may be overriden from the command line ("-D adit_prefix=ADIT_")
// -----------------------------------------------------------------------------

ifdef([[adit_prefix]],,
  [[define([[adit_prefix]],
    [[adit_]])]])

// ============================= Working Variables =============================

// macros have temporary working values named after them, e.g. "adit_chance1"

//------------------------------ adit_condition --------------------------------
// returns the value of a condition
// -----------------------------------------------------------------------------

//--------------------------------- adit_else ----------------------------------
// returns the value of an else clause
// -----------------------------------------------------------------------------

//------------------------------ adit_identifier -------------------------------
// returns the identifier of the current node
// -----------------------------------------------------------------------------

define(adit_identifier,
 [[]])

//----------------------------- adit_indent_initial ----------------------------
// returns the initial indentation offset
// -----------------------------------------------------------------------------

define(adit_indent_initial,
  [[0]])

//------------------------------- adit_level -----------------------------------
// returns the current indentation level (0 = top level)
// -----------------------------------------------------------------------------

define(adit_level,
  [[0]])

//-------------------------------- adit_index ----------------------------------
// returns the value of an index
// -----------------------------------------------------------------------------

//-------------------------------- adit_names ----------------------------------
// returns a space-separated list of names (processes, variables)
// -----------------------------------------------------------------------------

define(adit_names,
 [[]])

//-------------------------- adit_parent_identifier ----------------------------
// returns the identifier of the parent node
// -----------------------------------------------------------------------------

define(adit_parent_identifier,
 [[]])

//------------------------------- adit_process ---------------------------------
// returns the current process definition
// -----------------------------------------------------------------------------

//------------------------------ adit_processes --------------------------------
// returns the current list of process definitions
// -----------------------------------------------------------------------------

define(adit_processes,
 [[]])

//------------------------------ adit_questions --------------------------------
// returns a space-separated list of question process names
// -----------------------------------------------------------------------------

define(adit_questions,
 [[]])

//-------------------------------- adit_result ---------------------------------
// returns the current field evaluation result
// -----------------------------------------------------------------------------

//---------------------------- adit_specification ------------------------------
// returns the specification of the entire tree
// -----------------------------------------------------------------------------

define(adit_specification,
 [[]])

//--------------------------------- adit_then ----------------------------------
// returns the value of a then clause
// -----------------------------------------------------------------------------

//--------------------------------- adit_type ----------------------------------
// returns the type of the current node (initially at tree level)
// -----------------------------------------------------------------------------

define(adit_type,
 [[Tree]])

//------------------------------ adit_variables --------------------------------
// returns the current list of variable definitions
// -----------------------------------------------------------------------------

define(adit_variables,
 [[]])

// ============================== General Macros ===============================

// --------------------- adit_diagnostic(string1,...) --------------------------
// report the diagnostic message "string1", etc.
// -----------------------------------------------------------------------------

define(adit_diagnostic,
  [[ifelse(
      eval(adit_debug <= 0),1,
	[[errprint([[]]
adit_convert [error]: line __line__ node 'adit_identifier' - $@
)]])]])

// ------------------------ adit_error(string1,...) ----------------------------
// reports the error message "string1", etc.
// -----------------------------------------------------------------------------

define(adit_error,
  [[ifelse(
      eval(adit_debug <= 2),1,
	[[errprint([[]]
adit_convert [error]: line __line__ node 'adit_identifier' - $@
)]])]])

// ---------------------------- adit_indent_current ----------------------------
// returns indentation for "adit_level"
// -----------------------------------------------------------------------------

// ----------------------------- adit_indent_less ------------------------------
// decrements the current indenting level
// -----------------------------------------------------------------------------

// ----------------------------- adit_indent_more ------------------------------
// increments the current indenting level
// -----------------------------------------------------------------------------

// -------------------------- adit_indent_of(lev) ------------------------------
// returns indentation for the given level "lev"
// -----------------------------------------------------------------------------

define(adit_indent_of,
  [[adit_spaces(eval(adit_indent_initial + $1 * adit_indent))]])

// --------------------------- adit_indent_set(bool) ---------------------------
// enables indentation (initialises macro "adit_level" to 0, defines macros
// "adit_indent_current", "adit_indent_less", "adit_indent_more") or disables
// indentation (undefines macros "adit_indent_current", "adit_indent_less",
// "adit_indent_more") according to whether "bool" is 1 or 0
// -----------------------------------------------------------------------------

define(adit_indent_set,
  [[ifelse($1,1,
    [[define([[adit_level]],
      0)define([[adit_indent_current]],
	[[adit_indent_of(adit_level)]])define([[adit_indent_less]],
	  [[define([[adit_level]],
	    eval(adit_level-1))]])define([[adit_indent_more]],
	      [[define([[adit_level]],
		eval(adit_level+1))]])]],
    [[undefine([[adit_indent_current]])undefine([[adit_indent_less]])undefine(
      [[adit_indent_more]])]])]])

// ----------------------------- adit_lower(text) ------------------------------
// returns "text" initially lower-case
// -----------------------------------------------------------------------------

define(adit_lower,
  [[translit(substr($1,0,1),A-Z,a-z)[[]]substr($1,1)]])

// ---------------------------- adit_lowers(text) ------------------------------
// returns "text" all in lower-case
// -----------------------------------------------------------------------------

define(adit_lowers,
  [[translit($1,A-Z,a-z)]])

// ----------------------- adit_note(string1,...) ---------------------------
// reports the note message "string1", etc.
// -----------------------------------------------------------------------------

define(adit_note,
  [[ifelse(
      eval(adit_debug <= 1),1,
	[[errprint([[]]
adit_convert [warning]: line __line__ node 'adit_identifier' - $@
)]])]])

// ----------------------- adit_panic(string1,...) -----------------------------
// report the panic message "string1", etc.
// -----------------------------------------------------------------------------

define(adit_panic,
  [[ifelse(
      eval(adit_debug <= 3),1,
	[[errprint([[]]
adit_convert [panic]: line __line__ node 'adit_identifier' - $@
)]])]])

// --------------------------- adit_spaces(number) -----------------------------
// returns the number of spaces "number"
// -----------------------------------------------------------------------------

define(adit_spaces,
  [[ifelse(eval($1 < 0),1,[[adit_error(negative space not allowed)]],
    $1,0,,
    [[ adit_spaces(decr($1))]])]])

// --------------------------- adit_trim_spaces(text) --------------------------
// returns "text" without leading or trailing spaces
// -----------------------------------------------------------------------------

define(adit_trim_spaces,
  [[patsubst($1,\s+$)]])

// ----------------------------- adit_upper(text) ------------------------------
// returns "text" initially upper-case
// -----------------------------------------------------------------------------

define(adit_upper,
  [[translit(substr($1,0,1),a-z,A-Z)[[]]substr($1,1)]])

// ---------------------------- adit_uppers(text) ------------------------------
// returns "text" all in upper-case
// -----------------------------------------------------------------------------

define(adit_uppers,
  [[translit($1,a-z,A-Z)]])

// =========================== Internal Tree Macros ============================

// ---------------------------- adit_add_name(name) ----------------------------
// adds the name (if new, disregarding case) to "adit_names" as a space-separated
/* list
// -----------------------------------------------------------------------------

define(adit_add_name,
  [[ifelse(
      regexp(adit_lowers(adit_names),\b[[]]adit_lowers($1)\b),-1,
	[[define([[adit_names]],
	    ifelse(
	      adit_names,,
		$1,
	      [[adit_names $1]]))]])]])

// ------------------------- adit_add_question(process) ------------------------
// adds the process to "adit_questions" as a space-separated list
// -----------------------------------------------------------------------------

define(adit_add_question,
  [[define([[adit_questions]],
      ifelse(
	adit_questions,,
	  $1,
	  [[adit_questions $1]]))]])

// ------------------------- adit_allows_macros(field) -------------------------
// returns 1/0 if the field allows macros or not
// -----------------------------------------------------------------------------

define(adit_allows_macros,
  [[ifelse(
      $1,conjunction,
	1,
      $1,display,
	1,
      $1,error,
	1,
      $1,macros,
	1,
      $1,perform,
	1,
      $1,query,
	1,
      $1,reason,
	1,
	0)]])

// ------------------------------ adit_back -----------------------------------
// returns a call of "adit_parent_identifier" (if non-empty) as an alternative
// choice
// -----------------------------------------------------------------------------

define(adit_back,
  [[ifelse(
      adit_parent_identifier,,
	,
      eval(adit_gate & 2),0,
	[[
  [] (* or *)
    user !Back; (* go back *)
    adit_parent_identifier [user] (map) (* to parent *)]])]])

// ------------------------------ adit_behaviour -------------------------------
// returns a call of the top-level process, preceded by any initialisations
// -----------------------------------------------------------------------------

define(adit_behaviour,
  [[ifelse(
    adit_variables,,
      [[adit_process [user] ({}) (* call top-level process *)]],
      [[adit_variables
      adit_process [user] (map) (* call top-level process *)]])]])

// ------------------- adit_check_field(type,field,label) ----------------------
// returns the "label" corresponding to the "field" provided this matches the
// node "type" ("Question", "Tree", "-" meaning anything else); otherwise an
// error is reported
// -----------------------------------------------------------------------------

define(adit_check_field,
  [[ifelse(
      $1,Question,
	[[ifelse(
	  adit_type,Question,
	    $3,
	    [[adit_error(field '$2' is only for questions)]])]],
      $1,Tree,
	[[ifelse(
	  adit_type,Tree,
	    $3,
	    [[adit_error(field '$2' is only for trees)]])]],
	[[ifelse(
	  adit_type,Tree,
	    [[adit_error(field '$2' is not for trees)]],
	    $3)]])]])

// ---------------------------- adit_field(field) ------------------------------
// returns the tree field name corresponding to the given attribute name
// -----------------------------------------------------------------------------

define(adit_field,
  [[ifelse(
    $1,variables,
      [[adit_check_field(Tree,$1,)]],
    [[ifelse(
      $1,composed,
	[[adit_check_field(Tree,$1,ComposedHeadings)]],
      $1,conjunction,
	[[ifelse(
	    adit_type,Tree,
	      ComposedNodeJoiningText,
	      JoiningText)]],
      $1,conjunction,
	[[adit_check_field(Tree,$1,JoiningText)]],
      $1,dictionary,
	[[adit_check_field(Tree,$1,DictionaryFile)]],
      $1,display,
	[[ifelse(adit_type,Tree,IntroText,UserText)]],
      $1,error,
	[[adit_check_field(Question,$1,ValidationRuleText)]],
      $1,format,
	[[adit_check_field(Question,$1,InputType)]],
      $1,label,
	[[adit_check_field(-,$1,LongLabel)]],
      $1,macros,
	[[adit_check_field(Tree,$1,GlobalMacros)]],
      $1,neutral,
	[[adit_check_field(Tree,$1,NeutralPoint)]],
      $1,payoff,
	[[adit_check_field(-,$1,Payoff)]],
      $1,probability,
	[[adit_check_field(-,$1,Prob)]],
      $1,perform,
	[[adit_check_field(-,$1,InstructionText)]],
      $1,print,
	[[adit_check_field(-,$1,PrintIf)]],
      $1,query,
	[[adit_check_field(Question,$1,QuestionText)]],
      $1,reason,
	[[adit_check_field(-,$1,DocumentingText)]],
      $1,scale,
	[[adit_check_field(-,$1,ScaleIf)]],
      $1,valid,
	[[adit_check_field(Question,$1,ValidationRule)]],
      $1,variable,
	[[ifelse(
	    adit_type,Question,
	      BoundVariable,
	    adit_type,Tree,
	      TreeVariable,
	      [[adit_error(field 'variable' is only for questions)]])]],
      $1,version,
	[[adit_check_field(Tree,$1,RequiresVersion)]],
      $1,visible,
	[[adit_check_field(-,$1,ExistsIf)]],
	[[adit_error(field '$1' is unknown)]]): ]])]])

// ------------------------- adit_has_macros(value) ----------------------------
// returns 1/0 if the field has macros or not
// -----------------------------------------------------------------------------

define(adit_has_macros,
  [[ifelse(
     regexp($1,\<Mc\w+),-1,
       0,
       1)]])

// -------------------- adit_is_composed(type,id,label) ------------------------
// returns 1/0 if the label is/is not for a composed node
// -----------------------------------------------------------------------------

define(adit_is_composed,
  [[ifelse(
      regexp($3,&),-1,
	0,
	[[ifelse(
	  $1,Terminal,
	    [[adit_add_name($2)]],
	    [[define([[adit_identifier]],
		$2)adit_error(only Terminal nodes can be composed)]])1]])]])

// --------------------- adit_is_condition(expression) -------------------------
// returns 1/0 if the expression is a condition or not
// -----------------------------------------------------------------------------

define(adit_is_condition,
  [[ifelse(
      regexp($1,{\|]\s*->),-1,
	0,
	1)]])

// ----------------------- adit_is_empty(value,...) ----------------------------
// returns 1/0 if a single empty value is given
// -----------------------------------------------------------------------------

define(adit_is_empty,
  [[ifelse(
      $1,,
	[[ifelse(
	  $#,1,
	    1,
	    0)]],
	0)]])

// -------------------------- adit_is_ignored(name) ----------------------------
// returns 1/0 if the variable name should/should not be ignored (depending on
// whether it starts "pay" or "prob")
// -----------------------------------------------------------------------------

define(adit_is_ignored,
  [[ifelse(
      index($1,pay),0,
	1,
      index($1,prob),0,
	1,
	0)]])

// ------------------------- adit_is_question(process) -------------------------
// returns 1/0 if the process is a question or not
// -----------------------------------------------------------------------------

define(adit_is_question,
  [[ifelse(
      regexp(adit_questions,\b$1\b),-1,
	0,
	1)]])

// ---------------- adit_node(type,id,behaviour,attributes) --------------------
// defines a process for node "type", identifier "id", "behaviour" and
// "attributes"
// -----------------------------------------------------------------------------

define(adit_node,
  [[define([[adit_type]],
    $1)define([[adit_identifier]],
      $2)define([[adit_parent_identifier]],
	$2)adit_add_name($2)adit_set_attributes($4)adit_translate_process(
	  adit_type,adit_identifier,[[$3]])]])

// --------------------- adit_process_list(process1,...) -----------------------
// lists the given process definitions
// -----------------------------------------------------------------------------

define(adit_process_list,
  [[ifelse(
      $1,,
	[[ifelse($#,1,,[[$1adit_process_list(shift($@))]])]],
	[[$1adit_process_list(shift($@))]])]])

// ------------------ adit_set_attribute(attribute,value) ----------------------
// sets the "attribute" to "value", reporting an error if it contains a "Mc"
// macro call for anything except a text field
// -----------------------------------------------------------------------------

define(adit_set_attribute,
  [[ifelse(
      adit_allows_macros($1),0,
	[[ifelse(
	  adit_has_macros($2),1,
	    [[adit_error(
	      field '$1' may not use macros)]])]])ifelse(
		$1,format,
		  [[adit_set_attribute_aux(Format,[[$2]])]],
		$1,valid,
		  [[adit_set_attribute_aux(Valid,[[$2]])]],
		$1,variable,
		  [[adit_set_attribute_aux(Variable,[[$2]])]],
		$1,variables,
		  [[adit_set_attribute_aux(Variables,[[$2]])]],
		$1,visible,
		  [[adit_set_attribute_aux(Visible,[[$2]])]])]])

define(adit_set_attribute_aux,
  [[ifdef(
      adit_identifier[[]]_$1,
	[[ifelse(
	    regexp([[$2]],_$1),-1,
	      [[adit_error(
		  literal 'adit_lower($1)' value repeats prior definition)]])]],
	[[ifelse(
	    $2,adit_identifier[[]]_$1,
	      [[adit_error(definition needed for 'adit_lower($1)')]],
	      [[adit_value(adit_identifier[[]]_$1,$2)]])]])]])

// -------------------- adit_set_attributes(attributes) ------------------------
// defines node fields for all the space-separated "attributes" defined in the
// format 'field="value"' or 'field' (the latter being short for a value defined
// as '<Id>_Field', i.e. a macro);
// -----------------------------------------------------------------------------

define(adit_set_attributes,
  [[ifelse(
      $1,,
	,
	[[ifelse(
	    regexp($1,^\w+\s*=\s*"[^"]*"\s*[^]*),-1,
	      [[ifelse(
		  regexp($1,^\w+\s*[^]*),-1,
		    [[adit_error(closing quote missing in '$1')]],
		    [[adit_set_attributes(
		      regexp($1,^\(\w+\)\s*\([^]*\),
			[[\1="[[adit_identifier[[]]_[[]]adit_upper(\1)]]" \2]]))]])]],
	  [[regexp($1,^\(\w+\)\s*=\s*"\([^"]*\)"\s*\([^]*\),
	    [[adit_set_attribute(\1,[[\2]])adit_set_attributes(\3)]])]])]])]])

// ----------------------- adit_set_process(behaviour) ------------------------
// sets "adit_process" to the process name
// -----------------------------------------------------------------------------

define(adit_set_process,
  [[define([[adit_process]],
      regexp($1,Process \(\S+\),\1))]])

// --------------------------- adit_translate_accept ---------------------------
// returns an "accept" statement
// -----------------------------------------------------------------------------

define(adit_translate_accept,
  [[>> Accept map:Map In (* get new map *)]])

// --------------- adit_translate_assignment(last,expression) ------------------
// returns the translation of one assignment (or expression), given an
// indication of whether it is last in a sequence or not (1/0)
// -----------------------------------------------------------------------------

define(adit_translate_assignment,
  [[ifelse(
      $2,,
	,
	[[define([[adit_assignment1]],
	  regexp(
	    $2,
	    ^\(\w+\)\s*=\([^=][^;]*\)\(;\|$\),
	    [[ifelse(
		adit_is_condition(\2),1,
		  [[\2
adit_indent_lessadit_indent_currentadit_translate_acceptadit_indent_more
adit_translate_exit(\1)]],
		  [[adit_translate_exit(\1,\2)]])]]))ifelse(
		      adit_assignment1,,
			[[define([[adit_assignment1]],
			    regexp(
			      $2,
			      \([^;]*\)\(;\|$\),
			      [[ifelse(
				  adit_is_condition(\1),1,
				    \1,
				    [[adit_translate_exit(,\1)ifelse(
					$1,1,
					  ,
					  [[adit_note(useless expression '\1' in,
					      sequence)]])]])]]))]])adit_assignment1[[]]]])]])

// --------------------------- adit_translate_begin ----------------------------
// defines a "adit_if" macro that translates conditions
// -----------------------------------------------------------------------------

define(adit_translate_begin,
  [[define([[adit_if]],
      [[define([[adit_condition]],
	  adit_trim_spaces($]]1[[))define([[adit_then]],
	    adit_trim_spaces($]]2[[))define([[adit_else]],
	      adit_trim_spaces($]]3[[))ifelse(
	adit_then[[]]adit_else,,
	  ,
	  [[define([[adit_then]],
	      ifelse(
		adit_then,,
		  adit_translate_exit(,undefined),
		  [[adit_translate_sequence(adit_then,0)]]))define([[adit_else]],
		    ifelse(
		      adit_else,,
			adit_translate_exit(,undefined),
			[[adit_translate_sequence(adit_else,0)]]))
adit_indent_current(
adit_indent_current  [adit_translate_condition(true,adit_condition)] ->
adit_indent_moreadit_indent_moreadit_thenadit_indent_lessadit_indent_less
adit_indent_current[] (* or *)
adit_indent_current  [adit_translate_condition(false,adit_condition)] ->
adit_indent_moreadit_indent_moreadit_elseadit_indent_lessadit_indent_less
adit_indent_current)
]])]])]])

// -------------- adit_translate_condition(boolean,expression) -----------------
// returns a translated an expression according to the boolean value expected
// -----------------------------------------------------------------------------

define(adit_translate_condition,
  [[ifelse(
      $1,true,
	[[(adit_translate_expression($2)) eq 1]],
	[[(adit_translate_expression($2)) eq 0]])]])

// --------------------------- adit_translate_end ------------------------------
// undefines the "adit_if" macro for translating conditions
// -----------------------------------------------------------------------------

define(adit_translate_end,
  [[undefine(
      [[adit_if]])]])

// --------------- adit_translate_exit(variable,expression) --------------------
// translates to an Exit as follows:
//   neither	exit with current map
//   variable	exit with map having variable set to last result
//   expression	exit with map having expression set as last result
//   both	exit with map having variable/last result set to expression
// -----------------------------------------------------------------------------

define(adit_translate_exit,
  [[ifelse(
      $1,,
	[[ifelse(
	    $2,,
	      Exit(Map) (* exit with current map *),
	      [[adit_indent_currentExit(set(adit_translate_expression(
		  $2),map)) (* exit with last result set to "$2" *)]])]],
      $2,,
	[[adit_indent_currentExit(set($1,map)) (* exit with $1 set to last result *)]],
	[[adit_indent_currentExit(set($1,adit_translate_expression(
	    $2),map)) (* exit with $1 set to $2 *)]])]])

// ----------------- adit_translate_expression(expression) ---------------------
// returns the translated expression, with "!="/"false"/"true" turned into
// "/="/"0"/"1", and "variable" turned into "get(variable,map)"
// -----------------------------------------------------------------------------

define(adit_translate_expression,
  [[regexp(
      $1,
      [\^],
      [[adit_diagnostic('^' may be approximated)]])patsubst(
	  patsubst(
	    patsubst($1,!=,/=),
	    \([a-zA-Z]\w*\),
	    [[adit_translate_variable(\1)]]),
	    \([-+]?\(\b[0-9]+\([.][0-9]*\)?\|[.][0-9]+\)\),
	    [[ifelse(
		\1,0,
		  0,
		\1,1,
		  1,
		  adit_translate_number(\1))]])]])

// --------------------- adit_translate_format(format) -------------------------
// returns the translated format
// -----------------------------------------------------------------------------

define(adit_translate_format,
  [[define([[adit_format1]],
      0)define([[adit_format2]],
	patsubst($1,[[\([^,]*\),\([^,]*\)]],))define([[adit_format3]],
	  adit_identifier[[]]_Variable)ifelse(
	    eval(regexp($1,^BloodPressure) != -1),1,
	      [[ifelse(
		  patsubst(
		    adit_format3,
		    \(\w+\)|\(\w+\),
		    [[define([[adit_format4]],
			\1)define([[adit_format5]],
			  \2)]]),,
		      ,
		      [[adit_error(wrong BloodPressure variable 'adit_format3')]])
		user ?systolic:Number; (* get systolic pressure for adit_format4 *)
		user ?diastolic:Number; (* get diastolic pressure for adit_format5 *)
		Exit(set(adit_format4,systolic,set(adit_format5,diastolic,map))) (* exit with answers *)]],
	    eval(regexp($1,^Combo) != -1),1,
	      [[ifelse(
		  regexp($1,[[,]]),-1,
		    [[adit_error(format '$1' needs at least two answers)]])
		(adit_translate_format_aux(1,eval(len(adit_format2) + 1))
		)]],
	    eval(regexp($1,^Edit) != -1),1,
	      [[
		user ?number:Number; (* get number for adit_identifier[[]]_Variable *)
		Exit(set(adit_identifier[[]]_Variable,number,map)) (* exit with answer *)]],
	    eval(regexp($1,^HeightScale) != -1),1,
	      [[
		user ?height:Number (* get height for adit_identifier[[]]_Variable *)
		  [(number(+,t(1),t(5)) le height) and
		   (height le number(+,t(2),<>))];
		Exit(set(adit_identifier[[]]_Variable,height,map)) (* exit with answer *)]],
	    eval(regexp($1,^Radio) != -1),1,
	      [[ifelse(
		regexp($1,[[,]]),-1,
		  [[adit_error(format '$1' needs at least two answers)]])
		(adit_translate_format_aux(0,len(adit_format2))
		)]],
	    eval(regexp($1,^WeightScale) != -1),1,
	      [[
		user ?weight:Number (* get weight for adit_identifier[[]]_Variable *)
		  [(number(+,t(4)~0,<>) le weight) and
		   (weight le number(+,t(1)~4~5,<>))];
		Exit(set(adit_identifier[[]]_Variable,weight,map)) (* exit with answer *)]],
	      [[adit_error(unrecognised format '$1')]])]])

define(adit_translate_format_aux,
  [[ifelse(
      eval($1 > $2),1,
	,
	[[ifelse(
	    adit_format1,1,
	      [[
		[] (* or *)]])define([[adit_format1]],
	  1)
		  user !$1 Of Number; (* get $1 for adit_identifier[[]]_Variable *)
		  Exit(set(adit_identifier[[]]_Variable,$1,map)) (* exit with answer *)adit_translate_format_aux(
	  incr($1),$2)]])]])

// -------------------- adit_translate_if(expression) --------------------------
// returns an expression with "{ C ? E1 : E2 }" turned into
// "adit_if(C,E1,E2)"
//
// temporary variables: 1 = translated "if"

// -----------------------------------------------------------------------------
define(adit_translate_if,
  [[ifelse(
    $1,,
      ,
      [[define([[adit_if1]],
	patsubst($1,
	  {\([^?]+\)\?\([^{:}]*\)\(:\([^{}]*\)\)?},
	  [[adit_if(
	      adit_trim_statement(
		\1),adit_trim_assignment(
		  \2),adit_trim_assignment(
		    \4))]]))ifelse(
		      adit_if1,$1,
			adit_if1,
			[[adit_translate_if(adit_if1)]])]])]])

// ------------ adit_translate_initialisations(name1=value1;...) ---------------
// returns tree variable declarations "set(variable,value,map)" in
// "adit_variables" for pairs of "name" and optional initial "value"
// -----------------------------------------------------------------------------

define(adit_translate_initialisations,
  [[adit_translate_initialisations_aux($1)ifelse(
      adit_variables,,
	,
	[[define([[adit_variables]],
	Let map:Map = (* variable initialisations *)adit_variables In)]])]])

define(adit_translate_initialisations_aux,
  [[patsubst(
      $1,
      \(\w+\)\s*\(=\s*\([^;]+\)\)?;?,
      [[ifelse(
	  adit_is_ignored(\1),0,
	    [[adit_add_name(\1)ifelse(
		\3,,
		  ,
		  [[define([[adit_variables]],[[
    ]]set(\1,adit_translate_number(\3),ifelse(
		    adit_variables,,
		      {},
		      adit_variables)))]])]])]])]])

// --------------------------- adit_translate_names ----------------------------
// returns the LOTOS representation of the name list "adit_names"
// -----------------------------------------------------------------------------

define(adit_translate_names,
  [[Type Variable Is TextOps (* variable/process names *)
  Opns (* operations *)
    patsubst(adit_names,\s,[[,
    ]]): -> Text
  Eqns (* equations *)
    OfSort Text (* text operations *)patsubst(
  adit_names,
  \(\S+\)\s?,
  [[
      \1 = adit_translate_string(\1);]])
EndType (* end Variable *)]])

// ---------------------- adit_translate_number(number) ------------------------
// returns the LOTOS representation of the decimal "number" (e.g.
// "number(+,t(3)~5,4)" for "35.4")
// -----------------------------------------------------------------------------

define(adit_translate_number,
  [[define([[adit_sign]],substr($1,0,1))ifelse(
    adit_sign,+,
      [[define([[adit_number]],substr($1,1))]],
    adit_sign,-,
      [[define([[adit_number]],substr($1,1))]],
      [[define([[adit_sign]],+)define([[adit_number]],$1)]])define([[adit_index]],
	index(adit_number,.))ifelse(
	  adit_index,-1,
	    [[define([[adit_whole]],adit_number)define([[adit_fraction]],)]],
	    [[define([[adit_whole]],
	      substr(adit_number,0,adit_index))define([[adit_fraction]],
		substr(adit_number,
		  incr(adit_index)))]])number(adit_sign,adit_translate_string(
		    adit_whole),adit_translate_string(
		      adit_fraction))]])

// --------------- adit_translate_process(type,id,behaviour) -------------------
// defines a process for node type "type", identifier "id", and "behaviour"
//
// temporary variables: 1 = process heading
// -----------------------------------------------------------------------------

define(adit_translate_process,
  [[define([[adit_process1]],
      regexp(
	[[$2]],
	^\(.+\)\(Valid\|Visible\)$,
	[[Process $2 (map:Map) : Exit(map) := (* $1 \1 *)]]))ifelse(
	  adit_process1,,
	    [[define([[adit_process1]],
	      Process $2 [user] (map:Map) : Exit(Map) := (* $1 $2 *))]])adit_process1
$3
EndProc (* end $2 *)

]])

// -------------------- adit_translate_result(expression) ----------------------
// returns the setting of property "adit_type" in "expression"
// -----------------------------------------------------------------------------

define(adit_translate_result,
  [[define([[adit_result1]],
      regexp(
	adit_process,
	^\(.+\)\(Valid\|Visible\)$,
	\1))$1
adit_translate_accept
  Exit(set[[]]adit_value1[[]](adit_result1,map)) (* set adit_value1 for adit_result1 *)]])

// ----------------- adit_translate_sequence(expression,trim) ------------------
// returns the translation of "expression" as a sequence of sub-expressions
// separated/terminated by semicolons, omitting the sequence "true" if
// "trim" is 1 (the default if this parameter is omitted)
//
// temporary variables: 1 = current sub-expression is the last
// -----------------------------------------------------------------------------

define(adit_translate_sequence,
  [[ifelse(
      $1,true,
	[[ifelse(
	    $2,0,
	      adit_translate_assignment(1,true))]],
	adit_translate_sequence_aux($1))]])

define(adit_translate_sequence_aux,
  [[ifelse(
    $1,,
      ,
      [[regexp(
	  $1,
	  \([^;]+\);?\([^]*\),
	  [[define([[adit_sequence1]],
	      ifelse(
		[[\2]],,
		  1,
		  0))adit_translate_assignment(adit_sequence1,
		    adit_trim_assignment(\1))ifelse(
		      adit_sequence1,1,
			,
			[[
adit_indent_lessadit_indent_currentadit_translate_acceptadit_indent_more
adit_translate_sequence_aux(\2)]])]])]])]])

// ---------------------- adit_translate_string(string) ------------------------
// returns a translated string in the form "t(C1)~C2~...", removing forbidden
// characters "$'(),#:;?![]_`|^; comma {,} and hash {#} must be quoted in the
// input, and parentheses must be balanced or quoted; to avoid very long
// translated strings, a period in the input is followed by a newline
// -----------------------------------------------------------------------------

define(adit_translate_string,
  [[ifelse(
    len($1),0,
      [[<>]],
      [[define([[adit_string1]],
	translit([[$1]],
	  [["$'(),#:;?![]_`|^]]))define([[adit_string1]],
	    translit(adit_string1,
	      [[ ]],^))define([[adit_string1]],
		patsubst(adit_string1,
		  \([-%^&*+[[]]@~\<>/]\),
		  ~ \1 ))define([[adit_string1]],
		    patsubst(adit_string1,
		      \([a-hj-zA-HJ-Z0-9]\),
		      ~\1))define([[adit_string1]],
			patsubst(adit_string1,
			  \([iI]\),
			  ~\1\1))define([[adit_string1]],
			    patsubst(adit_string1,
			      \([=]\),
			      ~ \1\1 ))define([[adit_string1]],
				patsubst(adit_string1,
				  \([.]\),
				  ~ \1
  ))define([[adit_string1]],
    substr(adit_string1,1))patsubst(adit_string1,
      ^\(\w+\| \W \),t(\1))]])]])

// -------------------- adit_translate_validity(process) -----------------------
// returns a check of "process" validity as required
// -----------------------------------------------------------------------------

define(adit_translate_validity,
  [[ifelse(
      adit_validity($1),1,
	[[
		$1Valid (map)
	      adit_translate_accept
		(
		  [valid($1,map)] -> (* valid answer? *)
		    Exit(satisfy($1,map)) (* exit with question satisfied *)
		[] (* or *)
		  [not(valid($1,map))] -> (* invalid answer? *)
		    $1Aux [user] (map) (* repeat question *)
		)]],
	[[
		Exit(satisfy($1,map)) (* exit with question satisfied *)]])]])

// ---------------------- adit_translate_variable(name) ------------------------
// returns the translation of "name" as the original name if a keyword, 0/1 for
// false/true, or a variable access
// -----------------------------------------------------------------------------

define(adit_translate_variable,
  [[ifelse(
    $1,and,
      $1,
    $1,defined,
      $1,
    $1,exp,
      [[adit_diagnostic('exp' may be approximated)$1]],
    $1,false,
      0 Of Number,
    $1,log,
      [[adit_diagnostic('log' may be approximated)$1]],
    $1,not,
      $1,
    $1,or,
      $1,
    $1,true,
      1 Of Number,
    $1,undefined,
      $1,
      [[ifelse(
	  adit_is_ignored($1),1,
	    [[adit_error(
	      variable '$1' is not allowed in an expression)]])get($1,map)]])]])

// ----------------- adit_translate_visibility(process1,..) --------------------
// returns sequential calls of any visibility processes for the given process
// list
// -----------------------------------------------------------------------------

define(adit_translate_visibility,
  [[define([[adit_visibility1]],
      0)adit_translate_visibility_aux($@)]])

define(adit_translate_visibility_aux,
  [[ifelse(
      $1,,
	[[ifelse($#,1,,[[adit_translate_visibility_aux(shift($@))]])]],
	[[adit_set_process($1)define([[adit_visibility2]],
	    adit_visibility(adit_process))ifelse(
	      adit_visibility2,2,
		[[ifelse(
		    adit_visibility1,0,
		      ,
		    [[
  adit_translate_accept]])
    adit_process[[]]Visible (map) (* check adit_process visibility *)define([[adit_visibility1]],
	1)]])adit_translate_visibility_aux(shift($@))]])]])

// ------------------- adit_trim_assignment(expression) ------------------------
// returns "expression" without assignments to variables whose names start "pay"
// (payoff) or "prob" (probability)
// -----------------------------------------------------------------------------

define(adit_trim_assignment,
  [[adit_trim_statement(adit_trim_assignment_aux($1))]])

define(adit_trim_assignment_aux,
 [[patsubst(
     $1,
     \s*\b\(\w+\)\s*=\s*\([^=][^;:}]*\)\(;?\),
     [[ifelse(
       adit_is_ignored(\1),0,
	 [[\1=\2\3]])]])]])

// ---------------------- adit_trim_comment(expression) ------------------------
// returns the expression with "//" comments removed up to but not including
// their trailing newline
// -----------------------------------------------------------------------------

define(adit_trim_comment,
 [[patsubst(
     $1,
     [[//.*]])]])

// ------------------------- adit_trim_if(expression) --------------------------
// returns the expression with "if..then..else..fi" turned into "{..?..:..}"
// -----------------------------------------------------------------------------

define(adit_trim_if,
 [[patsubst(
     patsubst(
       patsubst(
	 patsubst(
	   $1,
	   \bif\b,
	   {),
	 \bthen\b,
	 ?),
       \b\else\b,
       :),
     \bfi\b,
     })]])

// ------------------------ adit_trim_statement(text) --------------------------
// returns "text" without leading, trailing or duplicated semicolons, and
// without leading or trailing spaces
// -----------------------------------------------------------------------------

define(adit_trim_statement,
  [[adit_trim_spaces(
      patsubst(
	patsubst(
	  patsubst($1,\(\s*;\s*\)+,;),
	  \s*;\s*$),
	^\s*;\s*))]])

// -------------------------- adit_validity(process) ---------------------------
// returns 1/0 if the "<process>_Valid" indicates if a process needs/does not
// need a validity check (i.e. this macro is defined and is non-empty)
// -----------------------------------------------------------------------------

define(adit_validity,
  [[ifdef([[$1_Valid]],
      [[ifelse(
	$1_Valid,,
	  0,
	  1)]],
      0)]])

// ------------------------- adit_visibility(process) --------------------------
// uses the "<process>_Visible" macro to return:
//    0	process is invisible (visibility is false)
//    1	process needs no visibility check (macro is empty or undefined)
//    2	process needs a visibility check (macro has a non-empty/defined value)
// -----------------------------------------------------------------------------

define(adit_visibility,
  [[ifdef([[$1_Visible]],
      [[ifelse(
	$1_Visible,,
	  1,
	regexp(
	  $1_Visible,
	  [[Exit(set(0 Of Number,map)) (\* exit with last]]),-1,
	    2,
	    0)]],
      1)]])

// ============================ Visible Tree Macros ============================

// ------------- adit_chance(id,label,attributes,process1,..) ------------------
// returns a chance process with identifier "id", short label "label",
// "attributes", and processes "process1" etc. as deterministic (user-decided)
// alternatives
//
// temporary variables: 1 = sub-processes, 2 = sub-process visibility calls,
// 3 = later sub-process?, 4 = current sub-process visibility
// -----------------------------------------------------------------------------

define(adit_prefix[[]]chance,
  [[ifelse(
      adit_is_composed(Chance,$1,$2),0,
	[[define([[adit_chance3]],
	  0)ifelse(eval($# < 4),1,
	    [[adit_error(too few parameters for [[[[chance]]]])]],
	    [[define([[adit_chance1]],
	      [[shift(shift(shift($@)))]])adit_node(
		  Chance,$1,
		    [[  user !$1;]] (* enter node *)define([[adit_chance2]],
		      [[adit_translate_visibility(adit_chance1)]])ifelse(
			adit_chance2,,
			  ,
			  [[
  (adit_chance2
  )
adit_translate_accept]])
  (
adit_chance_aux(adit_chance1)[[[[adit_back]]]]
  ),$3)define([[adit_processes]],
adit_processes[[]]adit_process_list(adit_chance1))]])]])]])

define(adit_chance_aux,
  [[ifelse(
    $1,,
      [[ifelse($#,1,,[[adit_chance_aux(shift($@))]])]],
      [[adit_set_process($1)define([[adit_chance4]],
	  adit_visibility(adit_process))ifelse(
	    adit_chance4,0,
	      ,
	      [[ifelse(
		  adit_chance3,1,
		    [[
  [] (* or *)
]])ifelse(
      adit_chance4,2,
	[[    [visible(adit_process,map)] -> (* adit_process visible? *)
      (ifelse(eval(adit_gate & 1),0,
	  [[
	Hide adit_lowers(adit_process) In (* hide internal choice gate *)
	  adit_lowers(adit_process); (* system choice *)
	  adit_process [user] (map) (* to adit_process *)]],
	  [[
	i; (* system choice *)
	adit_process [user] (map) (* to adit_process *)]])
      )]],
	[[ifelse(eval(adit_gate & 1),0,
	  [[    (
      Hide adit_lowers(adit_process) In (* hide internal choice gate *)
	adit_lowers(adit_process); (* system choice *)
	adit_process [user] (map) (* to adit_process *)
    )]],
	  [[    i; (* system choice *)
    adit_process [user] (map) (* to adit_process *)]])]])define([[adit_chance3]],
      1)]])adit_chance_aux(shift($@))]])]])

// --------------------------- adit_comment(text) ------------------------------
// returns a tree comment
// -----------------------------------------------------------------------------

define(adit_prefix[[]]comment,
  [[(* $1 *)]])

// ------------- adit_decision(id,label,attributes,process1,..) ----------------
// returns a decision process with identifier "id", short label "label",
// "attributes", and processes "process1" etc. as deterministic (user-decided)
// alternatives
//
// temporary variables: 1 = sub-processes, 2 = sub-process visibility calls,
// 3 = later sub-process?, 4 = current sub-process visibility
// -----------------------------------------------------------------------------

define(adit_prefix[[]]decision,
  [[ifelse(
      adit_is_composed(Decision,$1,$2),0,
	[[define([[adit_decision3]],
	    0)ifelse(eval($# < 4),1,
	      [[adit_error(too few parameters for [[[[decision]]]])]],
	      [[define([[adit_decision1]],
		[[shift(shift(shift($@)))]])adit_node(
		    Decision,$1,
		      [[  user !$1;]] (* enter node *)define([[adit_decision2]],
			[[adit_translate_visibility(adit_decision1)]])ifelse(
			  adit_decision2,,
			    ,
			    [[
  (adit_decision2
  )
adit_translate_accept]])
  (
adit_decision_aux(adit_decision1)[[[[adit_back]]]]
  ),$3)define([[adit_processes]],
adit_processes[[]]adit_process_list(adit_decision1))]])]])]])

define(adit_decision_aux,
  [[ifelse(
    $1,,
      [[ifelse($#,1,,[[adit_decision_aux(shift($@))]])]],
      [[adit_set_process($1)define([[adit_decision4]],
	  adit_visibility(adit_process))ifelse(
	    adit_decision4,0,
	      ,
	      [[ifelse(
		  adit_decision3,1,
		    [[
  [] (* or *)
]])ifelse(
      adit_decision4,2,
	[[    [visible(adit_process,map)] -> (* adit_process visible? *)
      (
	adit_process [user] (map) (* to adit_process *)
      )]],
	[[    adit_process [user] (map) (* to adit_process *)]])define([[adit_decision3]],
      1)]])adit_decision_aux(shift($@))]])]])

// ------------- adit_question(id,label,attributes,process1,..) ----------------
// returns a question node with identifier "id", short label "label",
// "attributes", and optional processes "process1", etc. subordinate to the
// question
//
// temporary variables: 1 = sub-processes, 2 = sub-process visibility calls,
// 3 = later sub-process?, 4 = current sub-process visibility,
// 5 = sub-process calls
// -----------------------------------------------------------------------------

define(adit_prefix[[]]question,
  [[ifelse(
      adit_is_composed(Question,$1,$2),0,
	[[define([[adit_question3]],
	  0)ifelse(eval($# < 3),1,
	    [[adit_error(too few parameters for [[[[question]]]])]],
	    [[define([[adit_question1]],
	      [[shift(shift(shift($@)))]])adit_add_question(
		$1)adit_node(
		  Question,$1,
		    define([[adit_question2]],
		      adit_translate_visibility(adit_question1))ifelse(
			adit_question2,,
			  ,
			[[  (adit_question2
  )
adit_translate_accept
]])  $1Aux [user] (map) (* ask question *)
adit_translate_accept
  (
    [backing(map)] -> (* going back? *)
      [[[[adit_parent_identifier]]]] [user] (unback(map)) (* to parent, not going back *)
  [] (* or *)
    [not(backing(map))] -> (* not going back? *)ifelse(
	adit_is_empty(adit_question1),1,
	  [[
      Exit({}) (* exit leaf node *)]],
	  [[
      (
	Let map:Map = unskip(map) In (* stop skipping questions *)
	  (define([[adit_question5]],
	     adit_question_aux(adit_question1))ifelse(
	       adit_question5,,
		 [[
	    adit_translate_exit]],
		 [[adit_question5]])
	  )
      )]])
  )

Where (* local definitions *)

  Process $1Aux [user] (map:Map) : Exit(Map) := (* Auxiliary $1 *)
    [skipping(map)] -> (* skipping all questions? *)
      Exit(map) (* exit question *)
  [] (* or *)
    [not(skipping(map))] -> (* not skipping all questions? *)
      (
	user !$1; (* enter node *)
	(
	  [satisfied($1,map)] -> (* question satisfied? *)
	    user !Preserve; (* ask to preserve current answer *)
	    (
	      user !Keep; (* keep current answer *)
	      Exit(map) (* exit question *)
	    [] (* or *)
	      user !Change; (* change current answer *)
	      $1Aux [user] (unsatisfy($1,map)) (* ask question again *)
	    )
	[] (* or *)
	  [not(satisfied($1,map))] -> (* question unsatisfied? *)
	    user !Query; (* prompt for answer *)
	    (
	      user !Skip; (* skip question *)
	      Exit(map) (* exit question with no answer *)
	    [] (* or *)
	      user !SkipAll; (* skip all questions *)
	      Exit(skip(map)) (* exit question, skipping all others *)
	    [] (* or *)
	      ($1_Format
	      adit_translate_accept[[]][[adit_translate_validity($1)]]
	      )
	    )
	)ifelse(
	  eval(adit_gate & 2),0,
	    [[
      [] (* or *)
	user !Back; (* go back *)
	Exit(back(map)) (* exit question, going back *)]])
      )
  EndProc (* end of $1Aux *)
,$3)define([[adit_processes]],
adit_processes[[]]adit_process_list(adit_question1))]])]])]])

define(adit_question_aux,
  [[ifelse(
      $1,,
	[[ifelse($#,1,,[[adit_question_aux(shift($@))]])]],
	[[adit_set_process($1)define([[adit_question4]],
	    adit_visibility(adit_process))ifelse(
	      adit_is_question(adit_process),0,
	      [[ifelse(
		$2,,
		  ,
		  [[ifelse(
		      adit_is_question(regexp($2,Process \(\S+\),\1)),1,
			[[adit_error(questions must precede non-questions)]],
			[[adit_error(at most one non-[[question]] child allowed)]])]])]])ifelse(
		      adit_question4,0,
			,
			[[ifelse(
			    adit_question3,1,
			  [[          adit_translate_accept[[]]]])ifelse(
      adit_question4,2,
	[[
	    (
	      [visible(adit_process,map)] -> (* adit_process visible? *)
		adit_process [user] (map) (* to adit_process *)
	    [] (* or *)
	      [not(visible(adit_process,map))] -> (* adit_process invisible? *)
		adit_translate_exit
	    )
]],
	[[
	    adit_process [user] (map) (* to adit_process *)ifelse(
      $2,,
	,
	[[
]])]])define([[adit_question3]],
  1)]])adit_question_aux(shift($@))]])]])

// ------------------ adit_terminal(id,label,attributes) -----------------------
// returns a terminal node with identifier "id", short label "label", and
// "attributes"
// -----------------------------------------------------------------------------

define(adit_prefix[[]]terminal,
  [[ifelse(eval($# < 2),1,
    [[adit_error(too few parameters for [[[[terminal]]]])]],
    [[ifelse(
	adit_is_composed(Terminal,$1,$2),1,
	  ,
	  [[adit_node(
      Terminal,$1,
      [[  user !$1;]] (* enter $1 *)
  (
    Exit({}) (* exit leaf node *)[[[[adit_back]]]]
  ),$3)]])]])]])

// ----------------- adit_tree(id,label,attributes,process) --------------------
// returns a tree specification with ame "label", "attributes" and "process"
// -----------------------------------------------------------------------------

define(adit_prefix[[]]tree,
  [[define([[adit_type]],
      Tree)define([[adit_identifier]],
	$1)define([[adit_label]],
	  $2)adit_set_attributes($3)adit_set_process($4)define(
	    [[adit_back]])define([[adit_processes]],
	      adit_processes[[]]$4)ifelse(
		$#,4,
		  [[define([[adit_tree1]],
		    [[include(adit_lotos.lot)]])adit_tree1]],
		    [[adit_error(
			incorrect parameter count for [[[[tree]]]])]])]])

// ----------------------- adit_value(name,expression) ------------------------
// defines an expression attribute to have name "name" with the translation into
// LOTOS of "expression"; comments, assignments to payoff/probability variables,
// empty conditions and an isolated "true" condition are eliminated in the
// translation; an expression is ignored unless its name ends with "_Format",
// "_Valid", "_Variable", "_Variables" or "_Visible"
//
// temporary variables: 1 = type, 2 = translation result, identifier =
// current process identifier
// -----------------------------------------------------------------------------

define(adit_prefix[[]]value,
  [[define([[adit_process]],
      regexp(
	[[$1]],
	^\(.+\)_\(Format\|Valid\|Variable\|Variables\|Visible\)$,
	[[define([[adit_identifier]],\1)define([[adit_value1]],\2)\1\2]]))ifdef(
	  [[$1]],
	  [[adit_error('adit_value1' definition already exists)]])ifelse(
	    adit_process,,
	      ,
	      [[ifelse(
		  adit_value1,Format,
		    [[define([[$1]],
			adit_translate_format($2))]],
		  adit_value1,Variable,
		    [[define([[$1]],
			$2)]],
		  adit_value1,Variables,
		    [[define([[$1]],
			adit_translate_initialisations(adit_trim_comment($2)))]],
		    [[define([[adit_value2]],
			adit_translate_if(
			  adit_trim_assignment(adit_trim_if(adit_trim_comment(
			    $2)))))adit_translate_begin[[]]define([[adit_value2]],
			      adit_indent_moreadit_translate_sequence(
				adit_trim_statement(
				  adit_value2))adit_indent_less)adit_translate_end(
				    )adit_indent_set(
					1)define([[adit_value2]],
					  translit(adit_value2,))adit_indent_set(
					    0)ifelse(
					      adit_value2,,
						[[define([[$1]])]],
						[[define([[adit_value2]],
						  adit_translate_result(
						    adit_value2))define([[$1]],
						      adit_value2)define(
							[[adit_processes]],
							adit_processes[[]]adit_translate_process(
							    adit_value1,adit_process,
							      adit_value2))]])]])]])]])

// ================================== Trailer ==================================

divert

