// ChordTutor.java	 (c) Y. P. Tay, K. J. Turner	04/08/03

// Main class for chord tutor

import waba.fx.*;
import waba.sys.*;
import waba.util.*;
import waba.ui.*;

public class ChordTutor extends MainWindow {
  // class variables
  private Score Score;
  private ChordDB ChordDB;

  private int page, lastX, lastY, lastCaretX, noteSelected, startDistance;
  private int chordIndex = -1, keyIndex = 0, pageCount = 1;
  private int vecIndex = -1, keySignIndex = -1, numNotes = 0;
  private Graphics g;
  private boolean redraw, drawKeySignature=false;
  private String[] itemChord, intervalArray;
  private String chordDetails, chordStructDetails, chordName, chordAKA, chordType,
    chordIntervals;
  private String selectedKey, intervals, chordNotes, accidentals = "0";
  public String songTitle;
  public String keySignature;
  private Vector notesVec;
  private MenuBar mbar;
  private MessageBox overwriteBox =
    new MessageBox ("Warning", "Score Exists|Overwrite?",
      new String[] {"OK", "Cancel"});
  private MessageBox successBox =
    new MessageBox ("Information", "Save Successful");

  // quiz variables
  private int correct=0, inCorrect=0, total=0;
  private String[] questionArray = new String[3];
  private int answerNum;
  MessageBox quizbox;
  Button firstAns = new Button ("");
  Button secondAns = new Button ("");
  Button thirdAns = new Button ("");

  // page constants
  private static final int NO_PAGE = 0;
  private static final int MAIN_PAGE = 1;
  private static final int SCORE_PAGE = 2;
  private static final int CHORDDB_PAGE = 3;
  private static final int CHORDQUIZ_PAGE = 4;
  private static final int SCORE_PREFERANCE = 5;
  private static final int HELP_PAGE = 5;

  private static final int NOTEDISTANCE = 15;

  private static final Color MESSAGE_COLOUR = new Color (80, 160, 240);

  // main page
  Button mainExit = new Button ("Exit");
  Button mainScore = new Button ("Score");
  Button mainChordDB = new Button ("Chord Database");
  Button mainChordQuiz = new Button ("Chord Quiz");

  // score page
  Edit scoreEdit = new Edit ();
  Button scoreDone = new Button ("Done");
  Button scoreSave = new Button ("Save");
  Button scorePlay = new Button ("Play");
  Button left = Button.createArrowButton (g.ARROW_LEFT, 5, Color.BLACK);
  Button right = Button.createArrowButton (g.ARROW_RIGHT, 5, Color.BLACK);
  ComboBox keySignatureBox;

  // about/help text
  private MessageBox aboutMessage = new MessageBox (
    "About ChordTutor",
    "CHORD AND SCORE ENTRY||" +
    "Yin Ping Tay (original developer)| <keyle_t@hotmail.com>|" +
    "Kenneth J. Turner (maintainer)| <kjt@cs.stir.ac.uk>|" +
    "www.cs.stir.ac.uk/~kjt/software/|  pda/chordtutor.html||" +
    "Version 1.0 (C) 4th August 2003|" +
    "GPL Licence: no fee, no warranty!|" +
    "The caret at the top indicates|" +
    "the current note position. Notes|" +
    "are normally added at the end of|" +
    "the current list. If necessary,|" +
    "a score can extend over several|" +
    "pages. Click the left/right|" +
    "buttons to move between pages.||" +
    "To enter a note, click a|" +
    "duration symbol then click a|" +
    "position on the stave.|" +
    "To add an accidental, click a note|" +
    "then click an accidental symbol.||" +
    "The following are available from|" +
    "the menu or by keystroke:||" +
    "A (About ChordTutor): select About|" +
    "C (Clear Score): select Clear|" +
    "D (Delete Note): click an existing|" +
    "note, select Delete|" +
    "I (Insert Note): click an existing|" +
    "note, click new position on the|" +
    "stave, select Insert to append|" +
    "L (Load Score): enter a title into|" +
    "the text box, select Load|" +
    "S (Save Score): enter a title into|" +
    "the text box, select Save||||||" +
    "Created with SuperWaba||http://www.superwaba.org/||" +
    "See the software license at|this URL, or email|" +
    "<guich@superwaba.org>",
    new String[] {"OK"});

  // chord database page
  ComboBox chordDatabaseKey;
  ComboBox chordDatabaseChord;
  Button chordDatabaseDone = new Button ("Done");
  Button chordDatabasePlay = new Button ("Play");
  ListBox chordDatabaseDetails = new ListBox ();

  // chord quiz page
  Label chordQuizTitle = new Label ("Chord Quiz");
  Button chordQuizDone = new Button ("Done");
  Button chordQuizPlay = new Button ("Play");
  Button chordQuizNext = new Button ("Next");
  ComboBox chordQuizChord;

  public ChordTutor () {
    // initialisation
    keySignature = "C Major";
    chordDetails = "";
    chordNotes = "";
    chordStructDetails = "";

    Score = new Score (this);
    ChordDB = new ChordDB ();
    g = new Graphics (this);

    page = NO_PAGE;
    redraw = true;
    startDistance = 20;

    itemChord = new String[CreateChords.CHORDS];
    intervalArray = null;
    notesVec = new Vector ();
    String[] keyArray = {
      "C Major", "D Major", "E Major", "F Major", "G Major", "A Major",
      "B Major", "Bb Major", "Eb Major", "Ab Major", "Db Major", "C# Major",
      "F# Major"};

    String itemKey[] = {
      "C","C#/Db","D","D#/Eb","E","F","F#/Gb","G","G#/Ab","A","A#/Bb","B"};

    itemChord[0] = "Major"; itemChord[1] = "5";       itemChord[2] = "-5";
    itemChord[3] = "6";     itemChord[4] = "6/9";     itemChord[5] = "7";
    itemChord[6] = "Add9";  itemChord[7] = "Maj7";    itemChord[8] = "Maj7+5";
    itemChord[9] = "Maj9";  itemChord[10] = "Maj11";  itemChord[11] = "Maj13";
    itemChord[12] = "m";    itemChord[13] = "m6";     itemChord[14] = "m6/9";
    itemChord[15] = "mMaj7";itemChord[16] = "mMaj9";  itemChord[17] = "mAdd9";
    itemChord[18] = "m7";   itemChord[19] = "m9";     itemChord[20] = "m11";
    itemChord[21] = "m13";  itemChord[22] = "m-5";    itemChord[23] = "Dim";
    itemChord[24] = "m7-5"; itemChord[25] = "7";      itemChord[26] = "7-9";
    itemChord[27] = "7+9";  itemChord[28] = "7-5";    itemChord[29] = "7+5";
    itemChord[30] = "7/6";  itemChord[31] = "9";      itemChord[32] = "9-5";
    itemChord[33] = "9+5";  itemChord[34] = "9/6";    itemChord[35] = "9+11";
    itemChord[36] = "11";   itemChord[37] = "11-9";   itemChord[38] = "13";
    itemChord[39] = "13-9"; itemChord[40] = "13-9-5"; itemChord[41] = "13-9+11";
    itemChord[42] = "13+11";itemChord[43] = "7/13";   itemChord[44] = "Aug";
    itemChord[45] = "Sus2"; itemChord[46] = "Sus4";   itemChord[47] = "7Sus4";
    itemChord[48] = "-9";   itemChord[49] = "-9+5";   itemChord[50] = "-9+11";
    itemChord[51] = "-9-5"; itemChord[52] = "+5";     itemChord[53] = "+9";
    itemChord[54] = "+11";

    chordDatabaseKey = new ComboBox (itemKey);
    chordDatabaseChord = new ComboBox (itemChord);
    chordQuizChord = new ComboBox (itemChord);
    keySignatureBox = new ComboBox (keyArray);

    // main page widgets
    mainScore.setRect (CENTER, 40, 35, 15);
    mainChordDB.setRect (CENTER, 70, 80, 15);
    mainChordQuiz.setRect (CENTER, 100, 55, 15);
    mainExit.setRect (138, 143, 20, 15);

    // score page widgets
    scoreEdit.setRect (40, 143, 50, 15);
    scoreDone.setRect (132, 143, 26, 15);
    scoreSave.setRect (108, 143, 24, 15);
    scorePlay.setRect (1, 143, 23, 15);
    left.setRect (10, 85, 10, 10);
    right.setRect (150, 85, 10, 10);
    keySignatureBox.setRect (100, 0, 60, 15);

    // chord database page widgets
    chordDatabaseDone.setRect (132, 143, 26, 15);
    chordDatabasePlay.setRect (108, 143, 24, 15);
    chordDatabaseKey.setRect (2, 75, 69, 15);
    chordDatabaseChord.setRect (90, 75, 69, 15);
    chordDatabaseDetails.setRect (2, 100, 157, 40);

    // chord quiz page widgets
    chordQuizDone.setRect (132, 143, 26, 15);
    chordQuizPlay.setRect (108, 143, 24, 15);
    chordQuizNext.setRect (5, 143, 30, 15);
    chordQuizChord.setRect (5, 115, 60, 15);
  }

  // Does the checking of the position of the note, assign the note
  // value and write to the vector array. This method is for the black keys.

  public void addBlackKey (int x) {
    int xpos = x;
    int noteValue = 0, noteDuration, noteFrequency;
    String tmpBuf;

    switch (xpos) {
      case 10:
	noteValue = Score.CSHARP;
	break;
      case 18:
	noteValue = Score.DSHARP;
	break;
      case 34:
	noteValue = Score.FSHARP;
	break;
      case 42:
	noteValue = Score.AFLAT;
	break;
      case 50:
	noteValue = Score.BFLAT;
	break;
      case 66:
	noteValue = Score.HIGH_CSHARP;
	break;
      case 74:
	noteValue = Score.HIGH_EFLAT;
	break;
      case 90:
	noteValue = Score.HIGH_FSHARP;
	break;
      case 98:
	noteValue = Score.HIGH_AFLAT;
	break;
      case 106:
	noteValue = Score.HIGH_BFLAT;
	break;
      case 122:
	noteValue = Score.HIGHER_CSHARP;
	break;
      case 130:
	 noteValue = Score.HIGHER_EFLAT;
	break;
      case 146:
	noteValue = Score.HIGHER_FSHARP;
	break;
      case 154:
	noteValue = Score.HIGHER_AFLAT;
	break;
    }
    // set the duration
    noteDuration = Score.CROTCHET_DURATION;
    // get the frequency
    noteFrequency = Score.getNoteFrequency (noteValue);
    // formulate the string
    tmpBuf = 0 + ":" + 0 + ":" + 0 + ":" + noteFrequency + ":" + noteDuration + ":" +  0  + ":" +  0;
    // add to vector
    notesVec.add (tmpBuf);
  }

  // Does the checking of the position of the note, assign the note
  // value and write to the vector array. This method is for the white keys.

  public void addWhiteKey (int x) {
    int xpos = x;
    int noteValue=0, noteDuration, noteFrequency;
    String tmpBuf;

    switch (xpos) {
      case 6:
	noteValue = Score.C;
	break;
      case 14:
	noteValue = Score.D;
	break;
      case 22:
	noteValue = Score.E;
	break;
      case 30:
	noteValue = Score.F;
	break;
      case 38:
	noteValue = Score.G;
	break;
      case 46:
	noteValue = Score.A;
	break;
      case 54:
	noteValue = Score.B;
	break;
      case 62:
	noteValue = Score.HIGH_C;
	break;
      case 70:
	noteValue = Score.HIGH_D;
	break;
      case 78:
	noteValue = Score.HIGH_E;
	break;
      case 86:
	noteValue = Score.HIGH_F;
	break;
      case 94:
	noteValue = Score.HIGH_G;
	break;
      case 102:
	noteValue = Score.HIGH_A;
	break;
      case 110:
	noteValue = Score.HIGH_B;
	break;
      case 118:
	noteValue = Score.HIGHER_C;
	break;
      case 126:
	noteValue = Score.HIGHER_D;
	break;
      case 134:
	noteValue = Score.HIGHER_E;
	break;
      case 142:
	noteValue = Score.HIGHER_F;
	break;
      case 150:
	noteValue = Score.HIGHER_G;
	break;
    }
    // set the duration
    noteDuration = Score.CROTCHET_DURATION;
    // get the frequency
    noteFrequency = Score.getNoteFrequency (noteValue);
    // formulate the string
    tmpBuf = 0 + ":" + 0 + ":" + 0 + ":" + noteFrequency + ":" + noteDuration + ":" +  0  + ":" +  0;
    // add to vector
    notesVec.add (tmpBuf);
  }

  // draw notes and insert record into the vector

  public void assignNote (int assignedNote, int YPos) {
    int noteDuration, noteFrequency, noteValue;
    int accType = 0;
    String tmpBuf;

    // assign the position
    if (startDistance < 155) {			// within page?
      startDistance = startDistance + NOTEDISTANCE;
      // draw the note on the stave
      // get the duration of the note
      noteDuration = Score.getNoteDuration (noteSelected);
      noteValue = assignedNote;
      // change the note value if it belongs to the key signature
      noteValue = changeNoteValue (assignedNote, 0);
      // get the frequency of the note
      noteFrequency = Score.getNoteFrequency (noteValue);
      // formulate the string to write to vector
      tmpBuf = noteSelected + ":" + startDistance + ":" + YPos + ":" + noteFrequency + ":" + noteDuration + ":" + noteValue + ":" + accType;
      // add to vector
      notesVec.add (tmpBuf);
     drawAgain ();
    }
  }

  // draw the key signature and the start position is return to
  // method assignStartPos

  public int assignKeySignature (int acc, int num) {
     int accidental, number, firstPos = 20;
     accidental = acc;
     number = num;

     // accidental is sharp
     if (accidental == 1) {
	switch (number) {
	  // F#
	  case 1:
	     Score.drawSharp (g, 15, 38);
	     firstPos = 20;
	     break;
	  // F#, C#
	  case 2:
	     Score.drawSharp (g, 15, 38);
	     Score.drawSharp (g, 19, 47);
	     firstPos = 20;
	     break;
	  // F#, C#, G#
	  case 3:
	     Score.drawSharp (g, 15, 38);
	     Score.drawSharp (g, 19, 47);
	     Score.drawSharp (g, 27, 35);
	     firstPos = 30;
	     break;
	  // F#, C#, G#, D#
	  case 4:
	     Score.drawSharp (g, 15, 38);
	     Score.drawSharp (g, 19, 47);
	     Score.drawSharp (g, 27, 35);
	     Score.drawSharp (g, 32, 44);
	     firstPos = 35;
	     break;
	  // F#, C#, G#, D#, A#
	  case 5:
	     Score.drawSharp (g, 15, 38);
	     Score.drawSharp (g, 19, 47);
	     Score.drawSharp (g, 27, 35);
	     Score.drawSharp (g, 32, 44);
	     Score.drawSharp (g, 36, 53);
	     firstPos = 45;
	     break;
	  // F#, C#, G#, D#, A#, E#
	  case 6:
	     Score.drawSharp (g, 15, 38);
	     Score.drawSharp (g, 19, 47);
	     Score.drawSharp (g, 27, 35);
	     Score.drawSharp (g, 32, 44);
	     Score.drawSharp (g, 36, 53);
	     Score.drawSharp (g, 44, 41);
	     firstPos = 50;
	     break;
	  // F#, C#, G#, D#, A#, E#, B#
	  case 7:
	     Score.drawSharp (g, 15, 38);
	     Score.drawSharp (g, 19, 47);
	     Score.drawSharp (g, 27, 35);
	     Score.drawSharp (g, 32, 44);
	     Score.drawSharp (g, 36, 53);
	     Score.drawSharp (g, 44, 41);
	     Score.drawSharp (g, 48, 50);
	     firstPos = 50;
	     break;
	}
     }
     // accidental is flat
     else if (accidental == 2) {
	switch (number) {
	  // Bb
	  case 1:
	     Score.drawFlat (g, 16, 47);
	     firstPos = 20;
	     break;
	  // Bb, Eb
	  case 2:
	     Score.drawFlat (g, 16, 47);
	     Score.drawFlat (g, 22, 37);
	     firstPos = 20;
	     break;
	  // Bb, Eb, Ab
	  case 3:
	     Score.drawFlat (g, 16, 47);
	     Score.drawFlat (g, 22, 37);
	     Score.drawFlat (g, 25, 49);
	     firstPos = 30;
	     break;
	  // Bb, Eb, Ab, Db
	  case 4:
	     Score.drawFlat (g, 16, 47);
	     Score.drawFlat (g, 22, 37);
	     Score.drawFlat (g, 25, 49);
	     Score.drawFlat (g, 32, 41);
	     firstPos = 35;
	     break;
	  // Bb, Eb, Ab, Db, Gb
	  case 5:
	     Score.drawFlat (g, 16, 47);
	     Score.drawFlat (g, 22, 37);
	     Score.drawFlat (g, 25, 49);
	     Score.drawFlat (g, 32, 41);
	     Score.drawFlat (g, 36, 53);
	     firstPos = 45;
	     break;
	}
     }
     return firstPos;
  }

  // return the start position of the first note based on the key signature

  public int assignStartPos () {
    int pos = 20;
    g.setForeColor (Color.BLACK);

    if (keySignature.equals ("C Major")) {
      accidentals = "0";
    }
    else if (keySignature.equals ("D Major")) {
      pos = assignKeySignature (1, 2);
      accidentals = "2Sharp";
    }
    else if (keySignature.equals ("E Major")) {
      pos = assignKeySignature (1, 4);
      accidentals = "4Sharp";
    }
    else if (keySignature.equals ("F Major")) {
      pos = assignKeySignature (2, 1);
      accidentals = "1Flat";
    }
    else if (keySignature.equals ("G Major")) {
      pos = assignKeySignature (1, 1);
      accidentals = "1Sharp";
    }
    else if (keySignature.equals ("A Major")) {
      pos = assignKeySignature (1, 3);
      accidentals = "3Sharp";
    }
    else if (keySignature.equals ("B Major")) {
      pos = assignKeySignature (1, 5);
      accidentals = "5Sharp";
    }
    else if (keySignature.equals ("Bb Major")) {
      pos = assignKeySignature (2, 2);
      accidentals = "2Flat";
    }
    else if (keySignature.equals ("Eb Major")) {
      pos = assignKeySignature (2, 3);
      accidentals = "3Flat";
    }
    else if (keySignature.equals ("Ab Major")) {
      pos = assignKeySignature (2, 4);
      accidentals = "4Flat";
    }
    else if (keySignature.equals ("Db Major")) {
      pos = assignKeySignature (2, 5);
      accidentals = "5Flat";
    }
    else if (keySignature.equals ("C# Major")) {
      pos = assignKeySignature (1, 7);
      accidentals = "7Sharp";
    }
    else if (keySignature.equals ("F# Major")) {
      pos = assignKeySignature (1, 6);
      accidentals = "6Sharp";
    }
    return pos;
  }

  // make error or alerting sound

  private void beep () {
    Sound.tone (1000, 100);
  }

  // change the note value for those that belongs to the key signature

  public int changeNoteValue (int noteVal, int checkNote) {
    int noteValue = noteVal;
    int newNoteVal = 0;
    int num = Convert.toInt (accidentals.substring (0, 1));

    // if key has sharps
    if (accidentals.endsWith ("Sharp")) {
      // for one sharp
      if (Score.checkFNote (noteValue) && (num != 0)) {
	 newNoteVal = Score.changeFNote (noteValue, Score.SHARP);
	 return newNoteVal;
      }
      // for two sharps
      else if (Score.checkCNote (noteValue) && (num >= 2)) {
	 newNoteVal = Score.changeCNote (noteValue, Score.SHARP);
	 return newNoteVal;
      }
      // for three sharps
      else if (Score.checkGNote (noteValue) && (num >= 3)) {
	 newNoteVal = Score.changeGNote (noteValue, Score.SHARP);
	 return newNoteVal;
      }
      // for four sharps
      else if (Score.checkDNote (noteValue) && (num >= 4)) {
	 newNoteVal = Score.changeDNote (noteValue, Score.SHARP);
	 return newNoteVal;
      }
      // for five sharps
      else if (Score.checkANote (noteValue) && (num >= 5)) {
	 newNoteVal = Score.changeANote (noteValue, Score.SHARP);
	 return newNoteVal;
      }
      // for six sharps
      else if (Score.checkENote (noteValue) && (num >= 6)) {
	 newNoteVal = Score.changeENote (noteValue, Score.SHARP);
	 return newNoteVal;
      }
      // for seven sharps
      else if (Score.checkBNote (noteValue) && (num>=7)) {
	 newNoteVal = Score.changeBNote (noteValue, Score.SHARP);
	 return newNoteVal;
      }
    }
    // if key is flats
    else if (accidentals.endsWith ("Flat")) {
      // for one flat
      if (Score.checkBNote (noteValue) && (num!=0)) {
	 newNoteVal = Score.changeBNote (noteValue, Score.FLAT);
	 return newNoteVal;
      }
      // for two flats
      else if (Score.checkENote (noteValue) && (num >= 2)) {
	 newNoteVal = Score.changeENote (noteValue, Score.FLAT);
	 return newNoteVal;
      }
      // for three flats
      else if (Score.checkANote (noteValue) && (num >= 3)) {
	 newNoteVal = Score.changeANote (noteValue, Score.FLAT);
	 return newNoteVal;
      }
      // for four flats
      else if (Score.checkDNote (noteValue) && (num >= 4)) {
	 newNoteVal = Score.changeDNote (noteValue, Score.FLAT);
	 return newNoteVal;
      }
      // for five flats
      else if (Score.checkGNote (noteValue) && (num >= 5)) {
	 newNoteVal = Score.changeGNote (noteValue, Score.FLAT);
	 return newNoteVal;
      }
    }
    return noteValue;
  }

  // change the note based on the accidental

  public int changeToAccidental (int noteVal) {
    int noteValue = noteVal;
    int newNoteVal = 0;

    // for C note
    if (Score.checkCNote (noteValue)) {
      newNoteVal = Score.changeCNote (noteValue, noteSelected);
      return newNoteVal;
    }
    // for D note
    else if (Score.checkDNote (noteValue)) {
      newNoteVal = Score.changeDNote (noteValue, noteSelected);
      return newNoteVal;
    }
    // for E note
    else if (Score.checkENote (noteValue)) {
      newNoteVal = Score.changeENote (noteValue, noteSelected);
      return newNoteVal;
    }
    // for F note
    else if (Score.checkFNote (noteValue)) {
      newNoteVal = Score.changeFNote (noteValue, noteSelected);
      return newNoteVal;
    }
    // for G note
    else if (Score.checkGNote (noteValue)) {
      newNoteVal = Score.changeGNote (noteValue, noteSelected);
      return newNoteVal;
    }
    // for A note
    else if (Score.checkANote (noteValue)) {
      newNoteVal = Score.changeANote (noteValue, noteSelected);
      return newNoteVal;
    }
    // for B note
    else if (Score.checkBNote (noteValue)) {
      newNoteVal = Score.changeBNote (noteValue, noteSelected);
      return newNoteVal;
    }
    // if the note value is note change, the old value will be returned
    return noteValue;
  }

 // check the user's answer and the correct answer

  public void checkAnswer (int butNum) {
    int quizAnswer;

    quizAnswer = butNum;
    // if answer is correct, post another question, increase correct count
    if (quizAnswer == answerNum) {
      correct++;
      intervalArray = null;
      postQuestion ();
      drawAgain ();
    }
    // if answer is wrong, increase incorrect count, prompt with right answer
    else {
      inCorrect++;
      popupBox (new MessageBox ("Quiz", "Sorry, the correct|answer is "+
	Convert.toString (answerNum)));
    }
  }

  // calls the respective check position method to get selected note position

  public int checkPointPos () {
    int ptpos = 0;
    int acc = Convert.toInt (accidentals.substring (0, 1));

    // if it's page 1, have to take into consideration the
    // different key signature
    if (pageCount == 1) {
      if (acc == 0 || acc == 1 || acc == 2)
	 ptpos = checkPos ();
      else if (acc == 3)
	 ptpos = checkPos2 ();
      else if (acc == 4)
	 ptpos = checkPos3 ();
      else if (acc == 5)
	 ptpos = checkPos4 ();
      else if (acc == 6 || acc == 7)
	 ptpos = checkPos5 ();
    }
    else
      ptpos = checkPosG ();
    return ptpos;
  }

  // checks the position for keys that have 0, 1 or 2 accidentals

  public int checkPos () {
    int ptpos = 0;

    if (lastX >= 30 && lastX <= 40) {
      vecIndex = 0;
      ptpos = 32;
    }
    else if (lastX >= 45 && lastX <= 55) {
      vecIndex = 1;
      ptpos = 47;
    }
    else if (lastX >= 60 && lastX <= 70) {
      vecIndex = 2;
      ptpos = 62;
    }
    else if (lastX >= 75 && lastX <= 85) {
      vecIndex = 3;
      ptpos = 77;
    }
    else if (lastX >= 90 && lastX <= 100) {
      vecIndex = 4;
      ptpos = 92;
    }
    else if (lastX >= 105 && lastX <= 115) {
      vecIndex = 5;
      ptpos = 107;
    }
    else if (lastX >= 120 && lastX <= 130) {
      vecIndex = 6;
      ptpos = 122;
    }
    else if (lastX >= 135 && lastX <= 145) {
      vecIndex = 7;
      ptpos = 137;
    }
    else if (lastX >= 150 && lastX <= 160) {
      vecIndex = 8;
      ptpos = 152;
    }
    numNotes = 9;
    return ptpos;
  }

  // checks the position for keys that have 3 accidentals

  public int checkPos2 () {
    int ptpos = 0;

    if (lastX >= 40 && lastX <= 50) {
      vecIndex = 0;
      ptpos = 42;
    }
    else if (lastX >= 55 && lastX <= 65) {
      vecIndex = 1;
      ptpos = 57;
    }
    else if (lastX >= 70 && lastX <= 80) {
      vecIndex = 2;
      ptpos = 72;
    }
    else if (lastX >= 85 && lastX <= 95) {
      vecIndex = 3;
      ptpos = 87;
    }
    else if (lastX >= 100 && lastX <= 110) {
      vecIndex = 4;
      ptpos = 102;
    }
    else if (lastX >= 115 && lastX <= 125) {
      vecIndex = 5;
      ptpos = 117;
    }
    else if (lastX >= 130 && lastX <= 140) {
      vecIndex = 6;
      ptpos = 132;
    }
    else if (lastX >= 145 && lastX <= 155) {
      vecIndex = 7;
      ptpos = 147;
    }
    numNotes = 8;
    return ptpos;
  }

  // checks the position for keys that have 4 accidentals

  public int checkPos3 () {
    int ptpos=0;

    if (lastX >= 45 && lastX <= 55) {
      vecIndex = 0;
      ptpos = 47;
    }
    else if (lastX >= 60 && lastX <= 70) {
      vecIndex = 1;
      ptpos = 62;
    }
    else if (lastX >= 75 && lastX <= 85) {
      vecIndex = 2;
      ptpos = 77;
    }
    else if (lastX >= 90 && lastX <= 100) {
      vecIndex = 3;
      ptpos = 92;
    }
    else if (lastX >= 105 && lastX <= 115) {
      vecIndex = 4;
      ptpos = 107;
    }
    else if (lastX >= 120 && lastX <= 130) {
      vecIndex = 5;
      ptpos = 122;
    }
    else if (lastX >= 135 && lastX <= 145) {
      vecIndex = 6;
      ptpos = 137;
    }
    else if (lastX >= 150 && lastX <= 160) {
      vecIndex = 7;
      ptpos = 152;
    }
    numNotes = 8;
    return ptpos;
  }

  // checks the position for keys that have 5 accidentals

  public int checkPos4 () {
    int ptpos = 0;

    if (lastX >= 55 && lastX <= 65) {
      vecIndex = 0;
      ptpos = 57;
    }
    else if (lastX >= 70 && lastX <= 80) {
      vecIndex = 1;
      ptpos = 72;
    }
    else if (lastX >= 85 && lastX <= 95) {
      vecIndex = 2;
      ptpos = 87;
    }
    else if (lastX >= 100 && lastX <= 110) {
      vecIndex = 3;
      ptpos = 102;
    }
    else if (lastX >= 115 && lastX <= 125) {
      vecIndex = 4;
      ptpos = 117;
    }
    else if (lastX >= 130 && lastX <= 140) {
      vecIndex = 5;
      ptpos = 132;
    }
    else if (lastX >= 145 && lastX <= 155) {
      vecIndex = 6;
      ptpos = 147;
    }
    numNotes = 7;
    return ptpos;
  }

  // checks the position for keys that have 6 or 7 accidentals

  public int checkPos5 () {
    int ptpos=0;

    if (lastX >= 60 && lastX <= 70) {
      vecIndex = 0;
      ptpos = 62;
    }
    else if (lastX >= 75 && lastX <= 85) {
      vecIndex = 1;
      ptpos = 77;
    }
    else if (lastX >= 90 && lastX <= 100) {
      vecIndex = 2;
      ptpos = 92;
    }
    else if (lastX >= 105 && lastX <= 115) {
      vecIndex = 3;
      ptpos = 107;
    }
    else if (lastX >= 120 && lastX <= 130) {
      vecIndex = 4;
      ptpos = 122;
    }
    else if (lastX >= 135 && lastX <= 145) {
      vecIndex = 5;
      ptpos = 137;
    }
    else if (lastX >= 150 && lastX <= 160) {
      vecIndex = 6;
      ptpos = 152;
    }
    numNotes = 7;
    return ptpos;
  }

  // checks the position of the note from the 2nd page onwards

  public int checkPosG () {
    int ptpos = 0;

    if (lastX >= 10 && lastX <= 20) {
      vecIndex = numNotes + (pageCount - 2) * 10;
      ptpos = 12;
    }
    else if (lastX >= 25 && lastX <= 35) {
      vecIndex = numNotes + (pageCount - 2) * 10 + 1;
      ptpos = 27;
    }
    else if (lastX >= 40 && lastX <= 50) {
      vecIndex = numNotes + (pageCount - 2) * 10 + 2;
      ptpos = 42;
    }
    else if (lastX >= 55 && lastX <= 65) {
      vecIndex = numNotes + (pageCount - 2) * 10 + 3;
      ptpos = 57;
    }
    else if (lastX >= 70 && lastX <= 80) {
      vecIndex = numNotes + (pageCount - 2) * 10 + 4;
      ptpos = 72;
    }
    else if (lastX >= 85 && lastX <= 95) {
      vecIndex = numNotes + (pageCount - 2) * 10 + 5;
      ptpos = 87;
    }
    else if (lastX >= 100 && lastX <= 110) {
      vecIndex = numNotes + (pageCount - 2) * 10 + 6;
      ptpos = 102;
    }
    else if (lastX >= 115 && lastX <= 125) {
      vecIndex = numNotes + (pageCount - 2) * 10 + 7;
      ptpos = 117;
    }
    else if (lastX >= 130 && lastX <= 140) {
      vecIndex = numNotes + (pageCount - 2) * 10 + 8;
      ptpos = 132;
    }
    else if (lastX >= 145 && lastX <= 155) {
      vecIndex = numNotes + (pageCount - 2) * 10 + 9;
      ptpos = 147;
    }
    return ptpos;
  }

  // Check which symbol is selected and assigned to a variable. The
  // different position will indicate what is the note or accidental or
  // rest selected.

  public void checkSelectedSymbol (int lastX, int lastY) {
    String vecStr;
    String[] tmparray=null;
    int xpos, ypos, dum, vecCount, newValue, frequency, noteValue;
    int oldValue, accType, oldAccType = 0;
    this.lastX = lastX;
    this.lastY = lastY;

    // this check for the note selected. Highlighting is done on the rectangle
    // that the note is in.
    if (lastY >= 98 && lastY <= 115) {
      if (lastX >= 0 && lastX <= 10) {
	 noteSelected = Score.SEMIBREVE;
	 doSelection ();
      }
      else if (lastX >= 11 && lastX <= 25) {
	 noteSelected = Score.MINIM;
	 doSelection ();
      }
      else if (lastX >= 26 && lastX <= 40) {
	 noteSelected = Score.DOTTED_MINIM;
	 doSelection ();
	      }
      else if (lastX >= 41 && lastX <= 55) {
	 noteSelected = Score.CROTCHET;
	 doSelection ();
      }
      else if (lastX >= 56 && lastX <= 70) {
	 noteSelected = Score.DOTTED_CROTCHET;
	 doSelection ();
      }
      else if (lastX >= 71 && lastX <= 85) {
	 noteSelected = Score.QUAVER;
	 doSelection ();
      }
      else if (lastX >= 86 && lastX <= 100) {
	 noteSelected = Score.SEMIQUAVER;
	 doSelection ();
      }
    }
    // This is the area of the accidentals and rests. On selection of either
    // the flat, sharp or natural, the accidental will be drawn next to the note
    // if any.
    else if ((lastY >= 115 && lastY <= 130) && (lastX <= 100)) {
      vecCount = notesVec.getCount ();
      if (vecCount != 0) {
	 if (vecIndex == -1)
	    vecIndex = vecCount - 1;
	 vecStr = (String) notesVec.get (vecIndex);
	 // split the different information into a temp array.
	 tmparray = Convert.tokenizeString (vecStr, ':');
	 oldAccType = Convert.toInt (tmparray[6]);
      }
      // flat selected
      if (lastX >= 0 && lastX <= 10) {
	 noteSelected = Score.FLAT;
	 doSelection ();
	 // proceed only if the temp array is not null
	 if (tmparray != null) {
	    noteValue = Convert.toInt (tmparray[0]);
	    // and also if the note selected is not a rest
	    if (!(noteValue >= Score.CROTCHET_REST &&
		  noteValue <= Score.QUAVER_REST)) {
	       // old/new note values checked to make sure that this
	       // note is not within the key signature
	       oldValue = Convert.toInt (tmparray[5]);
	       newValue = changeToAccidental (Convert.toInt (tmparray[5]));
	       if (newValue != oldValue) {
		  // if  not the same, draw the flat and overwrite
		  // this record in the vector with the new information
		  xpos = Convert.toInt (tmparray[1]) - 8;
		  ypos = Convert.toInt (tmparray[2]) - 6;
		  Score.drawFlat (g, xpos, ypos);
		  frequency = Score.getNoteFrequency (newValue);
		  if (oldAccType == Score.SHARP)
		    accType = 0;
		  else
		    accType = noteSelected;
		  String newStr =
		    tmparray[0] + ":" + tmparray[1] + ":" + tmparray[2] + ":" +
		      frequency + ":" + tmparray[4] + ":" + newValue + ":" +
			accType;
		  notesVec.set (vecIndex, newStr);
		  vecIndex = -1;
		  tmparray = null;
	       }
	       drawAgain ();
	    }
	 }
      }
      // sharp selected
      else if (lastX >= 11 && lastX <= 25) {
	 noteSelected = Score.SHARP;
	 doSelection ();
	 // proceed only if the temp array is not null
	 if (tmparray != null) {
	    noteValue = Convert.toInt (tmparray[0]);
	    // and also if the note selected is not rest.
	    if (!(noteValue >= Score.CROTCHET_REST &&
		  noteValue <= Score.QUAVER_REST)) {
	       // old/new note values checked to make sure that this
	       // note is not within the key signature
	       oldValue = Convert.toInt (tmparray[5]);
	       newValue = changeToAccidental (Convert.toInt (tmparray[5]));
	       if (newValue != oldValue) {
		  // if not the same, draw the flat and overwrite
		  // this record in the vector with the new information
		  xpos = Convert.toInt (tmparray[1]) - 8;
		  ypos = Convert.toInt (tmparray[2]) - 2;
		  Score.drawSharp (g, xpos, ypos);
		  frequency = Score.getNoteFrequency (newValue);
		  if (oldAccType == Score.FLAT)
		    accType = 0;
		  else
		    accType = noteSelected;
		  String newStr =
		    tmparray[0] + ":" + tmparray[1] + ":" + tmparray[2] + ":" +
		      frequency + ":" + tmparray[4] + ":" + newValue + ":" +
			accType;
		  notesVec.set (vecIndex, newStr);
		  vecIndex = -1;
		  tmparray = null;
	       }
	       drawAgain ();
	    }
	 }
      }
      else if (lastX >= 26 && lastX <= 40) {
	 noteSelected = Score.CROTCHET_REST;
	 doSelection ();
      }
      else if (lastX >= 41 && lastX <= 55) {
	 noteSelected = Score.MINIM_REST;
	 doSelection ();
      }
      else if (lastX >= 56 && lastX <= 70) {
	 noteSelected = Score.BREVE_REST;
	 doSelection ();
      }
      else if (lastX >= 71 && lastX <= 85) {
	 noteSelected = Score.QUAVER_REST;
	 doSelection ();
      }
      else if (lastX >=86 && lastX <= 100) {
	 noteSelected = Score.NATURAL;
	 doSelection ();
	 // proceed only if the temp array is not null
	 if (tmparray != null) {
	    noteValue = Convert.toInt (tmparray[0]);
	    // and also if the note selected is not rest.
	    if (!(noteValue >= Score.CROTCHET_REST &&
		  noteValue <= Score.QUAVER_REST)) {
	       // old/new note values checked to make sure that this
	       // note is not within the key signature
	       oldValue = Convert.toInt (tmparray[5]);
	       newValue = changeToAccidental (Convert.toInt (tmparray[5]));
	       if (newValue != oldValue) {
		  // if not the same, draw the natural sign and overwrite
		  // this record in the vector with the new information
		  xpos = Convert.toInt (tmparray[1]) - 7;
		  ypos = Convert.toInt (tmparray[2]) - 6;
		  Score.drawNatural (g, xpos, ypos);
		  frequency = Score.getNoteFrequency (newValue);
		  if (oldAccType == Score.NATURAL)
		    accType = 0;
		  else
		    accType = noteSelected;
		  String newStr =
		    tmparray[0] + ":" + tmparray[1] + ":" + tmparray[2] + ":" +
		      frequency + ":" + tmparray[4] + ":" + newValue + ":" +
			accType;
		  notesVec.set (vecIndex, newStr);
		  vecIndex = -1;
		  tmparray = null;
	       }
	       else {
		  if (oldAccType == Score.NATURAL)
		    accType = 0;
		  else
		    accType = noteSelected;
		  String newStr =
		    tmparray[0] + ":" + tmparray[1] + ":" + tmparray[2] + ":" +
		      tmparray[3] + ":" + tmparray[4] + ":" +
			tmparray[5] + ":" + accType;
		  notesVec.set (vecIndex, newStr);
		  vecIndex = -1;
		  tmparray = null;
	       }
	       drawAgain ();
	    }
	 }
      }
    }
  }

  // delete a note

  public void deleteNote () {
    String[] tmparray, tmparray1;
    int newXpos, vecCount;
    String newString;

    // get the number of items in the vector
    vecCount = notesVec.getCount ();

    if (vecCount > 0) {
      // set the vector index
      lastX = lastCaretX;
      checkPointPos ();

      // delete the note based on the vector index
      notesVec.del (vecIndex);
      // get the number of items again
      vecCount = notesVec.getCount ();
      // change the X position of the rest of the notes
      for (int i = vecIndex; i < vecCount; i++) {
	String noteString = (String) notesVec.get (i);
	tmparray = Convert.tokenizeString (noteString, ':');
	newXpos = Convert.toInt (tmparray[1]) - 15;
	if (newXpos == 0) {
	  String tmpString = (String) notesVec.get (i - 1);
	  tmparray1 = Convert.tokenizeString (tmpString, ':');
	  newXpos = Convert.toInt (tmparray1[1]) + 15;
	}
	// formulate the new string
	newString = tmparray[0] + ":" + newXpos + ":" + tmparray[2] + ":" + tmparray[3] + ":" + tmparray[4] + ":" + tmparray[5] + ":" + tmparray[6];
	// write back to the vector
	notesVec.set (i, newString);
      }
      // re-assign the note position
      startDistance = startDistance - 15;
      if (startDistance == 0) {
	String tmpString = (String) notesVec.get (vecCount - 2);
	tmparray1 = Convert.tokenizeString (tmpString, ':');
	startDistance = Convert.toInt (tmparray1[1]) + 15;
      }
    }
  }

  // does a redraw of all the frames again

  public void deSelect () {
    g.setForeColor (Color.BLACK);
    g.drawLine (10, 98, 10, 130);
    g.drawLine (25, 98, 25, 130);
    g.drawLine (40, 98, 40, 130);
    g.drawLine (55, 98, 55, 130);
    g.drawLine (70, 98, 70, 130);
    g.drawLine (85, 98, 85, 130);
    g.drawLine (100, 98, 100, 130);

    g.drawLine (0, 98, 100, 98);
    g.drawLine (0, 98, 0, 130);
    g.drawLine (0, 115, 100, 115);
    g.drawLine (0, 130, 100, 130);
  }

  // display the answers on the 3 buttons

  public void displayQuestions () {
    int quNum;
    String quStr, key="";
    String[] tmparray;

    // get the key of the chord
    tmparray = Convert.tokenizeString (questionArray[0], ' ');
    key = tmparray[0];

    // assign the first item in the question array to the first button
    firstAns.setText (questionArray[0]);

    // assign the second item in the question array to the second button
    secondAns.setText (questionArray[1]);

    // assign the third item in the question array to the third button
    thirdAns.setText (questionArray[2]);

    g.drawText (key, 33, 65);
  }

  // check for the position that the user clicks and assign
  // the value of the note accordingly. The return value is used for insertion
  // if neccessary

  public String doAssignment (int lastY, boolean insertion) {
    int noteValue=0;
    int YPos = lastY;

    if (YPos >= 30 && YPos <= 32) {
      YPos = 31;
      noteValue = Score.HIGH_B;
    }
    else if (YPos >= 36 && YPos <= 38) {
      YPos = 37;
      noteValue = Score.HIGH_G;
    }
    else if (YPos >= 42 && YPos <= 44) {
      YPos = 43;
      noteValue = Score.HIGH_E;
    }
    else if (YPos >= 48 && YPos <= 50) {
      YPos = 49;
      noteValue = Score.HIGH_C;
    }
    else if (YPos >= 54 && YPos <= 56) {
      YPos = 55;
      noteValue = Score.A;
    }
    else if (YPos >= 60 && YPos <= 62) {
      YPos = 61;
      noteValue = Score.F;
    }
    else if (YPos >= 66 && YPos <= 68) {
      YPos = 67;
      noteValue = Score.D;
    }
    else if (YPos >= 72 && YPos <= 74) {
      YPos = 73;
      noteValue = Score.LOW_B;
    }
    else if (YPos >= 78 && YPos <= 80) {
      YPos = 79;
      noteValue = Score.LOW_G;
    }
    else if (YPos >= 27 && YPos <= 29) {
      YPos = 28;
      noteValue = Score.HIGHER_C;
    }
    else if (YPos >= 33 && YPos <= 35) {
      YPos = 34;
      noteValue = Score.HIGH_A;
    }
    else if (YPos >= 39 && YPos <= 41) {
      YPos = 40;
      noteValue = Score.HIGH_F;
    }
    else if (YPos >= 45 && YPos <= 47) {
      YPos = 46;
      noteValue = Score.HIGH_D;
    }
    else if (YPos >= 51 && YPos <= 53) {
      YPos = 52;
      noteValue = Score.B;
    }
    else if (YPos >= 57 && YPos <= 59) {
      YPos = 58;
      noteValue = Score.G;
    }
    else if (YPos >= 63 && YPos <= 65) {
      YPos = 64;
      noteValue = Score.E;
    }
    else if (YPos >= 69 && YPos <= 71) {
      YPos = 70;
      noteValue = Score.C;
    }
    else if (YPos >= 75 && YPos <= 77) {
      YPos = 76;
      noteValue = Score.LOW_A;
    }

    // if this is not the insertion option
    if ((YPos > 26 && YPos < 77) && (insertion == false))
      assignNote (noteValue, YPos);
    // for insertion option
    else
      return (noteValue + ":" + YPos + ":" + noteSelected);

    return "0";
  }

  // Does the assignment of the different types of rests. Each is
  // assigned a x and y position like the notes. The information is also added
  // to the vector.

  public void doRestAssign (int YPos) {
    int noteDuration, noteFrequency=0, noteValue=0;
    String tmpBuf;

    // get the note duration
    noteDuration = Score.getNoteDuration (noteSelected);
    // assign the position
    startDistance = startDistance + NOTEDISTANCE;
    // formulate the string to write to vector
    tmpBuf = noteSelected + ":" +  (startDistance-2) + ":" + YPos + ":" + noteFrequency + ":" + noteDuration + ":" + noteValue + ":" + 0;
    // add to vector
    notesVec.add (tmpBuf);
  }

  // does the highlighting of the symbol which the user selects

  public void doSelection () {
    int note = noteSelected;

    switch (note) {
      // selection of semibreve
      case (0):
	deSelect ();
	selectSemiBreve ();
	break;
      // selection of dotted minim
      case (1):
	deSelect ();
	selectDottedMinim ();
	break;
      // selection of minim
      case (2):
	deSelect ();
	selectMinim ();
	break;
      // selection of dotted crotchet
      case (3):
	deSelect ();
	selectDottedCrotchet ();
	break;
      // selection of crotchet
      case (4):
	deSelect ();
	selectCrotchet ();
	break;
      // selection of quaver
      case (5):
	deSelect ();
	selectQuaver ();
	break;
      // selection of semiquaver
      case (6):
	deSelect ();
	selectSemiQuaver ();
	break;
      // selection of flat
      case (7):
	deSelect ();
	selectFlat ();
	break;
      // selection of sharp
      case (8):
	deSelect ();
	selectSharp ();
	break;
      // selection of crotchet rest
      case (9):
	deSelect ();
	selectCrotchetRest ();
	break;
      // selection of minim rest
      case (10):
	deSelect ();
	selectMinimRest ();
	break;
      // selection of semibreve rest
      case (11):
	deSelect ();
	selectSemiBreveRest ();
	break;
      // selection of quaver rest
      case (12):
	deSelect ();
	selectQuaverRest ();
	break;
      // selection of natural
      case (13):
	deSelect ();
	selectNatural ();
	break;
    }
  }

  // to call the onPaint method

  public void drawAgain () {
    redraw = true;
    repaint ();
  }

  // draw a note on the screen with the note selected and the position

  public void drawNote (int noteSelected, int lastX, int lastY) {
    this.noteSelected = noteSelected;
    this.lastX = lastX;
    this.lastY = lastY;

    // call the respective method for the note selected
    // if selection is semibreve
    if (noteSelected == Score.SEMIBREVE)
      Score.drawBreve (g, lastX, lastY);
    // if selection is minim
    else if (noteSelected == Score.MINIM)
      Score.drawMinim (g, lastX, lastY, false);
    // if selection is dotted minim
    else if (noteSelected == Score.DOTTED_MINIM)
      Score.drawMinim (g, lastX, lastY, true);
    // if selection is crotchet
    else if (noteSelected == Score.CROTCHET)
      Score.drawCrotchet (g, lastX, lastY, false);
    // if selection is dotted crotchet
    else if (noteSelected == Score.DOTTED_CROTCHET)
      Score.drawCrotchet (g, lastX, lastY, true);
    // if selection is quaver
    else if (noteSelected == Score.QUAVER)
      Score.drawSQuaver (g, lastX, lastY, 0);
    // if selection is semiquaver
    else if (noteSelected == Score.SEMIQUAVER)
      Score.drawSQuaver (g, lastX, lastY, 1);
    // if selection is crotchet rest
    else if (noteSelected == Score.CROTCHET_REST)
      Score.drawCrotchetRest (g, lastX, lastY);
    // if selection is semibreve rest
    else if (noteSelected == Score.BREVE_REST)
      Score.drawBreveRest (g, lastX, lastY);
    // if selection is minim rest
    else if (noteSelected == Score.MINIM_REST)
      Score.drawMinimRest (g, lastX, lastY);
    // if selection is quaver rest
    else if (noteSelected == Score.QUAVER_REST)
      Score.drawQuaverRest (g, lastX, lastY);
  }

  // draw the notes each time onPaint is called

  public void drawNotes () {
    String[] array;
    int noteSelected;
    int x_pos, y_pos, accType;
    int startIndex = 0, endIndex = 0;
    int vecLength;

    // get the number of items in the vector
    vecLength = notesVec.getCount ();
    // if it is on the first page
    if (pageCount == 1) {
      checkPointPos ();
      startIndex = 0;
      if (vecLength < numNotes)
	endIndex = vecLength;
      else
	endIndex = numNotes;
   }
    // for second page onwards
    else {
      startIndex = numNotes + (pageCount - 2) * 10;
      endIndex = startIndex + 10;
       if (vecLength < endIndex)
	 endIndex = vecLength;
    }

    // loop for the number of items, split the string into individual fields
    // and draw again
    for (int i = startIndex; i < endIndex; i++) {
       String tmpString = (String) notesVec.get (i);
       if (tmpString != null) {
	  array = Convert.tokenizeString (tmpString, ':');
	  noteSelected = Convert.toInt (array[0]);
	  x_pos = Convert.toInt (array[1]);
	  y_pos = Convert.toInt (array[2]);
	  accType = Convert.toInt (array[6]);
	  drawNote (noteSelected, x_pos, y_pos);
	  // for accidental type SHARP
	  if (accType == Score.SHARP) {
	      x_pos = x_pos - 8;
	      y_pos = y_pos - 2;
	      Score.drawSharp (g, x_pos, y_pos);
	  }
	  // for accidental type FLAT
	  else if (accType == Score.FLAT) {
	      x_pos = x_pos - 8;
	      y_pos = y_pos - 6;
	      Score.drawFlat (g, x_pos, y_pos);
	  }
	  // for accidental type NATURAL
	  else if (accType == Score.NATURAL) {
	      x_pos = x_pos - 7;
	      y_pos = y_pos - 6;
	      Score.drawNatural (g, x_pos, y_pos);
	  }
       }
    }
  }

  // generates the 3 answers and assign them to an array

  public void generateAll (String key, String chordname, int chordIndex) {
    int index1 = chordIndex, index2;
    String answerStr, tmpString1, tmpString2="";
    String chorddet, name;
    String[] tmparray;
    Random rnd = new Random ();

    // randomly pick a number to be the answer
    answerNum = rnd.rand (1, 4);
    // formulate the answer string
    answerStr = key + " " + chordname;

    // write the answer string to the question array
    questionArray[answerNum - 1] = answerStr;

    if (answerNum == 1) {
      // choose a index which is not equal to the chord index
      while (index1 == chordIndex) {
	 index1 = rnd.rand (0, CreateChords.CHORDS);
      }
      // assign it to the 2nd index, to do checking later
      index2 = index1;
      // get the details of index1 and assign it to the question array
      chorddet = ChordDB.getChordDetails (index1);
      tmparray = Convert.tokenizeString (chorddet, ':');
      name = tmparray[0];
      tmpString1 = key + " " + name;
      questionArray[1] = tmpString1;

      // choose a index which is not equal to the chord index and index1
      while (index2 == chordIndex || index2 == index1) {
	index2 = rnd.rand (0, CreateChords.CHORDS);
      }
      // get the details for index2 and assign it to the question array
      chorddet = ChordDB.getChordDetails (index2);
      tmparray = Convert.tokenizeString (chorddet, ':');
      name = tmparray[0];
      tmpString2 = key+" "+name;
      questionArray[2] = tmpString2;
    }
    else if (answerNum == 2) {
      // choose a index which is not equal to the chord index
      while (index1 == chordIndex) {
	index1 = rnd.rand (0, CreateChords.CHORDS);
      }
      // assign it to the 2nd index, to do checking later
      index2 = index1;
      // get the details of index1 and assign it to the question array
      chorddet = ChordDB.getChordDetails (index1);
      tmparray = Convert.tokenizeString (chorddet, ':');
      name = tmparray[0];
      tmpString1 = key+" "+name;
      questionArray[0] = tmpString1;

      // choose a index which is not equal to the chord index and index1
      while (index2 == chordIndex || index2 == index1) {
	index2 = rnd.rand (0, CreateChords.CHORDS);
      }
      // get the details for index2 and assign it to the question array
      chorddet = ChordDB.getChordDetails (index2);
      tmparray = Convert.tokenizeString (chorddet, ':');
      name = tmparray[0];
      tmpString2 = key+" "+name;
      questionArray[2] = tmpString2;

    }
    else {					// answer is 3
      // choose a index which is not equal to the chord index
      while (index1 == chordIndex) {
	index1 = rnd.rand (0, CreateChords.CHORDS);
      }
      // assign it to the 2nd index, to do checking later
      index2 = index1;
      // get the details of index1 and assign it to the question array
      chorddet = ChordDB.getChordDetails (index1);
      tmparray = Convert.tokenizeString (chorddet, ':');
      name = tmparray[0];
      tmpString1 = key + " " + name;
      questionArray[0] = tmpString1;

      // choose a index which is not equal to the chord index and index1
      while (index2 == chordIndex || index2 == index1) {
	index2 = rnd.rand (0, CreateChords.CHORDS);
      }
      // get the details for index2 and assign it to the question array
      chorddet = ChordDB.getChordDetails (index2);
      tmparray = Convert.tokenizeString (chorddet, ':');
      name = tmparray[0];
      tmpString2 = key+" "+name;
      questionArray[1] = tmpString2;
    }
  }

  // insert a note

  public void insertNote () {
    int noteValue = 0, noteSelected = 0, newYpos = 0, vecCount = 0,
      tmpIndex = 0, i = 0, j = 0;
    int duration = 0, frequency = 0, xpos = 0, tmpvecCount = 0, newXpos = 0;
    String tmpString = "", noteString = "", newString = "";
    String[] tmparray;
    Vector tmpVec = new Vector ();

    // get the number of items in the vector
    vecCount = notesVec.getCount ();

    // check the position and assign the vector index to a temporary index
    int pos = checkPointPos ();
    tmpIndex = vecIndex;

    // get the note's information
    tmpString = doAssignment (lastY, true);
    if (tmpString != null && !tmpString.equals ("")) {
      tmparray = Convert.tokenizeString (tmpString, ':');
      noteValue = Convert.toInt (tmparray[0]);
      newYpos = Convert.toInt (tmparray[1]);
      noteSelected = Convert.toInt (tmparray[2]);

      // if the temporary index is not the last item in the vector
      if (tmpIndex != vecCount - 1) {
	 tmpString = (String) notesVec.get (tmpIndex + 1);

	 tmparray = Convert.tokenizeString (tmpString, ':');
	 xpos = Convert.toInt (tmparray[1]);

	 // change the X position of the notes after the inserted note
	 // and store them in a temp vector
	 for (i = tmpIndex + 1; i < vecCount; i++) {
	    noteString = (String) notesVec.get (i);
	    tmparray = Convert.tokenizeString (noteString, ':');
	    newXpos = Convert.toInt (tmparray[1]) + 15;
	    if (newXpos > 160)
	       newXpos = 15;
	    newString =
	      tmparray[0] + ":" + newXpos + ":" + tmparray[2] + ":" +
		tmparray[3] + ":" + tmparray[4] + ":" + tmparray[5] + ":" +
		  tmparray[6];
	    tmpVec.add (newString);
	 }

	 // get the duration and frequency of the new note
	 duration = Score.getNoteDuration (noteValue);
	 frequency = Score.getNoteFrequency (noteSelected);

	 // formulate the new string
	 String newStr =
	   noteSelected + ":" + xpos + ":" + newYpos + ":" + frequency + ":" +
	     duration + ":" + noteValue + ":" + 0;
	 // insert the string at the right position
	 notesVec.set (tmpIndex + 1, newStr);

	tmpIndex += 2;
	 // delete the rest of the notes after the insertion point
	 for (j = tmpIndex; j < vecCount; j++) {
	    if (notesVec.get (tmpIndex) != null)
	       notesVec.del (tmpIndex);
	 }

	 // write back the notes from the temp vector to the original vector
	 tmpvecCount = tmpVec.getCount ();
	 for (i = 0; i < tmpvecCount; i++) {
	    notesVec.add (tmpVec.get (i));
	 }
	 // re-assign the note position
	 startDistance = newXpos;
	 tmpVec.clear ();
      }
    }
  }

  // when an event happens

  public void onEvent (Event e) {
    // if button is pressed
    boolean result;
    String title;

    if (e.type == ControlEvent.PRESSED) {
       // if it is the start screen exit, exit out of program
       if (e.target == mainExit) {
	  exit (0);
	   }
       // on clicking Score button
       else if (e.target == mainScore) {
	  // open the score database if it's not already open
	  if (!Score.scoreCat.isOpen ())
	    Score.openCatalog ();
	  wipeScreen ();
	  page = SCORE_PAGE;
	  redraw = true;
       }
       // on clicking Chord database button
       else if (e.target == mainChordDB) {
	  // check if chord database is open
	  result = ChordDB.openCatalog ();
	  wipeScreen ();

	  // if chord database not present, show error
	  if (result)
	    page = CHORDDB_PAGE;
	  else {
	    page = MAIN_PAGE;
	    popupBox (new MessageBox ("Error", "Chord Database Missing"));
	  }
	  redraw = true;
       }
       // on clicking on Chord quiz button
       else if (e.target == mainChordQuiz) {
	  // check if chord database is open
	  result = ChordDB.openCatalog ();
	  wipeScreen ();

	  // if chord database not present, show error
	  if (result) {
	     page = CHORDQUIZ_PAGE;
	     postQuestion ();
	  }
	  else {
	    page = MAIN_PAGE;
	    popupBox (new MessageBox ("Error", "Chord Database Missing"));
	  }
	  redraw = true;
       }
       // on click the Done button on the Score page
       else if (e.target == scoreDone) {
	  notesVec.clear ();
	  Score.closeCatalog ();
	  wipeScreen ();
	  page = MAIN_PAGE;
	  redraw = true;
       }
       // on clicking Play button on Score page
       else if (e.target == scorePlay) {
	  Score.playNotes (notesVec);
       }
       // on clicking Save button on Score page
       else if (e.target == scoreSave) {
	  title = scoreEdit.getText ();
	  MessageBox errorbox;

	  // if title is empty, show error
	  if (title.equals (""))
	    popupBox (new MessageBox ("Error", "Enter Score Title"));
	  // else check for existent of title
	  else {
	     result = Score.checkTitleExist (title);
	     // if exist, prompt to overwrite
	     if (result)
	       popupBox (overwriteBox);
	     // if not, write to score database
	     else {
	       if (notesVec.getCount () != 0) {
		 Score.insertRecord (title, keySignature, notesVec);
		 popupBox (successBox);
	       }
	     }
	  }
       }
       // if a key is chosen,set flag to prevent user from choosing again
       else if (e.target == keySignatureBox) {
	  if (drawKeySignature == false && notesVec.getCount () == 0) {
	    keySignature = (String) keySignatureBox.getSelectedItem ();
	    if (keySignature.equals (""))
	      keySignature = "C Major";
	    drawKeySignature = true;
	  }

       }
       // on clicking the Done button in the Chord database page
       // close chord database
       else if (e.target == chordDatabaseDone) {
	ChordDB.closeCatalog ();
	// clear the vector if there is any item
	if (notesVec.getCount () != 0)
	  notesVec.clear ();
	wipeScreen ();
	page = MAIN_PAGE;
	redraw = true;
       }
       else if (e.target == chordDatabasePlay) {
	  Score.playNotes (notesVec);
       }
      // combo for key selection
      else if (e.target == chordDatabaseKey) {
	 // get the key index
	 keyIndex = chordDatabaseKey.getSelectedIndex ();
	 // get the key in string
	 selectedKey = ChordDB.getKey (keyIndex);
	 if (chordIndex != -1 && chordIndex < 67) { // only if chord selected
	    processDetails (chordIndex);
	 }
	 drawAgain ();
      }
      else if (e.target == chordDatabaseChord) {
	 // get the chord index
	 chordIndex = chordDatabaseChord.getSelectedIndex ();
	 if (selectedKey == null )	// if no key is selected, display error
	   popupBox (new MessageBox ("Error", "Choose A Key"));
	 // proceed only if Chord index is within range
	 else if (chordIndex != -1 && chordIndex < 67) {
	   processDetails (chordIndex);
	   drawAgain ();
	 }
      }
      // on clicking the left arrow to go to next page
      else if (e.target == left) {
	if (pageCount != 1) {
	  pageCount--;
	  drawAgain ();
	}
      }
      // on clicking the right arrow to go back to the previous page
      else if (e.target == right) {
	pageCount++;
	if (startDistance > 149)
	  startDistance = 0;
	drawAgain ();
      }
      // on clicking the Done button in chord quiz
      else if (e.target == chordQuizDone) {
	ChordDB.closeCatalog ();
	// clear the vector array if there is item
	if (notesVec.getCount () != 0)
	  notesVec.clear ();
	correct = 0;
	inCorrect = 0;
	wipeScreen ();
	page = MAIN_PAGE;
	redraw = true;
      }
      // on clicking the first answer in chord quiz
      else if (e.target == firstAns) {
	checkAnswer (1);
      }
      // on clicking the second answer in chord quiz
      else if (e.target == secondAns) {
	checkAnswer (2);
      }
      // on clicking the third answer in chord quiz
      else if (e.target == thirdAns) {
	checkAnswer (3);
      }
      // on clicking the Next button, proceed to prompt another question
      else if (e.target == chordQuizNext) {
	intervalArray = null;
	postQuestion ();
	drawAgain ();
      }
      // play the notes of the chord
      else if (e.target == chordQuizPlay) {
	Score.playNotes (notesVec);
      }
    }
    // on window closed
    else if (e.type == ControlEvent.WINDOW_CLOSED) {
      // menu bar selection
      if (e.target == mbar) {
	int sel = mbar.getSelectedMenuItem ();
	onEventMenu (sel);
      }
      // on clicking the OK button on the answer message box
      else if (e.target == quizbox) {
	if (quizbox.getPressedButtonIndex () == 0) {
	  postQuestion ();
	}
      }
      // on clicking OK on the overwriting message box
      else if (e.target == overwriteBox) {
	if (overwriteBox.getPressedButtonIndex () == 0) {
	   // get title name
	   title = scoreEdit.getText ();
	   // proceed to insert the record only if the vector is not empty
	   if (notesVec.getCount () != 0) {
	     Score.insertRecord (title, keySignature, notesVec);
	     popupBox (successBox);
	   }
	}
      }
      drawAgain ();
    }
    // on pen down
    else if (e.type == PenEvent.PEN_DOWN)
       onEventPenDown (e);
    // on key press
    else if (e.target != scoreEdit && e.type == KeyEvent.KEY_PRESS)
      onEventKey (e);
  }

 // handle keypress

  public void onEventKey (Event event) {
    int key = ((KeyEvent) event).key;		// get event key

    switch (key) {
      case 'a': case 'A':			// about?
	onEventMenu (4);
	break;
      case 'c': case 'C':			// clear score?
	onEventMenu (3);
	break;
       case 'd': case 'D':			// delete note?
	onEventMenu (102);
	break;
      case 'i': case 'I':			// insert note?
	onEventMenu (101);
	break;
      case 'l': case 'L':			// load score?
	onEventMenu (1);
	break;
      case 's': case 'S':			// save score?
	onEventMenu (2);
	break;
      default:
	beep ();
    }
    drawAgain ();
  }

  // on selection of the menu items

  public void onEventMenu (int selection) {
    String title;

    switch (selection) {
      // load score
      case 1:
	title = scoreEdit.getText ();
	// prompt error if title is empty
	if (title.equals (""))
	  popupBox (new MessageBox ("Error", "Enter Score Title"));
	else {
	  // clear vector if count is not 0
	  if (notesVec.getCount () != 0)
	     notesVec.clear ();
	  // call loadSong method
	  notesVec = Score.loadSong (title);
	  // prompt error if no song if found
	  if (notesVec.getCount () == 0)
	    popupBox (new MessageBox ("Error", "No Such Score"));
	  // if song found, reassign the start distance for further editing
	  else {
	     drawKeySignature = true;
	     reAssignDistance ();
	     drawAgain ();
	  }
	}
	break;

      // save score
      case 2:
	onEvent (new ControlEvent (ControlEvent.PRESSED, scoreSave));
	break;

      // clear score
      case 3:
	notesVec.clear ();
	drawKeySignature = false;
	startDistance = 20;
	drawAgain ();
	break;

      // display the about/help message box
      case 4:
	popupBox (aboutMessage);
	break;

      // insert note
      case 101:
	insertNote ();
	break;

      // delete note
      case 102:
	deleteNote ();
	break;
    }
  }

  // when the pen is down

  public void onEventPenDown (Event e) {
    PenEvent pe = (PenEvent) e;
    g.drawArrow (lastCaretX, 15, 3, g.ARROW_DOWN, true, true,
      getBackColor ());
    this.lastX = pe.x;
    this.lastY = pe.y;

    // perform the following only if user is on the score page
    if (page == SCORE_PAGE) {
       // draw pointer only if condition fulfilled
       if (lastX <= startDistance && lastX >= 10 && lastY < 98) {
	  lastCaretX = checkPointPos ();
	  g.drawArrow (lastCaretX, 15, 3, g.ARROW_DOWN, true, true,
	    Color.BLACK);
       }
       else {
	  // check which note is selected
	  checkSelectedSymbol (lastX, lastY);

	  // for semibreve rest
	  if (noteSelected == Score.BREVE_REST) {
	     doRestAssign (46);
	     Score.drawBreveRest (g, startDistance, 46);
	     noteSelected = 0;
	  }
	  // for minim rest
	  else if (noteSelected == Score.MINIM_REST) {
	     doRestAssign (49);
	     Score.drawMinimRest (g, startDistance, 49);
	     noteSelected = 0;
	  }
	  // for crotchet rest
	  else if (noteSelected == Score.CROTCHET_REST) {
	     doRestAssign (46);
	     Score.drawCrotchetRest (g, startDistance-2, 46);
	     noteSelected = 0;
	  }
	  // for quaver rest
	  else if (noteSelected == Score.QUAVER_REST) {
	     doRestAssign (49);
	     Score.drawQuaverRest (g, startDistance-2, 49);
	     noteSelected = 0;
	  }
	  // if no accidentals are selected, proceed with the note assignment
	  else if (noteSelected != Score.FLAT && noteSelected != Score.SHARP &&
		   noteSelected != Score.NATURAL) {
	     doAssignment (lastY, false);
	     lastCaretX = startDistance - 3;
	     g.drawArrow (lastCaretX, 15, 3, g.ARROW_DOWN, true, true,
	       Color.BLACK);
	  }
       }
    }
  }

  public void onPaint (Graphics g) {
    if (redraw) {
      if (page == MAIN_PAGE)
	toStart ();
      else if (page == SCORE_PAGE)
	toScore (g);
      else if (page == CHORDDB_PAGE)
	toChordDatabase (g);
      else if (page == CHORDQUIZ_PAGE)
	toChordQuiz (g);
      redraw = false;
    }
  }

  public void onStart () {
    page = MAIN_PAGE;
    redraw = true;
  }

  // pop up message box

  void popupBox (MessageBox box) {
    box.setBackColor (MESSAGE_COLOUR);
    beep ();
    popupModal (box);
  }

  // does the posting of questions

  public void postQuestion () {
    int keyindex, chordIndex;
    String chorddet, key, chordname;
    String[] tmparray;
    Random rnd = new Random ();

    key = "";
    // get a key index randomly
    keyindex = rnd.rand (0, 12);
    // get the key in string
    key = ChordDB.getKey (keyindex);

    // get the scale of the key
    chordStructDetails = ChordDB.getScale (keyindex);

    if (!chordStructDetails.equals ("")) {
      // get a chord index randomly
      chordIndex = rnd.rand (0, CreateChords.CHORDS);
      // get the details of the chord
      chorddet = ChordDB.getChordDetails (chordIndex);
      // split it into individual details
      tmparray = Convert.tokenizeString (chorddet, ':');
      chordname = tmparray[0];

      // generate all the answers
      generateAll (key, chordname, chordIndex);

      // call method to split the details and display them
      // on the keyboard
      splitDetails (chorddet);
      processChordStructure ();
    }
  }

  // process the chord details, and display the notes on the keyboard

  public void processChordStructure () {
    String[] scaleArray;
    int length, x = 0, y = 0, index, oldy = 0;
    String[] scaleNote;
    String note;

    chordNotes = "";
    // clear the vector if it is not empty
    if (notesVec.getCount () != 0)
      notesVec.clear ();

    // get the interval array length
    length = intervalArray.length;
    if (!chordStructDetails.equals ("")) {
       scaleArray = Convert.tokenizeString (chordStructDetails, ':');

       // loop the interval array and display the notes
       for (int i = 0; i < length; i++) {
	  // for intervals that is double flat (bb)
	  if (intervalArray[i].startsWith ("bb")) {
	     index = Convert.toInt (intervalArray[i].substring (2));
	     scaleNote = Convert.tokenizeString (scaleArray[index - 1], ',');
	     if (scaleNote[0].endsWith ("#"))
		note = scaleNote[0].substring (0, 1) + "b";
	     else
		note = scaleNote[0] + "bb";
	     oldy = Convert.toInt (scaleNote[2]);
	     if (note.equals ("Bb") || note.equals ("Cb") ||
		 note.equals ("Eb") || note.equals ("Fb")) {
		x = Convert.toInt (scaleNote[1]) - 12;
		if (oldy == 30)
		   y = 44;
		else
		   y = 30;
	     }
	     // don't change the Y position for double flat
	     // if it's not any of the 4 notes above
	     else {
		x = Convert.toInt (scaleNote[1]) - 8;
		y = oldy;
	     }
	     // add the notes to the vector
	     if (y == 30)
	       addBlackKey (x);
	     else
	       addWhiteKey (x);
	  }
	  // for intervals that is flat (b)
	  else if (intervalArray[i].startsWith ("b")) {
	     index = Convert.toInt (intervalArray[i].substring (1));
	     scaleNote = Convert.tokenizeString (scaleArray[index - 1], ',');
	     if (scaleNote[0].endsWith ("#"))
		note = scaleNote[0].substring (0, 1);
	     else
		note = scaleNote[0] + "b";
	     oldy = Convert.toInt (scaleNote[2]);
	     // don't change the Y position if it's either of these three notes
	     if (note.equals ("Cb") || note.equals ("E") ||
		 note.equals ("Fb") || note.equals ("B")) {
		x = Convert.toInt (scaleNote[1]) - 8;
		y = oldy;
	     }
	     else {
		x = Convert.toInt (scaleNote[1]) - 4;
		if (oldy == 30)
		   y = 44;
		else if (oldy == 44)
		   y = 30;
	     }
	     // add the notes to the vector
	     if (y == 30)
	       addBlackKey (x);
	     else
	       addWhiteKey (x);
	  }
	  // for intervals that are sharp (#)
	  else if (intervalArray[i].startsWith ("#")) {
	    index = Convert.toInt (intervalArray[i].substring (1));
	    scaleNote = Convert.tokenizeString (scaleArray[index - 1], ',');
	    if (scaleNote[0].endsWith ("b"))
	       note = scaleNote[0].substring (0, 1);
	    else
	       note = scaleNote[0] + "#";

	    oldy = Convert.toInt (scaleNote[2]);
	    // don't change the Y position if it's either of these two notes
	    if (note.equals ("B#")|| note.equals ("E#")) {
	       x = Convert.toInt (scaleNote[1]) + 8;
	       y = oldy;
	    }
	    else {
	       x = Convert.toInt (scaleNote[1]) + 4;

	       if (oldy == 30)
		  y = 44;
	       else if (oldy == 44)
		  y = 30;
	    }
	     // add the notes to the vector
	    if (y == 30)
	      addBlackKey (x);
	    else
	      addWhiteKey (x);
	 }
	 // for the optional intervals
	 else if (intervalArray[i].startsWith ("(")) {
	    index = Convert.toInt (intervalArray[i].
	      substring (1, intervalArray[i].length () - 1));
	    scaleNote = Convert.tokenizeString (scaleArray[index - 1], ',');
	    note = scaleNote[0];
	    x = Convert.toInt (scaleNote[1]);
	    y = Convert.toInt (scaleNote[2]);

	     // add the notes to the vector
	    if (y == 30)
	      addBlackKey (x);
	    else
	      addWhiteKey (x);

	 }
	 // for intervals with no accidental
	 else {
	    index = Convert.toInt (intervalArray[i]);
	    scaleNote = Convert.tokenizeString (scaleArray[index - 1], ',');
	    note = scaleNote[0];
	    x = Convert.toInt (scaleNote[1]);
	    y = Convert.toInt (scaleNote[2]);

	     // add the notes to the vector
	    if (y == 30)
	      addBlackKey (x);
	    else
	      addWhiteKey (x);
	 }
	 g.setForeColor (Color.WHITE);
	 g.drawCircle (x, y, 2);
	 if (i == (length - 1))
	    chordNotes = chordNotes + note;
	 else
	    chordNotes = chordNotes + note + ",";

      }
      chordNotes = "Notes: " + chordNotes;
      g.setForeColor (Color.BLACK);
      g.drawText (chordNotes, 5, 53);
    }
  }

  public void processDetails (int chordIndex) {
    // remove chord details
    chordDatabaseDetails.removeAll ();

    // initialise the variables again
    chordName = " ";
    chordAKA = " ";
    chordIntervals = " ";
    chordNotes = " ";

    // get the chord details and split it into the respective fields
    chordDetails = ChordDB.getChordDetails (chordIndex);

    // proceed only if the details are not null
    if (chordDetails != null) {
       splitDetails (chordDetails);
       // add to list box
       chordDatabaseDetails.add (chordName);
       if (!chordAKA.equals ("")) {
	 String akaList[] = Convert.tokenizeString (chordAKA, ',');
	 for (int i = 0; i < akaList.length; i++)
	   chordDatabaseDetails.add ("A.K.A:   " + akaList[i]);
       }
       chordDatabaseDetails.add (chordType);
       chordDatabaseDetails.add (chordIntervals);

       // draw the notes on the keyboard
       chordStructDetails = ChordDB.getScale (keyIndex);
    }
  }

  // re-assigns the distance of the note after loading of a song is done

  public void reAssignDistance () {
    int count, distance;
    String[] tmparray;

    // get the number of items in the vector array
    count = notesVec.getCount ();

    // get the last item of the array
    tmparray = Convert.tokenizeString ((String) notesVec.get (count-1), ':');

    // get the distance field
    distance = Convert.toInt (tmparray[1]);

    // reassign the note position
    startDistance = distance;
  }

  // highlights the crotchet

  public void selectCrotchet () {
    g.drawHighLightFrame (40, 98, 16, 18, Color.WHITE, Color.WHITE, true);
  }

  // highlights the crotchet rest

  public void selectCrotchetRest () {
    g.drawHighLightFrame (25, 115, 16, 16, Color.WHITE, Color.WHITE, true);
  }

  // highlights the dotted crotchet

  public void selectDottedCrotchet () {
    g.drawHighLightFrame (55, 98, 16, 18, Color.WHITE, Color.WHITE, true);
  }

  // highlights the dotted minim

  public void selectDottedMinim () {
    g.drawHighLightFrame (25, 98, 16, 18, Color.WHITE, Color.WHITE, true);
  }

  // highlights the flat

  public void selectFlat () {
    g.drawHighLightFrame (0, 115, 11, 16, Color.WHITE, Color.WHITE, true);
  }

  // highlights the minim

  public void selectMinim () {
    g.drawHighLightFrame (10, 98, 16, 18, Color.WHITE, Color.WHITE, true);
  }

  // highlights the minim rest

  public void selectMinimRest () {
    g.drawHighLightFrame (40, 115, 16, 16, Color.WHITE, Color.WHITE, true);
  }

  // highlights the natural

  public void selectNatural () {
    g.drawHighLightFrame (85, 115, 16, 16, Color.WHITE, Color.WHITE, true);
  }

  // highlights the quaver

  public void selectQuaver () {
    g.drawHighLightFrame (70, 98, 16, 18, Color.WHITE, Color.WHITE, true);
  }

  // highlights the quaver rest

  public void selectQuaverRest () {
    g.drawHighLightFrame (70, 115, 16, 16, Color.WHITE, Color.WHITE, true);
  }

  // highlights the semibreve

  public void selectSemiBreve () {
    g.drawHighLightFrame (0, 98, 11, 18, Color.WHITE, Color.WHITE, true);
  }

  // method highlights the semibreve rest

  public void selectSemiBreveRest () {
    g.drawHighLightFrame (55, 115, 16, 16, Color.WHITE, Color.WHITE, true);
  }

  // highlights the semiquaver

  public void selectSemiQuaver () {
    g.drawHighLightFrame (85, 98, 16, 18, Color.WHITE, Color.WHITE, true);
  }

  // highlights the sharp

  public void selectSharp () {
    g.drawHighLightFrame (10, 115, 16, 16, Color.WHITE, Color.WHITE, true);
  }

  // split the detail string into different fields and assign
  // to the respective variables

  public void splitDetails (String chordDetails) {
    String[] details;
    String[] akaArray;
    String akaList = "";
    int length;

    if ((!chordDetails.equals ("")) && (chordDetails != null)) {
       details = Convert.tokenizeString (chordDetails, ':');
       chordName = "Name: " + selectedKey + " " + details[0];
       if (!details[1].equals ("")) {
	 akaArray = Convert.tokenizeString (details[1], ',');
	 length = akaArray.length;
	 for (int i = 0; i < length; i++) {
	   if (i == (length - 1))
	      akaList = akaList + selectedKey + " " + akaArray[i];
	   else
	      akaList = akaList + selectedKey + " " + akaArray[i] + ",";
	 }
       }
       chordAKA =  akaList;
       chordType = "Type:  " + details[2];
       intervals = details[3];
       intervalArray = Convert.tokenizeString (intervals, ',');
       chordIntervals = "Intervals: " + intervals;
    }
  }

  // paint the chord database widgets

  public void toChordDatabase (Graphics g) {
    g.setBackColor (Color.WHITE);
    g.setForeColor (Color.BLACK);
    g.drawText ("Chord database", 0, 0);
    g.drawText ("Key", 5, 63);
    g.drawText ("Chord", 90, 63);

    // add widgets
    add (chordDatabaseKey);
    add (chordDatabaseChord);
    add (chordDatabaseDetails);
    add (chordDatabasePlay);
    add (chordDatabaseDone);

    ChordDB.drawKeyboard (g);

    // display the notes on the keyboard to cater for repaint ()
    if (intervalArray != null) {
      processChordStructure ();
      intervalArray = null;
    }
  }

  public void toChordQuiz (Graphics g) {
    int xpos = 12, ypos = 85;

    // set the text of the button to empty if they are not
    if (!firstAns.getText ().equals (""))
      firstAns.setText ("");
    if (!secondAns.getText ().equals (""))
      secondAns.setText ("");
    if (!thirdAns.getText ().equals (""))
      thirdAns.setText ("");

    // display the text using drawText
    g.setBackColor (Color.WHITE);
    g.setForeColor (Color.BLACK);
    g.drawText ("Chord Quiz", 0, 0);
    g.drawText ("Key: ", 5, 65);
    g.drawText ("What chord is this?", 5, 75);
    g.drawText ("Incorrect:", 50, 0);
    g.drawText (Convert.toString (inCorrect), 95, 0);
    g.drawText ("Correct:", 110, 0);
    g.drawText (Convert.toString (correct), 150, 0);
    g.drawLine (0, 135, 160, 135);
    g.drawText ("1. ", 5, 85);
    g.drawText ("2. ", 5, 100);
    g.drawText ("3. ", 5, 115);

    // set up answer buttons
    firstAns.setRect (xpos, ypos, 90, 15);
    firstAns.setBorder (Button.BORDER_NONE);
    ypos = ypos + 15;
    secondAns.setRect (xpos, ypos, 90, 15);
    secondAns.setBorder (Button.BORDER_NONE);
    ypos = ypos + 15;
    thirdAns.setRect (xpos, ypos, 90, 15);
    thirdAns.setBorder (Button.BORDER_NONE);

    // add widgets
    add (chordQuizDone);
    add (chordQuizPlay);
    add (chordQuizNext);
    add (firstAns);
    add (secondAns);
    add (thirdAns);

    // reuse the same graphics from chord database
    ChordDB.drawKeyboard (g);

    // to cater for repaint, draw the notes of the chord again
    if (intervalArray != null) {
      processChordStructure ();
    }

    // to cater for repaint too, making sure the question is present
    if (questionArray[0] != null)
      displayQuestions ();
  }

  // paint the score's widgets

  public void toScore (Graphics g) {
    int startingPos = 20;

    // menu items
    String MenuCol0[] =
      {"Score", "Load      L", "Save      S", "Clear     C", "About   A"};
    String MenuCol1[] =
      {"Note", "Insert    I", "Delete  D"};

    setMenuBar (mbar = new MenuBar (new String[][] {MenuCol0, MenuCol1}));

    // display title and page number
    g.setForeColor (Color.BLACK);
    g.drawText ("Score", 0, 0);
    g.drawText (Convert.toString (pageCount), 70, 85);

    // add widgets
    add (scoreEdit);
    add (scoreDone);
    add (scorePlay);
    add (scoreSave);
    left.setBorder (Button.BORDER_NONE);
    add (left);
    right.setBorder (Button.BORDER_NONE);
    add (right);
    add (keySignatureBox);

    Score.drawScore (g, pageCount);

    // items that are displayed only on first page
    if (pageCount == 1) {
      left.setVisible (false);		// only visible on 2nd page onwards
      Score.drawTrebleClef (g);
      if (drawKeySignature)		// key signature is fixed once selected
	 startingPos = assignStartPos ();
    }
    else
      left.setVisible (true);

    if (notesVec != null)
      drawNotes ();

    // draw pointer only if condition fulfils
    if ((lastX <= startDistance && lastX > 20) && (lastY < 98)) {
      lastCaretX = checkPointPos ();
      g.drawArrow (lastCaretX, 15, 3, g.ARROW_DOWN, true, true, Color.BLACK);
    }

    // assign the starting position of the notes
    // to cater for different key signatures
    if (notesVec.getCount () == 0) {
      startDistance = startingPos;
    }
  }

  // paint the start screen's widgets

  public void toStart () {
    // display title and add widgets
    g.setForeColor (Color.BLACK);
    g.drawText ("CHORD TUTOR", 50, 5);
    add (mainExit);
    add (mainScore);
    add (mainChordDB);
    add (mainChordQuiz);
  }

  // clear the screens

  public void wipeScreen () {
    // clear the start page
    if (page == MAIN_PAGE) {
      remove (mainExit);
      remove (mainScore);
      remove (mainChordDB);
      remove (mainChordQuiz);
    }
    // clear the score page
    else if (page == SCORE_PAGE) {
      remove (scoreEdit);
      remove (scoreDone);
      remove (scorePlay);
      remove (scoreSave);
      remove (left);
      remove (right);
      remove (keySignatureBox);
    }
    // clear the chord database page
    else if (page == CHORDDB_PAGE) {
      remove (chordDatabaseDone);
      remove (chordDatabasePlay);
      remove (chordDatabaseChord);
      remove (chordDatabaseKey);
      remove (chordDatabaseDetails);
    }
    // clear chord quiz page
    else if (page == CHORDQUIZ_PAGE) {
      remove (chordQuizDone);
      remove (chordQuizPlay);
      remove (chordQuizNext);
      remove (chordQuizChord);
      remove (firstAns);
      remove (secondAns);
      remove (thirdAns);
    }
  }
}

