// MidiConvert.java	(C) Y. P. Tay, K. J. Turner	31/07/03

// Self-standing program to convert the "ChordTutorDB" database to MIDI

import waba.fx.*;
import waba.sys.*;
import waba.util.*;
import waba.ui.*;
import waba.io.*;
import java.io.*;

public class MidiConverter extends MainWindow {

  private static final String CATALOG_NAME = "ChordTutorDB.ChTu.DATA";
  private Catalog cat;
  private ResizeStream rs;
  private DataStream ds;
  private Vector titleArray = new Vector();
  private ListBox titleLB = new ListBox();
  private Label lb = new Label ("Choose title to convert");
  private Button exitBut = new Button ("Exit");
  private Button convertBut = new Button ("Convert");
  private int numRecords = 0;

  // MIDI notes in decimal:Frequency in Superwaba
  private final static String[] notes = {
   "55:196", "56:208", "57:220", "58:233", "59:247", "60:262", "61:277",
   "62:294", "63:311", "64:330", "65:349", "66:370", "67:392", "68:415",
   "69:440", "70:466", "71:494", "72:523", "73:554", "74:587", "75:622",
   "76:659", "77:698", "78:740", "79:784", "80:831", "81:880", "82:932",
   "83:988", "84:1046", "85:1109"
  };

  // MIDI delta time:Duration in Superwaba
  private final static String[] durations = {
   "216:1600", "162:1200", "108:800", "144:600", "96:400", "48:200", "24:100"
  };

  public MidiConverter () {
    setDoubleBuffer(true);
    setBorderStyle(TAB_ONLY_BORDER);
    setTitle("MIDI Converter");

    exitBut.setRect(138, 143, 20, 15);
    add(exitBut);

    convertBut.setRect(95, 143, 38, 15);
    add(convertBut);

    titleLB.setRect(5, 50, 80, 110);
    add(titleLB);

    lb.setRect(5, 30, 100, 15);
    add(lb);
  }

  public void onStart () {
    int i;

    openCatalog();
    addTitle();
    for(i = 0; i < titleArray.getCount(); i++)
      titleLB.add(titleArray.get(i));
  }

  public void onEvent (Event e) {
    if (e.type == ControlEvent.PRESSED) {
      if (e.target == exitBut) {
	closeCatalog ();
	exit (0);
      }
      else if (e.target == titleLB) {
	Object element = titleLB.getSelectedItem ();
	Vm.debug("element : " + element);
      }
      else if (e.target == convertBut) {
	Object element = titleLB.getSelectedItem ();
	if (!element.equals (""))
	  writeSong ((String) element);
	else
	  Vm.debug ("choose a title");
      }
    }
  }

  public void openCatalog() {
    cat = new Catalog (CATALOG_NAME, Catalog.READ_WRITE);
    if (!cat.isOpen ()) {
      Vm.debug("can't open pdb file");
      exit(0);
    }
    else
      numRecords = cat.getRecordCount ();
    rs = new ResizeStream (cat, 512);
    ds = new DataStream (rs);
  }

  public void closeCatalog () {
    rs.close ();
    ds.close ();
  }

  public void addTitle () {
    String[] tmparray;
    int count = 0, numNotes;
    int i = 0;

    while (i < numRecords) {
      cat.setRecordPos(i);
      String songInfo = ds.readString();
      Vm.debug("song info : " + songInfo);
      tmparray = Convert.tokenizeString(songInfo, ',');
      titleArray.add(tmparray[0]);
      numNotes = Convert.toInt(tmparray[2]);
      i = i + numNotes + 1;
      count++;
    }
  }

  // convert SuperWaba note length as string to corresponding MIDI duration as
  // integer, returning 0 if length is not found

  public int noteDuration (String length) {
    String durationMap;
    int i = 0, duration = 0;

    while (i < durations.length) {
     durationMap = durations[i];
     if (durationMap.endsWith (":" + length)) {
       duration =
	 Convert.toInt (durationMap.substring (0, durationMap.indexOf (':')));
       i = durations.length;			// force immediate loop exit
     }
     else
       i++;
    }
    if (duration == 0)
     Vm.debug ("Unknown duration " + duration);
    return (duration);
  }

  // convert SuperWaba frequency as string to corresponding MIDI note as
  // integer, returning 0 if frequency is not found

  public int noteNumber (String frequency) {
    String noteMap;
    int i = 0, number = 0;

    while (i < notes.length) {
     noteMap = notes[i];
     if (noteMap.endsWith (":" + frequency)) {
       number =
	 Convert.toInt (noteMap.substring (0, noteMap.indexOf (':')));
       i = notes.length;			// force immediate loop exit
     }
     else
       i++;
    }
    if (number == 0)
      Vm.debug ("Unknown frequency " + frequency);
    return (number);
  }

  // write MIDI header to given file for given note count (max 1819, i.e.
  // length of two bytes)

  public void writeHeader (FileOutputStream file, int count)
   throws IOException {
    byte[] songHead = "MThd".getBytes();	// MIDI song header
    byte[] songLength = {0, 0, 0, 6};		// header size
    byte[] format = {0, 0};			// type 0
    byte[] track = {0, 1};			// one track
    byte[] division = {0, 96};			// delta time 96
    byte[] trackHead = "MTrk".getBytes();	// MIDI track header
    byte[] trackLength = {0, 0, 0, 0};		// track size (set below)

    file.write (songHead, 0, 4);
    file.write (songLength, 0, 4);
    file.write (format, 0, 2);
    file.write (track, 0, 2);
    file.write (division, 0, 2);
    file.write (trackHead, 0, 4);
    if (count < 1820) {
      count = 9 * count + 4;			// 9 bytes/note, 4 bytes trailer
      trackLength[2] = (byte) (count >> 8);
      trackLength[3] = (byte) (count & 255);
      file.write (trackLength, 0, 4);
    }
    else
      Vm.debug ("Too many notes " + count);
  }

  // write MIDI note to given file for given number and duration (max 16383)

  public void writeNote (FileOutputStream file, int number, int duration)
   throws IOException {
    file.write (00);				// delta time 0
    file.write (144);				// start note, channel 0
    file.write (number);			// note number
    file.write (127);				// note velocity
    if (duration < 16384) {
      file.write (128 + (duration >> 7));	// duration upper 7 bits
      file.write (duration & 127);		// duration lower 7 bits
    }
    else
      Vm.debug ("Note too long " + duration);
    file.write (128);				// stop note, channel 0
    file.write (number);			// note number
    file.write (00);				// note velocity
  }

  // write MIDI song file for given title

  public void writeSong (String title) {
    FileOutputStream file;
    String[] tmpArray;
    String[] noteArray;
    String songTitle;
    int i = 0, j;
    int number, duration, length;

    while (i < numRecords) {
      cat.setRecordPos (i);
      String songInfo = ds.readString ();
      tmpArray = Convert.tokenizeString (songInfo, ',');
      songTitle = tmpArray[0];
      if (songTitle.equalsIgnoreCase (title.trim ())) {
	try {
	 file = new FileOutputStream (songTitle + ".mid");
	 length = tmpArray.length;
	 writeHeader (file, length - 2);
	 for (j = 2; j < length; j++) {
	   noteArray = Convert.tokenizeString(tmpArray[j], ':');
	   number = noteNumber (noteArray[3]);
	   duration = noteDuration (noteArray[4]);
	   writeNote (file, number, duration);
	  }
	  writeTrailer (file);
	  file.close ();
	}
	catch (IOException exc) {
	 System.err.println ("File I/O Exception " + exc);
	}
	break;
      }
      else
	i++;
    }
  }

  // write MIDI trailer to given file

  public void writeTrailer (FileOutputStream file) throws IOException {
    byte[] trackEnd = {00, (byte) 255, 47, 00};

    file.write(trackEnd, 0, 4);
  }
}
