################################### Header ###################################

# "dill_flipflop_bb.m4"	Ji He, K. J. Turner	27/01/98

# This "m4" macro file contains templates for specification of digital logic
# components and circuits in LOTOS according to the DILL (Digital Logic in
# LOTOS) approach. Bit states (but not signals) may have an "X" (unknown)
# value - generally for their initial condition.
#
# This particular file deals with the black-box form of flip-flops (D, T, MS,
# JK).

# Copyright 1998 Ji He, K. J. Turner, University of Stirling

################################# Flip-Flops #################################

# "DFlipFlop_BB_Decl" defines a edge_triggered or master-slave  D flip-flop 

define(DFlipFlop_BB_Pos_Decl, `declare(`$0', `DFlipFlop_BB_Decl
  process DFlipFlop_BB_Pos [D, C, Q, Qbar] : noexit :=
    DFlipFlop_BB [D, C, Q, Qbar] (1 of bit)
  endproc (* DFlipFlop_Pos *)
')')

define(DFlipFlop_BB_Neg_Decl, `declare(`$0', `DFlipFlop_BB_Decl
  process DFlipFlop_BB_Neg [D, C, Q, Qbar] : noexit :=
    DFlipFlop_BB [D, C, Q, Qbar] (0 of bit)
  endproc (* DFlipFlop_Neg *)
')')

define(DFlipFlop_BB_Decl, `declare(`$0', `
  process DFlipFlop_BB [D, C, Q, Qbar] (edge : Bit) : noexit :=
    DFlipFlop_BB_Aux [D, C, Q, Qbar]
     (edge, X of bit, X of bit, X of bit, X of bit)
  
    where

    process DFlipFlop_BB_Aux [D, C, Q, Qbar]
     (edge, dtD, dtC, dtQ, dtQbar: Bit) : noexit :=
      D ? newdtD : Bit;
      DFlipFlop_BB_Aux [D, C, Q, Qbar] (edge, newdtD, dtC, dtQ, dtQbar)
    []
      C ? newdtC : Bit;  
      (   
	[(dtC eq not (edge)) and (newdtC eq edge)] -> (* appropriate pulse transition *)
	  (
	    [(dtD eq X of bit) and (dtQ ne X) ] ->   
	      DFlipFlop_BB_Aux [D, C, Q, Qbar] (edge, dtD, newdtC, dtQ, dtQbar)  
	  []
	    [(dtD eq X of bit) and (dtQ eq X of bit)] ->
	      (
	        Q ? newdtQ : Bit [newdtQ ne X of bit]; Qbar ! not (newdtQ);       
		DFlipFlop_BB_Aux [D, C, Q, Qbar]
		 (edge, dtD, newdtC, newdtQ, not (newdtQ))
	      []
		Qbar ? newdtQbar: Bit [newdtQbar ne X of bit];
		Q ! not (newdtQbar);
		DFlipFlop_BB_Aux [D, C, Q, Qbar]
		 (edge, dtD, newdtC, not (newdtQbar), not (dtQbar))  
	      )
	  []  
	    [(dtD ne X of bit) and (dtD ne dtQ) ] ->         
	      (
	        (Q ! dtD;exit ||| Qbar ! not (dtD); exit)
	      >>
		DFlipFlop_BB_Aux [D, C, Q, Qbar]
		 (edge, dtD, newdtC, dtD, not (dtD))
	      )
	  []
	    [(dtD ne X of bit) and (dtD eq dtQ) ] ->
	      DFlipFlop_BB_Aux [D, C, Q, Qbar] (edge, dtD, newdtC, dtQ, dtQbar)          
	  ) (* end of appropriat clock transition *)
      []
	[(dtC ne not (edge)) or (newdtC ne edge)] -> 
	  DFlipFlop_BB_Aux [D, C, Q, Qbar] (edge, dtD, newdtC, dtQ, dtQbar)
      )				(* end of C ? --- *)
    [] (* initial output *)
      [(dtQ eq X)] ->
	(
	  Q ? newdtQ : bit [newdtQ ne X];
	  Qbar ! not(newdtQ);
	  DFlipFlop_BB_Aux [D, C, Q, Qbar]
	    (edge, dtD, dtC, newdtQ, not (newdtQ))
	[]
	  Qbar ? newdtQbar : bit [newdtQbar ne X]; Q ! not (newdtQbar);
	  DFlipFlop_BB_Aux [D, C, Q, Qbar] 
	    (edge, dtD, dtC, not (newdtQbar), newdtQbar)
	)  
    endproc (* DFlipFlop_BB_Aux*)
  endproc (* DFlipFlop_BB *)
')')

# "TFlipFlop_BB_Decl" defines a edge_triggered T flip-flop

define(TFlipFlop_BB_Pos_Decl, `declare(`$0', `TFlipFlop_BB_Decl
  process TFlipFlop_BB_Pos [C, Q, Qbar] : noexit :=
    TFlipFlop_BB [C, Q, Qbar] (1 of bit)
  endproc (* TFlipFlop_Pos *)
')')

define(TFlipFlop_BB_Neg_Decl, `declare(`$0', `TFlipFlop_BB_Decl
  process TFlipFlop_BB_Neg [C, Q, Qbar] : noexit :=
    TFlipFlop_BB [C, Q, Qbar] (0 of bit)
  endproc (* TFlipFlop_Neg *)
')')

define(TFlipFlop_BB_Decl, `declare(`$0', `
  process TFlipFlop_BB [C, Q, Qbar] (edge: Bit) : noexit :=
    TFlipFlop_BB_Aux [C, Q, Qbar] (edge, X of bit, X of bit, X of bit)
  
    where
  
    process TFlipFlop_BB_Aux [C, Q, Qbar] (edge, dtC, dtQ, dtQbar: Bit) : noexit :=
    
      C ? newdtC : Bit;  
      (   
	  [(dtC eq not (edge)) and (newdtC eq edge)] -> (* appropriate pulse transition *)
	    (
	      [(dtQ eq X of bit) ] ->  (* output Q has not been initialized *)
	        (
		  (
		    Q ? newdtQ : Bit [newdtQ ne X]; exit (newdtQ, any bit)
		  |||
		    Qbar ? newdtQbar : Bit [newdtQbar ne X];
		    exit (any bit, newdtQbar)
		  )
		>>
		  accept newdtQ, newdtQbar : Bit in 
	            TFlipFlop_BB_Aux [C, Q, Qbar]
		     (edge, newdtC, newdtQ, newdtQbar) 
	        )         
	    []				(* output toggle *)
	      [(dtQ ne X of bit) and (dtQbar ne dtQ)] -> 
	        (
		  (Q ! not (dtQ); exit ||| Qbar ! dtQ; exit)
	        >> 
	          TFlipFlop_BB_Aux [C, Q, Qbar] (edge, newdtC, not (dtQ), dtQ)
	        )
	    []
	      [(dtQ ne X of bit) and (dtQbar eq dtQ)] ->
	      (* if Q and Qbar has the same value at the beginnig time *)
		(
		  i;
		  Q ! not (dtQ);
		  TFlipFlop_BB_Aux [C, Q, Qbar]
		   (edge, newdtC, not (dtQ), dtQbar)
		[]
		  i;
		  Qbar ! not (dtQbar);
		  TFlipFlop_BB_Aux [C, Q, Qbar]
		   (edge, newdtC, dtQ, not (dtQbar))
		)
	    )
	  []
	  [(dtC eq edge) or (dtC eq X of bit) or (newdtC ne edge) ] ->
	    TFlipFlop_BB_Aux [C, Q, Qbar] (edge, newdtC, dtQ, dtQbar)
      )
    endproc (* TFlipFlop_BB_Aux*)
  endproc (* TFlipFlop_BB *)
')')

# "JKFlipFlop_BB_Decl" defines a edge_triggered JK flip-flop

define(JKFlipFlop_BB_Pos_Decl, `declare(`$0', `JKFlipFlop_BB_Decl
 process JKFlipFlop_BB_Pos [J, K, C, Q, Qbar] : noexit :=
    JKFlipFlop_BB [J, K, C, Q, Qbar] (1 of bit)
  endproc (* JKFlipFlop_Pos *)
')')

define(JKFlipFlop_BB_Neg_Decl, `declare(`$0', `JKFlipFlop_BB_Decl
  process JKFlipFlop_BB_Neg [J, K, C, Q, Qbar] : noexit :=
    JKFlipFlop_BB [J, K, C, Q, Qbar] (0 of bit)
  endproc (* JKFlipFlop_Neg *)
')')

define(JKFlipFlop_BB_Decl, `declare(`$0', `
  process JKFlipFlop_BB [J, K, C, Q, Qbar] (edge: Bit) : noexit :=
    JKFlipFlop_BB_Aux [J, K, C, Q, Qbar]
     (edge, X of bit, X of bit, X of bit, X of bit, X of bit)
  
    where

    process JKFlipFlop_BB_Aux [J, K, C, Q, Qbar]
     (edge, dtJ, dtK, dtC, dtQ, dtQbar: Bit) : noexit :=
      J ? newdtJ: Bit;
      JKFlipFlop_BB_Aux [J, K, C, Q, Qbar] (edge, newdtJ, dtK, dtC, dtQ, dtQbar)
    []
      K ? newdtK: Bit;
      JKFlipFlop_BB_Aux [J, K, C, Q, Qbar] (edge, dtJ, newdtK, dtC, dtQ, dtQbar)
    []
      C ? newdtC: Bit;
      (
	[(dtC eq not (edge)) and (newdtC eq edge)] -> (* approprate clock transition *)
	  (
	    [((dtJ eq X of bit) or (dtK eq X of bit)) and (dtQ ne X)] -> 
	      JKFlipFlop_BB_Aux [J, K, C, Q, Qbar]
	       (edge, dtJ, dtK, newdtC, dtQ, dtQbar)
	  []
	    [((dtJ eq X of bit) or (dtK eq X of bit)) and (dtQ eq X)] ->
	      (
	        (
		  Q ? newdtQ: Bit [newdtQ ne X]; exit (newdtQ, any bit)
		|||
		  Qbar ? newdtQbar : Bit [newdtQbar ne X];
		  exit (any bit, newdtQbar)
	        )
	      >>
		accept newdtQ, newdtQbar : bit in
		  JKFlipFlop_BB_Aux [J, K, C, Q, Qbar]
		   (edge, dtJ, dtK, newdtC, newdtQ, newdtQbar)     
	      )
	  []			(* J=1 of bit, K=0 *)
	    [(dtJ eq 1) and (dtK eq 0 of bit) and 
	     (dtQ ne 1) and (dtQbar ne 0 of bit)] ->      
	      (
	        (Q ! 1 of bit; exit ||| Qbar ! 0 of bit;exit) 
	      >>
		JKFlipFlop_BB_Aux [J, K, C, Q, Qbar] 
		 (edge, dtJ, dtK, newdtC, 1 of bit, 0 of bit)
	      )
	  []      
	    [(dtJ eq 1 of bit) and (dtK eq 0 of bit) 
	     and (dtQ eq 1) and (dtQbar eq 0 of bit)] -> 
	      JKFlipFlop_BB_Aux [J, K, C, Q, Qbar]
	       (edge, dtJ, dtK, newdtC, dtQ, dtQbar)
	  []
	    [(dtJ eq 1) and (dtK eq 0 of bit) and (dtQ  ne 1) 
	     and (dtQbar eq 0 of bit)] ->
	      Q ! 1 of bit;
	      JKFlipFlop_BB_Aux [J, K, C, Q, Qbar]
	       (edge, dtJ, dtK, newdtC, 1 of bit, dtQbar)
	  []
	    [(dtJ eq 1) and (dtK eq 0 of bit) and (dtQ  eq 1) 
	     and (dtQbar ne 0 of bit)] ->
	      Qbar! 0 of bit;
	      JKFlipFlop_BB_Aux [J, K, C, Q, Qbar]
	       (edge, dtJ, dtK, newdtC, dtQ, 0 of bit)
	  []				(* J=0 K=1 *)
	    [(dtJ eq 0 of bit) and (dtK eq 1) and (dtQ ne 0 of bit) 
	     and (dtQbar ne 1)] ->      
	      (
	        (Q ! 0 of bit; exit ||| Qbar ! 1 of bit;exit) 
	      >>
		JKFlipFlop_BB_Aux [J, K, C, Q, Qbar] 
		 (edge, dtJ, dtK, newdtC, 0 of bit, 1 of bit)
	      )
	  []
	    [(dtJ eq 0 of bit) and (dtK eq 1) and 
	     (dtQ eq 0 of bit) and (dtQbar eq 1)] ->
	      JKFlipFlop_BB_Aux [J, K, C, Q, Qbar]
	       (edge, dtJ, dtK, newdtC, dtQ, dtQbar)
	  []
	    [(dtJ eq 0 of bit) and (dtK eq 1) and (dtQ ne 0 of bit) 
	      and (dtQbar eq 1)] ->
	      Q ! 0 of bit;
	      JKFlipFlop_BB_Aux [J, K, C, Q, Qbar]
	       (edge, dtJ, dtK, newdtC, 0 of bit, dtQbar)
	  []
	    [(dtJ eq 0 of bit) and (dtK eq 1) and (dtQ eq 0 of bit) 
	      and (dtQbar ne 1)] ->
	      Qbar ! 1 of bit;
	      JKFlipFlop_BB_Aux [J, K, C, Q, Qbar]
	       (edge, dtJ, dtK, newdtC, dtQ, 1 of bit)
	  []				(* J=0 K=0 *)
	    [(dtJ eq 0 of bit) and (dtK eq 0 of bit) and (dtQ ne X)] -> 
	      JKFlipFlop_BB_Aux [J, K, C, Q, Qbar]
	       (edge, dtJ, dtK, newdtC, dtQ, dtQbar)
	  []
	    [(dtJ eq 0 of bit) and (dtK eq 0 of bit) and (dtQ eq X)] ->
	      (
	        (
		  Q ? newdtQ: Bit [newdtQ ne X];
		  exit (newdtQ, any bit)
		|||
		  Qbar ? newdtQbar : Bit [newdtQbar ne X];
		  exit (any bit, newdtQbar)
		)
	      >>
		accept newdtQ, newdtQbar : bit in
		  JKFlipFlop_BB_Aux [J, K, C, Q, Qbar]
		   (edge, dtJ, dtK, newdtC, newdtQ, newdtQbar)
	      )
	  []				(* J=1 K=1 *)
	    [(dtJ eq 1) and (dtK eq 1) and (dtQ eq X of bit) ] ->
	      (
	        (
		  Q ? newdtQ: Bit [newdtQ ne X];
		  exit (newdtQ, any bit)
	        |||
	          Qbar ? newdtQbar : Bit [newdtQbar ne X];
		  exit (any bit, newdtQbar)
	        )
	      >>
	        accept newdtQ, newdtQbar : bit in
	          JKFlipFlop_BB_Aux [J, K, C, Q, Qbar]
		   (edge, dtJ, dtK, newdtC, newdtQ, newdtQbar)     
	      )
	  []
	    [(dtJ eq 1) and (dtK eq 1) and ((dtQ xor dtQbar) eq 1) ] ->
	      (* dtQ = 1 and dtQbar=0; or; dtQ =0 and dtQbar =1 *)
	      (
		(
		  Q ! not (dtQ);
		  exit
		|||
		  Qbar ! not (dtQbar);
		  exit
		)
	      >>
		JKFlipFlop_BB_Aux [J, K, C, Q, Qbar]
		  (edge, dtJ, dtK, newdtC, not (dtQ), not (dtQbar))
	      )
	  []
	    [(dtJ eq 1) and (dtK eq 1) and ((dtQ xor dtQbar) eq 0 of bit)] ->
				(* dtQ=dtQbar=1; or; dtQ=dtQbar=0 *)
	      (
	        i;
		Q ! not (dtQ);
	       JKFlipFlop_BB_Aux [J, K, C, Q, Qbar]
	        (edge, dtJ, dtK, newdtC, not (dtQ), dtQbar)
	      [] 
	       i;
	       Qbar ! not (dtQbar);
	       JKFlipFlop_BB_Aux [J, K, C, Q, Qbar]
	        (edge, dtJ, dtK, newdtC, dtQ, not (dtQbar))
	      )
	  )
	[]				(* inappropiate transition *)
	  [(dtC eq edge) or (dtC eq X of bit) or (newdtC ne edge) ] ->
	    JKFlipFlop_BB_Aux [J, K, C, Q, Qbar]
	     (edge, dtJ, dtK, newdtC, dtQ, dtQbar)
      )
    endproc(* JKFlipFlop_BB_Aux *)
 endproc (* JKFlipFlop_BB *)
')')

# "RSFlipFlop_BB_Decl" defines a edge_triggered or master-slave RS flip-flop

define(RSFlipFlop_BB_Pos_Decl, `declare(`$0', `RSFlipFlop_BB_Decl
 process RSFlipFlop_BB_Pos [R, S, C, Q, Qbar] : noexit :=
    RSFlipFlop_BB [R, S, C, Q, Qbar] (1 of bit)
  endproc (* RSFlipFlop_Pos *)
')')

define(RSFlipFlop_BB_Neg_Decl, `declare(`$0', `RSFlipFlop_BB_Decl
  process RSFlipFlop_BB_Neg [R, S, C, Q, Qbar] : noexit :=
    RSFlipFlop_BB [R, S, C, Q, Qbar] (0 of bit)
  endproc (* RSFlipFlop_Neg *)
')')

define(RSFlipFlop_BB_Decl, `declare(`$0', `
  process RSFlipFlop_BB [R, S, C, Q, Qbar] (edge: Bit) : noexit :=
  
    RSFlipFlop_BB_Aux [R, S, C, Q, Qbar]
     (edge, X of bit, X of bit, X of bit, X of bit, X of bit)
  
    where
  
    process RSFlipFlop_BB_Aux [R, S, C, Q, Qbar]
      (edge, dtR, dtS, dtC, dtQ, dtQbar: Bit) : noexit :=
      R ? newdtR: Bit;
      RSFlipFlop_BB_Aux [R, S, C, Q, Qbar] (edge, newdtR, dtS, dtC, dtQ, dtQbar)
    []
      S ? newdtS: Bit;
      RSFlipFlop_BB_Aux [R, S, C, Q, Qbar] (edge, dtR, newdtS, dtC, dtQ, dtQbar)
    []
      C ? newdtC: Bit;
      (
	[(dtC eq not (edge)) and (newdtC eq edge)] -> (* approprate clock transition *)
	  (
	    [((dtR eq X of bit) or (dtS eq X of bit))] -> 
	      (
	        (
		  Q ? newdtQ : Bit [newdtQ ne X];
		  exit (newdtQ, any bit)
		|||
		  Qbar ? newdtQbar : Bit [newdtQbar ne X];
		  exit (any bit, newdtQbar)
	        )
	      >>
	        accept newdtQ, newdtQbar : bit in
	          RSFlipFlop_BB_Aux [R, S, C, Q, Qbar]
		   (edge, dtR, dtS, newdtC, newdtQ, newdtQbar)    
	      )       
	  []				(* reset *)
	    [(dtR eq 1) and (dtS eq 0 of bit) and (dtQ ne 0 of bit)
	      and (dtQbar ne 1)] ->     
	      (
	        (
		  Q ! 0 of bit;
		  exit 
		|||
		  Qbar ! 1 of bit;
		  exit
		) 
	      >>
		RSFlipFlop_BB_Aux [R, S, C, Q, Qbar] 
		 (edge, dtR, dtS, newdtC, 0 of bit, 1 of bit)
	      )
	  []      
	    [(dtR eq 1) and (dtS eq 0 of bit) and (dtQ eq 0 of bit)
	     and (dtQbar eq 1)] -> 
	      RSFlipFlop_BB_Aux [R, S, C, Q, Qbar]
	       (edge, dtR, dtS, newdtC, dtQ, dtQbar)
	  []
	    [(dtR eq 1) and (dtS eq 0 of bit) and (dtQ ne 0 of bit) 
	     and (dtQbar eq 1)] ->
	      Q ! 0 of bit; 
	      RSFlipFlop_BB_Aux [R, S, C, Q, Qbar]
	       (edge, dtR, dtS, newdtC, 0 of bit, dtQbar)
	  []
	    [(dtR eq 1) and (dtS eq 0 of bit) and (dtQ eq 0 of bit) 
	     and (dtQbar ne 1)] ->
	      Qbar ! 1 of bit; 
	      RSFlipFlop_BB_Aux [R, S, C, Q, Qbar]
	       (edge, dtR, dtS, newdtC, dtQ, 1 of bit)
	  []				(* set *)
	    [(dtR eq 0 of bit) and (dtS eq 1) and (dtQ ne 1) 
	     and (dtQbar ne 0 of bit)] ->     
	      (
	        (
		  Q ! 1 of bit;
		  exit
		|||
		  Qbar ! 0 of bit;
		  exit
	        ) 
	      >>
		RSFlipFlop_BB_Aux [R, S, C, Q, Qbar] 
		 (edge, dtR, dtS, newdtC, 1 of bit, 0 of bit)
	      )
	  []
	    [(dtR eq 0 of bit) and (dtS eq 1) and (dtQ eq 1) 
	     and (dtQbar eq 0 of bit)] ->
	      RSFlipFlop_BB_Aux [R, S, C, Q, Qbar]
	       (edge, dtR, dtS, newdtC, dtQ, dtQbar)
	  []
	    [(dtR eq 0 of bit) and (dtS eq 1) and (dtQ ne 1) 
	     and (dtQbar eq 0 of bit)] ->
	      Q ! 1 of bit;
	      RSFlipFlop_BB_Aux [R, S, C, Q, Qbar]
	       (edge, dtR, dtS, newdtC, 1 of bit, dtQbar)
	  []
	    [(dtR eq 0 of bit) and (dtS eq 1) and (dtQ eq 1) 
	     and (dtQbar ne 0 of bit)] ->
	      Qbar ! 0 of bit;
	      RSFlipFlop_BB_Aux [R, S, C, Q, Qbar]
	       (edge, dtR, dtS, newdtC, dtQ, 0 of bit)
	  []				(* R=0, S=0 *)
	    [(dtR eq 0 of bit) and (dtS eq 0 of bit) and 
	     (dtQ eq 0 of bit) and (dtQbar eq 0 of bit) ] -> (* previous state nonstable *)
	      (
	        i;
		Q ! 1 of bit;
		RSFlipFlop_BB_Aux [R, S, C, Q, Qbar]
		 (edge, dtR, dtS, newdtC, 1 of bit, dtQbar)
	      []
		i;
		Qbar ! 1 of bit;
		RSFlipFlop_BB_Aux [R, S, C, Q, Qbar]
		 (edge, dtR, dtS, newdtC, dtQ, 1 of bit)
	      )
	  []
	    [(dtR eq 0 of bit) and (dtS eq 0 of bit) and((dtQ eq 1) 
	     or (dtQbar eq 1))] -> 
	    (
	      (
		Q ? newdtQ: Bit [newdtQ ne X];
		exit (newdtQ, any bit)
	      |||
		Qbar ? newdtQbar : Bit [newdtQbar ne X];
		exit (any bit, newdtQbar)
	      )
	    >>
	      accept newdtQ, newdtQbar : bit in
		RSFlipFlop_BB_Aux [R, S, C, Q, Qbar]
		  (edge, dtR, dtS, newdtC, newdtQ, newdtQbar)    
	    )       
	  []				(* R=1 S=1 *)
	    [(dtR eq 1) and (dtS eq 1) and (dtQ eq 0 of bit) 
	     and (dtQbar eq 0 of bit)] ->
	      RSFlipFlop_BB_Aux [R, S, C, Q, Qbar]
	       (edge, dtR, dtS, newdtC, dtQ, dtQbar)
	  []
	    [(dtR eq 1) and (dtS eq 1) and (dtQ ne 0 of bit) 
	     and (dtQbar ne 0 of bit)] ->
	      (
	        (
		  Q !  0 of bit;
		  exit
		|||
		  Qbar ! 0 of bit;
		  exit
		)
	      >>
	        RSFlipFlop_BB_Aux [R, S, C, Q, Qbar]
		 (edge, dtR, dtS, newdtC, 0 of bit, 0 of bit)
	      )
	  []
	    [(dtR eq 1) and (dtS eq 1) and (dtQ ne 0 of bit) 
	     and (dtQbar eq 0 of bit)] ->
	      Q ! 0 of bit; 
	      RSFlipFlop_BB_Aux [R, S, C, Q, Qbar]
	       (edge, dtR, dtS, newdtC, 0 of bit, 0 of bit)
	  []
	    [(dtR eq 1) and (dtS eq 1) and (dtQ eq 0 of bit) 
	     and (dtQbar ne 0 of bit)] ->
	      Qbar ! 0 of bit; 
	      RSFlipFlop_BB_Aux [R, S, C, Q, Qbar]
	       (edge, dtR, dtS, newdtC, 0 of bit, 0 of bit)
	  )
	[] (* inappropiate clock transition *)
	  [(dtC eq edge) or (dtC eq X of bit) or (newdtC ne edge) ] ->
	    RSFlipFlop_BB_Aux [R, S, C, Q, Qbar]
	     (edge, dtR, dtS, newdtC, dtQ, dtQbar)
      )
    endproc(* RSFlipFlop_BB_Aux *)
 endproc (* RSFlipFlop_BB *)
')')

# "DFlipFlopLockOut_BB_Decl" defines a master-slave D FlipFlop with input data lockout.
# "DFlipFlopLockOut_BB_Pos_Decl" the outputs data appear at the positive edge of clock
# "DFlipFlopLockOut_BB_Neg_Decl" the outputs data appear at the negtive edge of clock

define(DFlipFlopLockOut_BB_Pos_Decl, `declare(`$0', `DFlipFlopLockOut_BB_Decl
  process DFlipFlopLockOut_BB_Pos [D, C, Q, Qbar] : noexit :=
    DFlipFlopLockOut_BB [D, C, Q, Qbar] (1)
  endproc (* DFlipFlopLockOut_BB_Pos *)
')')

define(DFlipFlopLockOut_BB_Neg_Decl, `declare(`$0', `DFlipFlopLockOut_BB_Decl
  process DFlipFlopLockOut_BB_Neg [D, C, Q, Qbar] : noexit :=
    DFlipFlopLockOut_BB [D, C, Q, Qbar] (0 of bit)
  endproc (* DFlipFlopLockOut_BB_Neg *)
')')

define(DFlipFlopLockOut_BB_Decl, `declare(`$0', `
  process DFlipFlopLockOut_BB [D, C, Q, Qbar] (edge: Bit) : noexit :=
   DFlipFlopLockOut_BB_Aux [D, C, Q, Qbar]
    (edge, X of bit, X of bit, X of bit, X of bit, X of bit)

    where
    
    process DFlipFlopLockOut_BB_Aux [D, C, Q, Qbar]
     (edge, dtD, saveD, dtC, dtQ, dtQbar:Bit) : noexit := 
      D ? newdtD: Bit;
      DFlipFlopLockOut_BB_Aux [D, C, Q, Qbar]
       (edge, newdtD, saveD, dtC, dtQ, dtQbar)
    []
      C ? newdtC: Bit;
      (
	[(dtC eq not (edge)) and (newdtC eq edge)] -> (*output clock transition *)
	  (
	    [(saveD eq X of bit) and (dtQ eq X)] ->
	      (
	        Q ? newdtQ : bit [newdtQ ne X];
		Qbar ! not (newdtQ);
	        DFlipFlopLockOut_BB_Aux [D, C, Q, Qbar]
		 (edge, dtD, newdtQ, newdtC, newdtQ, not (newdtQ))
	      []
	        Qbar ? newdtQbar : Bit [newdtQbar ne X];
	        Q ! not (newdtQbar);
	        DFlipFlopLockOut_BB_Aux [D, C, Q, Qbar]
		 (edge, dtD, not (newdtQbar), newdtC, not (newdtQbar), newdtQbar)
	      )       
	  []
	    [(saveD eq X of bit) and (dtQ ne X)] ->
	      DFlipFlopLockOut_BB_Aux [D, C, Q, Qbar]
	       (edge, dtD, saveD, newdtC, dtQ, dtQbar)  
	  [] 
	    [(saveD eq 0 of bit) and (dtQ ne 0 of bit)] ->         
	      (
	        (
		  Q ! 0 of bit;
		  exit
		|||
		  Qbar ! 1 of bit;
		  exit
		)
	      >>
	        DFlipFlopLockOut_BB_Aux [D, C, Q, Qbar]
	         (edge, dtD, saveD, newdtC, 0 of bit, 1 of bit)
	      )
	  [] 
	    [(saveD eq 1) and (dtQ ne 1)] ->
	      (
	        (
		  Q ! 1 of bit;
		  exit
		|||
		  Qbar ! 0 of bit;
		  exit
		)
	      >>
	        DFlipFlopLockOut_BB_Aux [D, C, Q, Qbar]
	         (edge, dtD, saveD, newdtC, 1 of bit, 0 of bit)
	      )
	  )			(* end of output clock transition *)
      []
	[(dtC eq edge) and (newdtC eq not (edge))] -> 
	  DFlipFlopLockOut_BB_Aux [D, C, Q, Qbar]
	   (edge, dtD, dtD, newdtC, dtQ, dtQbar)
      []
        [(dtC eq X of bit) or (dtC eq newdtC)] -> 
	  DFlipFlopLockOut_BB_Aux [D, C, Q, Qbar]
	   (edge, dtD, saveD, newdtC, dtQ, dtQbar)
      )				(* end of C? --- *)
    endproc (* DFlipFlopLockOut_BB_Aux *)
  endproc (* DFlipFlopLockOut_BB *)
')')

# "JKFlipFlopLockOut_BB_Decl" defines a master-slave JK flip-flop, with input
# data lock-out.

define(JKFlipFlopLockOut_BB_Pos_Decl, `declare(`$0', `JKFlipFlopLockOut_BB_Decl
 process JKFlipFlopLockOut_BB_Pos [J, K, C, Q, Qbar] : noexit :=
    JKFlipFlopLockOut_BB [J, K, C, Q, Qbar] (1)
  endproc (* JKFlipFlopLockOut_BB_Pos *)
')')

define(JKFlipFlopLockOut_BB_Neg_Decl, `declare(`$0', `JKFlipFlopLockOut_BB_Decl
  process JKFlipFlopLockOut_BB_Neg [J, K, C, Q, Qbar] : noexit :=
    JKFlipFlopLockOut_BB [J, K, C, Q, Qbar] (0 of bit)
  endproc (* JKFlipFlopLockOut_BB_Neg *)
')')

define(JKFlipFlopLockOut_BB_Decl, `declare(`$0', `
  process JKFlipFlopLockOut_BB [J, K, C, Q, Qbar] (edge: Bit) : noexit :=
    JKFlipFlopLockOut_BB_Aux [J, K, C, Q, Qbar]
     (edge, X of bit, X of bit, X of bit, X of bit, X of bit, X of bit, X of bit)

    where
    
    process JKFlipFlopLockOut_BB_Aux [J, K, C, Q, Qbar]
     (edge, dtJ, dtK, saveJ, saveK, dtC, dtQ, dtQbar:Bit) : noexit:= 
      J ? newdtJ: Bit;
      JKFlipFlopLockOut_BB_Aux [J, K, C, Q, Qbar]
       (edge, newdtJ, dtK, saveJ, saveK, dtC, dtQ, dtQbar)
    []
      K ? newdtK: Bit;
      JKFlipFlopLockOut_BB_Aux [J, K, C, Q, Qbar]
       (edge, dtJ, newdtK, saveJ, saveK, dtC, dtQ, dtQbar)
    []
      C ? newdtC: Bit;
      (
	[(dtC eq not (edge)) and (newdtC eq edge)] -> (* output clock transition *)
	  (
	    [(saveJ eq X of bit) or (saveK eq X of bit) and (dtQ eq X of bit)] ->
	    					(* no change on output *)
	      (
	        (
		  Q ? newdtQ: Bit [newdtQ ne X];
		  exit (newdtQ, any bit)
		|||
		  Qbar ? newdtQbar : Bit [newdtQbar ne X];
		  exit (any bit, newdtQbar)
	        )
	      >>
	        accept newdtQ, newdtQbar : bit in
	          JKFlipFlopLockOut_BB_Aux [J, K, C, Q, Qbar] 
		   (edge, dtJ, dtK, newdtQ, newdtQbar, newdtC, newdtQ, newdtQbar)     
	      )
	    []
	      [(saveJ eq X of bit) or (saveK eq X of bit) and (dtQ ne X of bit)] ->
	      					(* no change on output *)
	        JKFlipFlopLockOut_BB_Aux [J, K, C, Q, Qbar]
		 (edge, dtJ, dtK, saveJ, saveK, newdtC, dtQ, dtQbar)               
	    []					(* saveJ=1, saveK=0 *)
	      [(saveJ eq 1) and (saveK eq 0 of bit) and (dtQ ne 1) 
	       and (dtQbar ne 0 of bit)] ->         
		(
		  (
		    Q ! 1 of bit;
		    exit
		  |||
		    Qbar ! 0 of bit;
		    exit
		  )
		>>
		  JKFlipFlopLockOut_BB_Aux [J, K, C, Q, Qbar]
		   (edge, dtJ, dtK, saveJ, saveK, newdtC, 1 of bit, 0 of bit)
		)
	    []
	      [(saveJ eq 1) and (saveK eq 0 of bit) and (dtQ eq 1) 
	       and (dtQbar eq 0 of bit)] ->
	        JKFlipFlopLockOut_BB_Aux [J, K, C, Q, Qbar]
	         (edge, dtJ, dtK, saveJ, saveK, newdtC, dtQ, dtQbar)
	    []
	      [(saveJ eq 1) and (saveK eq 0 of bit) and (dtQ ne 1)
	       and (dtQbar eq 0 of bit)] ->
		Q ! 1 of bit;
		JKFlipFlopLockOut_BB_Aux [J, K, C, Q, Qbar]
		 (edge, dtJ, dtK, saveJ, saveK, newdtC, 1 of bit, dtQbar)
	    []
	      [(saveJ eq 1) and (saveK eq 0 of bit) and (dtQ eq 1) 
	       and (dtQbar ne 0 of bit)] ->
		Qbar ! 0 of bit;
		JKFlipFlopLockOut_BB_Aux [J, K, C, Q, Qbar]
		 (edge, dtJ, dtK, saveJ, saveK, newdtC, dtQ, 0 of bit)
	    [] (* saveJ=0 saveK=1 *)
	      [(saveJ eq 0 of bit) and (saveK eq 1) and (dtQ ne 0 of bit)
	        and (dtQbar ne 1 of bit)] ->
		(
		  (
		    Q ! 0 of bit;
		    exit
		  |||
		    Qbar ! 1 of bit;
		    exit
		  )
		>>
		  JKFlipFlopLockOut_BB_Aux [J, K, C, Q, Qbar]
		   (edge, dtJ, dtK, saveJ, saveK, newdtC, 0 of bit, 1 of bit)
		)
	    []
	      [(saveJ eq 0 of bit) and (saveK eq 1) and (dtQ eq 0 of bit)
	       and (dtQbar eq 1)] ->
	        JKFlipFlopLockOut_BB_Aux [J, K, C, Q, Qbar]
	         (edge, dtJ, dtK, saveJ, saveK, newdtC, dtQ, dtQbar)
	    []
	      [(saveJ eq 0 of bit) and (saveK eq 1) and (dtQ ne 0 of bit)
	       and (dtQbar eq 1)] ->
		Q ! 0 of bit;
		JKFlipFlopLockOut_BB_Aux [J, K, C, Q, Qbar]
		 (edge, dtJ, dtK, saveJ, saveK, newdtC, 0 of bit, dtQbar)
	    []
	      [(saveJ eq 0 of bit) and (saveK eq 1) and (dtQ eq 0 of bit)
	       and (dtQbar ne 1)] ->
		Qbar ! 1 of bit;
		JKFlipFlopLockOut_BB_Aux [J, K, C, Q, Qbar]
		 (edge, dtJ, dtK, saveJ, saveK, newdtC, dtQ, 1 of bit)
	    []				(* saveJ=0 saveK=0 *)
	      [(saveJ eq 0 of bit) and (saveK eq 0 of bit) and (dtQ ne X)] ->
	        JKFlipFlopLockOut_BB_Aux [J, K, C, Q, Qbar]
	         (edge, dtJ, dtK, saveJ, saveK, newdtC, dtQ, dtQbar)
	    []
	      [(saveJ eq 0 of bit) and (saveK eq 0 of bit) and 
	       (dtQ eq X of bit)] ->
		(
		  (
		    Q ? newdtQ: Bit [newdtQ ne X];
		    exit (newdtQ, any bit)
		  |||
		    Qbar ? newdtQbar : Bit [newdtQbar ne X];
		    exit (any bit, newdtQbar)
		  )
		>>
		  accept newdtQ, newdtQbar : bit in
		    JKFlipFlopLockOut_BB_Aux [J, K, C, Q, Qbar] 
		     (edge, dtJ, dtK, saveJ, saveK, newdtC, newdtQ, newdtQbar)     
		)
	    []				(* saveJ=1 saveK=1 *)
	      [(saveJ eq 1) and (saveK eq 1) and (dtQ eq X of bit) ] ->
	        (
		  (
		    Q ? newdtQ: Bit [newdtQ ne X];
		    exit (newdtQ, any bit)
		  |||
		    Qbar ? newdtQbar : Bit [newdtQbar ne X];
		    exit (any bit, newdtQbar)
		  )
		>>
		  accept newdtQ, newdtQbar : bit in
		    JKFlipFlopLockOut_BB_Aux [J, K, C, Q, Qbar] 
		     (edge, dtJ, dtK, saveJ, saveK, newdtC, newdtQ, newdtQbar)     
	        )
	    []
	      [(saveJ eq 1) and (saveK eq 1) and ((dtQ xor dtQbar) eq 1)] ->
	        (
		  (
		    Q ! not (dtQ);
		    exit (not (dtQ), any bit) 
		  |||
		    Qbar ! dtQ;
		    exit (any bit, dtQ)
		  )
		>>
		  accept newdtQ, newdtQbar :Bit in
		    JKFlipFlopLockOut_BB_Aux [J, K, C, Q, Qbar]
		     (edge, dtJ, dtK, saveJ, saveK, newdtC, newdtQ, newdtQbar)
	        )
	    []
	      [(saveJ eq 1) and (saveK eq 1) and ((dtQ xor dtQbar) eq 0 of bit)] ->
					(* dtQ=dtQbar=1; or; dtQ=dtQbar=0 *)
	        (
		  i;
		  Q ! not (dtQ);
		  JKFlipFlopLockOut_BB_Aux [J, K, C, Q, Qbar]
		   (edge, dtJ, dtK, saveJ, saveK, newdtC, not (dtQ), dtQbar)
		[] 
		  i;
		  Qbar ! not (dtQbar);
		  JKFlipFlopLockOut_BB_Aux [J, K, C, Q, Qbar]
		   (edge, dtJ, dtK, saveJ, saveK, newdtC, dtQ, not (dtQbar))
	        )
	  )				(* end of output clock transition *)
	[]
	  [(dtC eq edge) and (newdtC eq not (edge))] -> (* data input clock transition *)
	    JKFlipFlopLockOut_BB_Aux [J, K, C, Q, Qbar]
	     (edge, dtJ, dtK, dtJ, dtK, newdtC, dtQ, dtQbar)
	[]
	  [(dtC eq X of bit) or (dtC eq newdtC)] -> 
	    JKFlipFlopLockOut_BB_Aux [J, K, C, Q, Qbar]
	     (edge, dtJ, dtK, saveJ, saveK, newdtC, dtQ, dtQbar)
      )				(* end of C? --- *)
    endproc (* JKFlipFlopLockOut_BB_Aux *)
  endproc (* JKFlipFlopLockOut_BB *)
')')

# "RSFlipFlop_BB_PreClr_Decl" defines a master slave RS flip-flop with pre-set
# and pre-clear inputs; pre-set and pre-clear are 0 active.

define(RSFlipFlop_BB_PreClr_Pos_Decl, `declare(`$0', `RSFlipFlop_BB_PreClr_Decl
  process RSFlipFlop_BB_PreClr_Pos [R, S, Preset, Clear, Ck, Q, Qbar] : noexit :=
    RSFlipFlop_BB_PreClr [R, S, Preset, Clear, Ck, Q, Qbar] (1)
  endproc (* RSFlipFlop_BB_PreClr_Pos *)
')')

define(RSFlipFlop_BB_PreClr_Neg_Decl, `declare(`$0', `RSFlipFlop_BB_PreClr_Decl
  process RSFlipFlop_BB_PreClr_Neg [R, S, Preset, Clear, Ck, Q, Qbar] : noexit :=
    RSFlipFlop_BB_PreClr [R, S, Preset, Clear, Ck, Q, Qbar] (0 of bit)
  endproc (* RSFlipFlop_BB_PreClr_Neg *)
')')

define(RSFlipFlop_BB_PreClr_Decl, `
  process RSFlipFlop_BB_PreClr [R, S, Preset, Clear, Ck, Q, Qbar]
   (edge: Bit) :noexit :=
    RSFlipFlop_BB_Aux_PreClr [R, S, Preset, Clear, Ck, Q, Qbar]
     (edge, X of bit, X of bit, X of bit, X of bit, X of bit, X of bit, X of bit)

    where
    
    process RSFlipFlop_BB_Aux_PreClr [R, S, Preset, Clear, Ck, Q, Qbar]
     (edge, dtR, dtS, dtPreset, dtClear, dtCk, dtQ, dtQbar: Bit) : noexit :=
      R ? newdtR: Bit;
      RSFlipFlop_BB_Aux_PreClr [R, S, Preset, Clear, Ck, Q, Qbar]
       (edge, newdtR, dtS, dtPreset, dtClear, dtCk, dtQ, dtQbar)
    []
      S ? newdtS: Bit;
      RSFlipFlop_BB_Aux_PreClr [R, S, Preset, Clear, Ck, Q, Qbar]
       (edge, dtR, newdtS, dtPreset, dtClear, dtCk, dtQ, dtQbar)
    []
      Preset ? newdtPreset: Bit;
      RSFlipFlop_BB_Aux_PreClr [R, S, Preset, Clear, Ck, Q, Qbar]
       (edge, dtR, dtS, newdtPreset, dtClear, dtCk, dtQ, dtQbar)
    []
      Clear ? newdtClear: Bit;
      RSFlipFlop_BB_Aux_PreClr [R, S, Preset, Clear, Ck, Q, Qbar]
       (edge, dtR, dtS, dtPreset, newdtClear, dtCk, dtQ, dtQbar)
    []
      Ck ? newdtCk: Bit;
      (
        [(dtCk eq not (edge)) and (newdtCk eq edge)] -> (* appropriate clock transition *)
	  (
	    [(dtPreset eq 1) and (dtClear eq 1)] ->  (* preset and clear are inactive *)
	      (
		[((dtR eq X of bit) or (dtS eq X of bit)) and (dtQ ne X)] ->
			(* inputs undecided, outputs remain unchange *)
		  RSFlipFlop_BB_Aux_PreClr [R, S, Preset, Clear, Ck, Q, Qbar]
		  (edge, dtR, dtS, dtPreset, dtClear, newdtCk, dtQ, dtQbar)
	      []
		[((dtR eq X) or (dtS eq X)) and (dtQ eq X)] ->
		  (
		    (
		      Q ? newdtQ: Bit [newdtQ ne X];
		      exit (newdtQ, any bit)
		    |||
		      Qbar ? newdtQbar : Bit [newdtQbar ne X];
		      exit (any bit, newdtQbar)
		    )
		  >>
		    accept newdtQ, newdtQbar : bit in
		      RSFlipFlop_BB_Aux_PreClr [R, S, Preset, Clear, Ck, Q, Qbar] 
		      (edge, dtR, dtS, dtPreset, dtClear, newdtCk, newdtQ, newdtQbar)    
		  )        
	      []					(* reset *)
		[(dtR eq 1) and (dtS eq 0 of bit) and (dtQ ne 0 of bit) 
		 and (dtQbar ne 1)] ->     
		  (
		    (
		      Q ! 0 of bit;
		      exit
		    |||
		      Qbar ! 1 of bit;
		      exit
		    ) 
		  >>
		    RSFlipFlop_BB_Aux_PreClr [R, S, Preset, Clear, Ck, Q, Qbar] 
		     (edge, dtR, dtS, dtPreset, dtClear, newdtCk, 0 of bit, 1 of bit)
		  )
	      []      
		[(dtR eq 1) and (dtS eq 0 of bit) and (dtQ eq 0 of bit) 
		 and (dtQbar eq 1)] -> 
		  RSFlipFlop_BB_Aux_PreClr [R, S, Preset, Clear, Ck, Q, Qbar]
		   (edge, dtR, dtS, dtPreset, dtClear, newdtCk, dtQ, dtQbar)
	      []
		[(dtR eq 1) and (dtS eq 0 of bit) and (dtQ ne 0 of bit) 
		 and (dtQbar eq 1)] ->
		  Q ! 0 of bit; 
		  RSFlipFlop_BB_Aux_PreClr [R, S, Preset, Clear, Ck, Q, Qbar]
		   (edge, dtR, dtS, dtPreset, dtClear, newdtCk, 0 of bit, dtQbar)
	      []
		[(dtR eq 1) and (dtS eq 0 of bit) and (dtQ eq 0 of bit) 
		 and (dtQbar ne 1)] ->
		  Qbar ! 1 of bit; 
		  RSFlipFlop_BB_Aux_PreClr [R, S, Preset, Clear, Ck, Q, Qbar]
		   (edge, dtR, dtS, dtPreset, dtClear, newdtCk, dtQ, 1 of bit)
	      []					(* set *)
		[(dtR eq 0 of bit) and (dtS eq 1) and (dtQ ne 1) 
		 and (dtQbar ne 0 of bit)] ->     
		  (
		    (
		      Q ! 1 of bit;
		      exit
		    |||
		      Qbar ! 0 of bit;
		      exit
		    ) 
		  >>
		    RSFlipFlop_BB_Aux_PreClr [R, S, Preset, Clear, Ck, Q, Qbar] 
		     (edge, dtR, dtS, dtPreset, dtClear, newdtCk, 1 of bit, 0 of bit))
	      []
		[(dtR eq 0 of bit) and (dtS eq 1) and (dtQ eq 1) 
		 and (dtQbar eq 0 of bit)] ->
		  RSFlipFlop_BB_Aux_PreClr [R, S, Preset, Clear, Ck, Q, Qbar]
		   (edge, dtR, dtS, dtPreset, dtClear, newdtCk, dtQ, dtQbar)
	      []
		[(dtR eq 0 of bit) and (dtS eq 1) and (dtQ ne 1) 
		 and (dtQbar eq 0 of bit)] ->
		  Q ! 1 of bit;
		  RSFlipFlop_BB_Aux_PreClr [R, S, Preset, Clear, Ck, Q, Qbar]
		   (edge, dtR, dtS, dtPreset, dtClear, newdtCk, 1 of bit, dtQbar)
	      []
		[(dtR eq 0 of bit) and (dtS eq 1) and (dtQ eq 1) 
		 and (dtQbar ne 0 of bit)] ->
		  Qbar ! 0 of bit;
		  RSFlipFlop_BB_Aux_PreClr [R, S, Preset, Clear, Ck, Q, Qbar]
		   (edge, dtR, dtS, dtPreset, dtClear, newdtCk, dtQ, 0 of bit)
	      []				(* R=0, S=0 *)
		[(dtR eq 0 of bit) and (dtS eq 0 of bit) and 
		 (dtQ eq 0 of bit) and (dtQbar eq 0 of bit) ] ->
		 		(* the previous state is nonstable *)
		  (
		    i;
		    Q ! 1 of bit;
		    RSFlipFlop_BB_Aux_PreClr [R, S, Preset, Clear, Ck, Q, Qbar]
		     (edge, dtR, dtS, dtPreset, dtClear, newdtCk, 1 of bit, dtQbar)
		  []
		    i;
		    Qbar ! 1 of bit;
		    RSFlipFlop_BB_Aux_PreClr [R, S, Preset, Clear, Ck, Q, Qbar]
		      (edge, dtR, dtS, dtPreset, dtClear, newdtCk, dtQ, 1 of bit)
		  )
		[]
		  [(dtR eq 0 of bit) and (dtS eq 0 of bit) and (dtQ eq X)] ->
		    (
		      (
		        Q ? newdtQ: Bit [newdtQ ne X];
			exit (newdtQ, any bit)
		      |||
		        Qbar ? newdtQbar : Bit [newdtQbar ne X];
			exit (any bit, newdtQbar)
		      )
		    >>
		      accept newdtQ, newdtQbar : bit in
		        RSFlipFlop_BB_Aux_PreClr [R, S, Preset, Clear, Ck, Q, Qbar] 
		         (edge, dtR, dtS, dtPreset, dtClear, newdtCk, newdtQ, newdtQbar)    
		    )     
		[]
		  [(dtR eq 0 of bit) and (dtS eq 0 of bit) and (dtQ ne X)
		   and ((dtQ eq 1) or (dtQbar eq 1))] -> 
		    RSFlipFlop_BB_Aux_PreClr [R, S, Preset, Clear, Ck, Q, Qbar]
		     (edge, dtR, dtS, dtPreset, dtClear, newdtCk, dtQ, dtQbar)
		[]					(* R=1 S=1 *)
		  [(dtR eq 1) and (dtS eq 1) and (dtQ eq 0 of bit) 
		   and (dtQbar eq 0 of bit)] ->
		    RSFlipFlop_BB_Aux_PreClr [R, S, Preset, Clear, Ck, Q, Qbar]
		     (edge, dtR, dtS, dtPreset, dtClear, newdtCk, dtQ, dtQbar)
		[]
		  [(dtR eq 1) and (dtS eq 1) and (dtQ ne 0 of bit) 
		   and (dtQbar ne 0 of bit)] ->
		    (
		      (
		        Q !  0 of bit;
			exit
		      |||
		        Qbar ! 0 of bit;
			exit
		      )
		    >>
		      RSFlipFlop_BB_Aux_PreClr [R, S, Preset, Clear, Ck, Q, Qbar]
		       (edge, dtR, dtS, dtPreset, dtClear, newdtCk, 0 of bit, 0 of bit)
		    )
		[]
		  [(dtR eq 1) and (dtS eq 1) and (dtQ ne 0 of bit) 
		   and (dtQbar eq 0 of bit)] ->
		    Q ! 0 of bit; 
		    RSFlipFlop_BB_Aux_PreClr [R, S, Preset, Clear, Ck, Q, Qbar]
		     (edge, dtR, dtS, dtPreset, dtClear, newdtCk, 0 of bit, 0 of bit)
		[]
		  [(dtR eq 1) and (dtS eq 1) and (dtQ eq 0 of bit) 
		   and (dtQbar ne 0 of bit)] ->
		    Qbar ! 0 of bit; 
		    RSFlipFlop_BB_Aux_PreClr [R, S, Preset, Clear, Ck, Q, Qbar]
		     (edge, dtR, dtS, dtPreset, dtClear, newdtCk, 0 of bit, 0 of bit)
	      )				(* end of preset and clear inactive *)
	  []
	    [((dtPreset ne 1) or (dtClear ne 1)) and (dtQ ne X)] -> (* no change *)
	      RSFlipFlop_BB_Aux_PreClr [R, S, Preset, Clear, Ck, Q, Qbar]
	       (edge, dtR, dtS, dtPreset, dtClear, newdtCk, dtQ, dtQbar)
	  []
	    [((dtPreset ne 1) or (dtClear ne 1)) and (dtQ eq X)] ->
	      (
	        (
		  Q ? newdtQ: Bit [newdtQ ne X];
		  exit (newdtQ, any bit)
		|||
		  Qbar ? newdtQbar : Bit [newdtQbar ne X];
		  exit (any bit, newdtQbar)
	        )
	      >>
	        accept newdtQ, newdtQbar : bit in
	          RSFlipFlop_BB_Aux_PreClr [R, S, Preset, Clear, Ck, Q, Qbar] 
	           (edge, dtR, dtS, dtPreset, dtClear, newdtCk, newdtQ, newdtQbar)    
	      )      
	  )			(* end of appropriate clock transition *)
      []
        [(dtCk eq edge) or (newdtCk ne edge) or (dtCk eq X of bit)] -> 
	  RSFlipFlop_BB_Aux_PreClr [R, S, Preset, Clear, Ck, Q, Qbar]
	   (edge, dtR, dtS, dtPreset, dtClear, newdtCk, dtQ, dtQbar)
      )						(* end of clock transition *)
    []						(* preset *)
      [(dtPreset eq 0 of bit) and (dtClear eq 1)and (dtQ ne 1) 
       and (dtQbar ne 0 of bit)] ->          
        (
	  (
	    Q ! 1 of bit;
	    exit
	  |||
	    Qbar ! 0 of bit;
	    exit
	  ) 
	>>
	  RSFlipFlop_BB_Aux_PreClr [R, S, Preset, Clear, Ck, Q, Qbar] 
	   (edge, dtR, dtS, dtPreset, dtClear, dtCk, 1 of bit, 0 of bit)
        ) 
    []
      [(dtPreset eq 0 of bit) and (dtClear eq 1) and (dtQ ne 1) 
       and (dtQbar eq 0 of bit)] ->
        Q ! 1 of bit;
        RSFlipFlop_BB_Aux_PreClr [R, S, Preset, Clear, Ck, Q, Qbar]
         (edge, dtR, dtS, dtPreset, dtClear, dtCk, 1 of bit, dtQbar)
    []
      [(dtPreset eq 0 of bit) and (dtClear eq 1)and (dtQ eq 1) 
       and (dtQbar ne 0 of bit)] ->
	Qbar ! 0 of bit;
	RSFlipFlop_BB_Aux_PreClr [R, S, Preset, Clear, Ck, Q, Qbar]
	 (edge, dtR, dtS, dtPreset, dtClear, dtCk, dtQ, 0 of bit)
    []					(* clear *)
      [(dtPreset eq 1) and (dtClear eq 0 of bit) and (dtQ ne 0 of bit)
       and (dtQbar ne 1)] ->      
	(
	  (
	    Q ! 0 of bit;
	    exit
	  |||
	    Qbar ! 1 of bit;
	    exit
	  ) 
	>>
	  RSFlipFlop_BB_Aux_PreClr [R, S, Preset, Clear, Ck, Q, Qbar] 
	   (edge, dtR, dtS, dtPreset, dtClear, dtCk, 0 of bit, 1 of bit)
	)
    []
      [(dtPreset eq 1) and (dtClear eq 0 of bit)and (dtQ ne 0 of bit) 
       and (dtQbar eq 1)] ->
	Q ! 0 of bit; 
	RSFlipFlop_BB_Aux_PreClr [R, S, Preset, Clear, Ck, Q, Qbar]
	 (edge, dtR, dtS, dtPreset, dtClear, dtCk, 0 of bit, dtQbar)
    []
      [(dtPreset eq 1) and (dtClear eq 0 of bit)and (dtQ eq 0 of bit) 
       and (dtQbar ne 1)] ->
	Qbar ! 1 of bit; 
	RSFlipFlop_BB_Aux_PreClr [R, S, Preset, Clear, Ck, Q, Qbar]
	 (edge, dtR, dtS, dtPreset, dtClear, dtCk, dtQ, 1 of bit)
    [] 
      [(dtPreset eq 0 of bit) and (dtClear eq 0 of bit) and (dtQ ne 1)
       and (dtQbar ne 1)] ->
        (
	  (
	    Q ! 1 of bit;
	    exit
	  |||
	    Qbar ! 1 of bit;
	    exit
	  )
	>>
	  RSFlipFlop_BB_Aux_PreClr [R, S, Preset, Clear, Ck, Q, Qbar] 
	   (edge, dtR, dtS, dtPreset, dtClear, dtCk, 1 of bit, 1 of bit)
        )
    []
      [(dtPreset eq 0 of bit) and (dtClear eq 0 of bit) and (dtQ eq 1) 
       and (dtQbar ne 1)] -> 
	Qbar ! 1 of bit;
	RSFlipFlop_BB_Aux_PreClr [R, S, Preset, Clear, Ck, Q, Qbar]
	 (edge, dtR, dtS, dtPreset, dtClear, dtCk, dtQ, 1 of bit)
    []
      [(dtPreset eq 0 of bit) and (dtClear eq 0 of bit) and (dtQ ne 1)
       and (dtQbar eq 1)] -> 
	Q ! 1 of bit;
	RSFlipFlop_BB_Aux_PreClr [R, S, Preset, Clear, Ck, Q, Qbar]
	 (edge, dtR, dtS, dtPreset, dtClear, dtCk, 1 of bit, dtQbar)
    []	(* preset and clear return to inactive state from unstable state *)
      [(dtPreset eq 1) and (dtClear eq 1) and (dtQ eq 1) and (dtQbar eq 1)] -> 
	(
	  i;
	  Q ! 0 of bit;
	  RSFlipFlop_BB_Aux_PreClr [R, S, Preset, Clear, Ck, Q, Qbar]
	  (edge, dtR, dtS, dtPreset, dtClear, dtCk, 0 of bit, dtQbar)
	[]
	  i;
	  Qbar ! 0 of bit;
	  RSFlipFlop_BB_Aux_PreClr [R, S, Preset, Clear, Ck, Q, Qbar]
	   (edge, dtR, dtS, dtPreset, dtClear, dtCk, dtQ, 0 of bit)
	)
    endproc (* RSFlipFlop_BB_Aux_PreClr *)
  endproc (* RSFlipFlop_BB_PreClr *)
')')

# "JKFlipFlop_BB_PreClr_Decl" defines a master slave JK flip-flop with pre-set
# and pre-clear inputs; pre-set and pre-clear are 0 active.

define(JKFlipFlop_BB_PreClr_Pos_Decl, `declare(`$0', `JKFlipFlop_BB_PreClr_Decl
  process JKFlipFlop_BB_PreClr_Pos [J, K, Preset, Clear, Ck, Q, Qbar] : noexit :=
    JKFlipFlop_BB_PreClr [J, K, Preset, Clear, Ck, Q, Qbar] (1)
  endproc (* JKFlipFlop_BB_PreClr_Pos *)
')')

define(JKFlipFlop_BB_PreClr_Neg_Decl, `declare(`$0', `JKFlipFlop_BB_PreClr_Decl
  process JKFlipFlop_BB_PreClr_Neg [J, K, Preset, Clear, Ck, Q, Qbar] : noexit :=
    JKFlipFlop_BB_PreClr [J, K, Preset, Clear, Ck, Q, Qbar] (0 of bit)
  endproc (* JKFlipFlop_BB_PreClr_Neg *)
')')

define(JKFlipFlop_BB_PreClr_Decl, `
  process JKFlipFlop_BB_PreClr [J, K, Preset, Clear, Ck, Q, Qbar] (edge: Bit) :noexit :=
    JKFlipFlop_BB_Aux_PreClr [J, K, Preset, Clear, Ck, Q, Qbar]
     (edge, X of bit, X of bit, X of bit, X of bit, X of bit, X of bit, X of bit)

    where
    
    process JKFlipFlop_BB_Aux_PreClr [J, K, Preset, Clear, Ck, Q, Qbar]
     (edge, dtJ, dtK, dtPreset, dtClear, dtCk, dtQ, dtQbar: Bit) : noexit :=
      J ? newdtJ: Bit;
      JKFlipFlop_BB_Aux_PreClr [J, K, Preset, Clear, Ck, Q, Qbar]
       (edge, newdtJ, dtK, dtPreset, dtClear, dtCk, dtQ, dtQbar)
    []
      K ? newdtK: Bit;
      JKFlipFlop_BB_Aux_PreClr [J, K, Preset, Clear, Ck, Q, Qbar]
       (edge, dtJ, newdtK, dtPreset, dtClear, dtCk, dtQ, dtQbar)
    []
      Preset ? newdtPreset: Bit;
      JKFlipFlop_BB_Aux_PreClr [J, K, Preset, Clear, Ck, Q, Qbar]
       (edge, dtJ, dtK, newdtPreset, dtClear, dtCk, dtQ, dtQbar)
    []
      Clear ? newdtClear: Bit;
      JKFlipFlop_BB_Aux_PreClr [J, K, Preset, Clear, Ck, Q, Qbar]
       (edge, dtJ, dtK, dtPreset, newdtClear, dtCk, dtQ, dtQbar)
    []
      Ck ? newdtCk: Bit;
      (
        [(dtCk eq not (edge)) and (newdtCk eq edge)] -> (* appropriate clock transition *)
	  (
	    [(dtPreset eq 1) and (dtClear eq 1)] -> (* preset and clear inactive *)
	      (
		[((dtJ eq X) or (dtK eq X)) and (dtQ ne X)] ->
				(* inputs undecided, outputs remain unchanged *)
		  JKFlipFlop_BB_Aux_PreClr [J, K, Preset, Clear, Ck, Q, Qbar]
		   (edge, dtJ, dtK, dtPreset, dtClear, newdtCk, dtQ, dtQbar)
	      []
		[((dtJ eq X) or (dtK eq X)) and (dtQ eq X)] ->
		  (
		    (
		      Q ? newdtQ: bit [newdtQ ne X];
		      exit (newdtQ, any bit)
		    |||
		      Qbar ? newdtQbar: Bit [newdtQbar ne X];
		      exit (any bit, newdtQbar)
		    )
		  >>
		    accept newdtQ, newdtQbar : Bit in 
		      JKFlipFlop_BB_Aux_PreClr [J, K, Preset, Clear, Ck, Q, Qbar]
		       (edge, dtJ, dtK, dtPreset, dtClear, newdtCk, newdtQ, newdtQbar)
		  )
	      []			(*J=0 K=1  *)
	        [(dtJ eq 0 of bit) and (dtK eq 1) and (dtQ ne 0 of bit) 
		 and (dtQbar ne 1)] ->     
		  (
		    (
		      Q ! 0 of bit;
		      exit
		    |||
		      Qbar ! 1 of bit;
		      exit
		    ) 
		  >>
		    JKFlipFlop_BB_Aux_PreClr [J, K, Preset, Clear, Ck, Q, Qbar] 
		     (edge, dtJ, dtK, dtPreset, dtClear, newdtCk, 0 of bit, 1 of bit)
		  )
	      []      
	        [(dtJ eq 0 of bit) and (dtK eq 1) and (dtQ eq 0 of bit)
		 and (dtQbar eq 1)] -> 
		  JKFlipFlop_BB_Aux_PreClr [J, K, Preset, Clear, Ck, Q, Qbar]
		   (edge, dtJ, dtK, dtPreset, dtClear, newdtCk, dtQ, dtQbar)
	      []
	        [(dtJ eq 0 of bit) and (dtK eq 1) and (dtQ ne 0 of bit) 
		 and (dtQbar eq 1)] ->
		  Q ! 0 of bit; 
		  JKFlipFlop_BB_Aux_PreClr [J, K, Preset, Clear, Ck, Q, Qbar]
		   (edge, dtJ, dtK, dtPreset, dtClear, newdtCk, 0 of bit, dtQbar)
	      []
	        [(dtJ eq 0 of bit) and (dtK eq 1) and (dtQ eq 0 of bit) 
		  and (dtQbar ne 1)] ->
		  Qbar ! 1 of bit; 
		  JKFlipFlop_BB_Aux_PreClr [J, K, Preset, Clear, Ck, Q, Qbar]
		   (edge, dtJ, dtK, dtPreset, dtClear, newdtCk, dtQ, 1 of bit)
	      []				(* J=1 K=0 *)
	        [(dtJ eq 1) and (dtK eq 0 of bit) and (dtQ ne 1) 
		 and (dtQbar ne 0 of bit)] ->     
		  (
		    (
		      Q ! 1 of bit;
		      exit
		    |||
		      Qbar ! 0 of bit;
		      exit
		    ) 
		  >>
		    JKFlipFlop_BB_Aux_PreClr [J, K, Preset, Clear, Ck, Q, Qbar] 
		     (edge, dtJ, dtK, dtPreset, dtClear, newdtCk, 1 of bit, 0 of bit))
	      []
	        [(dtJ eq 1) and (dtK eq 0 of bit) and (dtQ eq 1) 
		 and (dtQbar eq 0 of bit)] ->
	          JKFlipFlop_BB_Aux_PreClr [J, K, Preset, Clear, Ck, Q, Qbar]
	           (edge, dtJ, dtK, dtPreset, dtClear, newdtCk, dtQ, dtQbar)
	      []
	        [(dtJ eq 1) and (dtK eq 0 of bit) and (dtQ ne 1)
		 and (dtQbar eq 0 of bit)] ->
	          Q ! 1 of bit;
	          JKFlipFlop_BB_Aux_PreClr [J, K, Preset, Clear, Ck, Q, Qbar]
	           (edge, dtJ, dtK, dtPreset, dtClear, newdtCk, 1 of bit, dtQbar)
	      []
	        [(dtJ eq 1) and (dtK eq 0 of bit) and (dtQ eq 1) 
		 and (dtQbar ne 0 of bit)] ->
	          Qbar ! 0 of bit;
	          JKFlipFlop_BB_Aux_PreClr [J, K, Preset, Clear, Ck, Q, Qbar]
	           (edge, dtJ, dtK, dtPreset, dtClear, newdtCk, dtQ, 0 of bit)
	      []			(* J=0, K=0 *)
	        [(dtJ eq 0 of bit) and (dtK eq 0 of bit) and (dtQ ne X)] -> 
	          JKFlipFlop_BB_Aux_PreClr [J, K, Preset, Clear, Ck, Q, Qbar]
	           (edge, dtJ, dtK, dtPreset, dtClear, newdtCk, dtQ, dtQbar)
	      []
	        [(dtJ eq 0 of bit) and (dtK eq 0 of bit) and (dtQ eq X)] ->
	          (
		    (
		      Q ? newdtQ: bit [newdtQ ne X];
		      exit (newdtQ, any bit)
		    |||
		      Qbar ? newdtQbar: Bit [newdtQbar ne X];
		      exit (any bit, newdtQbar)
		    )
		  >>
		    accept newdtQ, newdtQbar : Bit in 
		      JKFlipFlop_BB_Aux_PreClr [J, K, Preset, Clear, Ck, Q, Qbar]
		       (edge, dtJ, dtK, dtPreset, dtClear, newdtCk, newdtQ, newdtQbar)
	          )    
	      []			(* J=1  K=1 *)
	        [(dtJ eq 1) and (dtK eq 1) and (dtQ eq X)] ->    
		  (
		    (
		      Q ? newdtQ: bit [newdtQ ne X];
		      exit (newdtQ, any bit)
		    |||
		      Qbar ? newdtQbar: Bit [newdtQbar ne X];
		      exit (any bit, newdtQbar)
		    )
		  >>
		    accept newdtQ, newdtQbar : Bit in 
		      JKFlipFlop_BB_Aux_PreClr [J, K, Preset, Clear, Ck, Q, Qbar]
		       (edge, dtJ, dtK, dtPreset, dtClear, newdtCk, newdtQ, newdtQbar)
		  )           
	      []
	        [(dtJ eq 1) and (dtK eq 1) and ((dtQ xor dtQbar) eq 1)] ->
		  (
		    (
		      Q ! not (dtQ);
		      exit (not (dtQ), any bit) 
		    |||
		      Qbar ! dtQ;
		      exit (any bit, dtQ)
		    )
		  >>
		    accept newdtQ, newdtQbar :Bit in
		      JKFlipFlop_BB_Aux_PreClr [J, K, Preset, Clear, Ck, Q, Qbar]
		       (edge, dtJ, dtK, dtPreset, dtClear, newdtCk, newdtQ, newdtQbar)
		  )      
	      []
	        [(dtJ eq 1) and (dtK eq 1) and ((dtQ xor dtQbar) eq 0 of bit)] ->
					(* dtQ=dtQbar=1; or; dtQ=dtQbar=0 *)
		  (
		    i;
		    Q ! not (dtQ);
		    JKFlipFlop_BB_Aux_PreClr [J, K, Preset, Clear, Ck, Q, Qbar]
		     (edge, dtJ, dtK, dtPreset, dtClear, newdtCk, not (dtQ), dtQbar)
		  [] 
		    i; Qbar ! not (dtQbar);
		    JKFlipFlop_BB_Aux_PreClr [J, K, Preset, Clear, Ck, Q, Qbar]
		     (edge, dtJ, dtK, dtPreset, dtClear, newdtCk, dtQ, not (dtQbar))
		  )
	      ) (* end of preset and clear inactive *)
	  []
	    [((dtPreset ne 1) or (dtClear ne 1)) and (dtQ ne X)] -> (* nochange *)
	      JKFlipFlop_BB_Aux_PreClr [J, K, Preset, Clear, Ck, Q, Qbar]
	       (edge, dtJ, dtK, dtPreset, dtClear, newdtCk, dtQ, dtQbar)
	  []
	    [((dtPreset ne 1) or (dtClear ne 1)) and (dtQ eq X)] ->
	      (
	        (
		  Q ? newdtQ: bit [newdtQ ne X];
		  exit (newdtQ, any bit)
		|||
		  Qbar ? newdtQbar: Bit [newdtQbar ne X];
		  exit (any bit, newdtQbar)
	        )
	      >>
	        accept newdtQ, newdtQbar : Bit in 
	          JKFlipFlop_BB_Aux_PreClr [J, K, Preset, Clear, Ck, Q, Qbar]
		   (edge, dtJ, dtK, dtPreset, dtClear, newdtCk, newdtQ, newdtQbar)
	      )
	  )				(* end of appropriate clock transition *)
      []
	[(dtCk eq edge) or (newdtCk ne edge) or (dtCk eq X)] -> 
          JKFlipFlop_BB_Aux_PreClr [J, K, Preset, Clear, Ck, Q, Qbar]
	   (edge, dtJ, dtK, dtPreset, dtClear, newdtCk, dtQ, dtQbar)
      )					(* end of clock transition *)
    []					(* preset *)
      [(dtPreset eq 0 of bit) and (dtClear eq 1) and (dtQ ne 1) 
       and (dtQbar ne 0 of bit)] ->          
        (
	  (
	    Q ! 1 of bit;
	    exit
	  |||
	    Qbar ! 0 of bit;
	    exit
	  ) 
	>>
	  JKFlipFlop_BB_Aux_PreClr [J, K, Preset, Clear, Ck, Q, Qbar] 
	   (edge, dtJ, dtK, dtPreset, dtClear, dtCk, 1 of bit, 0 of bit)
        ) 
    []
      [(dtPreset eq 0 of bit) and (dtClear eq 1) and (dtQ ne 1) 
       and (dtQbar eq 0 of bit)] ->
        Q ! 1 of bit;
        JKFlipFlop_BB_Aux_PreClr [J, K, Preset, Clear, Ck, Q, Qbar]
         (edge, dtJ, dtK, dtPreset, dtClear, dtCk, 1 of bit, dtQbar)
    []
      [(dtPreset eq 0 of bit) and (dtClear eq 1) and (dtQ eq 1) 
       and (dtQbar ne 0 of bit)] ->
        Qbar ! 0 of bit;
        JKFlipFlop_BB_Aux_PreClr [J, K, Preset, Clear, Ck, Q, Qbar]
         (edge, dtJ, dtK, dtPreset, dtClear, dtCk, dtQ, 0 of bit)
    []					(* clear *)
      [(dtPreset eq 1) and (dtClear eq 0 of bit) and (dtQ ne 0 of bit) 
       and (dtQbar ne 1)] ->      
        (
	  (
	    Q ! 0 of bit;
	    exit
	  |||
	    Qbar ! 1 of bit;
	    exit
	  )
	>>
	  JKFlipFlop_BB_Aux_PreClr [J, K, Preset, Clear, Ck, Q, Qbar] 
	   (edge, dtJ, dtK, dtPreset, dtClear, dtCk, 0 of bit, 1 of bit)
        )
      
    []
      [(dtPreset eq 1) and (dtClear eq 0 of bit)and (dtQ ne 0 of bit) 
       and (dtQbar eq 1)] ->
        Q ! 0 of bit; 
        JKFlipFlop_BB_Aux_PreClr [J, K, Preset, Clear, Ck, Q, Qbar]
         (edge, dtJ, dtK, dtPreset, dtClear, dtCk, 0 of bit, dtQbar)
    []
      [(dtPreset eq 1) and (dtClear eq 0 of bit)and (dtQ eq 0 of bit)
       and (dtQbar ne 1)] ->
        Qbar ! 1 of bit; 
        JKFlipFlop_BB_Aux_PreClr [J, K, Preset, Clear, Ck, Q, Qbar]
         (edge, dtJ, dtK, dtPreset, dtClear, dtCk, dtQ, 1 of bit)
    [] 
      [(dtPreset eq 0 of bit) and (dtClear eq 0 of bit) and (dtQ ne 1) 
       and (dtQbar ne 1)] ->
        (
	  (
	    Q ! 1 of bit;
	    exit
	  |||
	    Qbar ! 1 of bit;
	    exit
	  )
	>>
	  JKFlipFlop_BB_Aux_PreClr [J, K, Preset, Clear, Ck, Q, Qbar] 
	   (edge, dtJ, dtK, dtPreset, dtClear, dtCk, 1 of bit, 1 of bit)
        )
    []
      [(dtPreset eq 0 of bit) and (dtClear eq 0 of bit) and (dtQ eq 1) 
       and (dtQbar ne 1)] -> 
	Qbar ! 1 of bit;
	JKFlipFlop_BB_Aux_PreClr [J, K, Preset, Clear, Ck, Q, Qbar]
	 (edge, dtJ, dtK, dtPreset, dtClear, dtCk, dtQ, 1 of bit)
    []
      [(dtPreset eq 0 of bit) and (dtClear eq 0 of bit) and (dtQ ne 1) 
       and (dtQbar eq 1)] -> 
	Q ! 1 of bit;
	JKFlipFlop_BB_Aux_PreClr [J, K, Preset, Clear, Ck, Q, Qbar]
	 (edge, dtJ, dtK, dtPreset, dtClear, dtCk, 1 of bit, dtQbar)
    []	(* preset and clear return to inactive simuteneously from nonstable state *)
      [(dtPreset eq 1) and (dtClear eq 1) and (dtQ eq 1) and (dtQbar eq 1)] -> 
	(
	  i;
	  Q ! 0 of bit;
	  JKFlipFlop_BB_Aux_PreClr [J, K, Preset, Clear, Ck, Q, Qbar]
	   (edge, dtJ, dtK, dtPreset, dtClear, dtCk, 0 of bit, dtQbar)
	[]
	  i;
	  Qbar ! 0 of bit;
	  JKFlipFlop_BB_Aux_PreClr [J, K, Preset, Clear, Ck, Q, Qbar]
	   (edge, dtJ, dtK, dtPreset, dtClear, dtCk, dtQ, 0 of bit)
	) 
    endproc (* JKFlipFlop_BB_Aux_PreClr *)
  endproc (* JKFlipFlop_BB_PreClr *)
')')

# "DFlipFlop_BB_PreClr_Decl" defines a master slave  D flip-flop with pre-set and 
#  pre-clear inputs; pre-set and pre-clear are 0 active.

define(DFlipFlop_BB_PreClr_Pos_Decl, `declare(`$0', `DFlipFlop_BB_PreClr_Decl
  process DFlipFlop_BB_PreClr_Pos [D, Preset, Clear, Ck, Q, Qbar] : noexit :=
    DFlipFlop_BB_PreClr [D, Preset, Clear, Ck, Q, Qbar] (1)
  endproc (* DFlipFlop_BB_PreClr_Pos *)
')')

define(DFlipFlop_BB_PreClr_Neg_Decl, `declare(`$0', `DFlipFlop_BB_PreClr_Decl
  process DFlipFlop_BB_PreClr_Neg [D, Preset, Clear, Ck, Q, Qbar] : noexit :=
    DFlipFlop_BB_PreClr [D, Preset, Clear, Ck, Q, Qbar] (0 of bit)
  endproc (* DFlipFlop_BB_PreClr_Neg *)
')')

define(DFlipFlop_BB_PreClr_Decl, `
  process DFlipFlop_BB_PreClr [D, Preset, Clear, Ck, Q, Qbar] (edge: Bit) :noexit :=
    DFlipFlop_BB_Aux_PreClr [D, Preset, Clear, Ck, Q, Qbar]
     (edge, X of bit, X of bit, X of bit, X of bit, X of bit, X of bit)

    where
    
    process DFlipFlop_BB_Aux_PreClr [D, Preset, Clear, Ck, Q, Qbar]
     (edge, dtD, dtPreset, dtClear, dtCk, dtQ, dtQbar: Bit) : noexit :=
      D ? newdtD: Bit;
      DFlipFlop_BB_Aux_PreClr [D, Preset, Clear, Ck, Q, Qbar]
       (edge, newdtD, dtPreset, dtClear, dtCk, dtQ, dtQbar)  
    []
      Preset ? newdtPreset: Bit;
      DFlipFlop_BB_Aux_PreClr [D, Preset, Clear, Ck, Q, Qbar]
       (edge, dtD, newdtPreset, dtClear, dtCk, dtQ, dtQbar)
    []
      Clear ? newdtClear: Bit;
      DFlipFlop_BB_Aux_PreClr [D, Preset, Clear, Ck, Q, Qbar]
       (edge, dtD, dtPreset, newdtClear, dtCk, dtQ, dtQbar)
    []
      Ck ? newdtCk: Bit;
      (
        [(dtCk eq not (edge)) and (newdtCk eq edge)] -> (* appropriate clock transition *)
          (
            [(dtPreset eq 1) and (dtClear eq 1)] ->  (* preset and clear inactive *)
              (
		[(dtD eq X) and (dtQ ne X)] -> (* inputs undecided, outputs remain unchanged *)
		  DFlipFlop_BB_Aux_PreClr [D, Preset, Clear, Ck, Q, Qbar]
		   (edge, dtD, dtPreset, dtClear, newdtCk, dtQ, dtQbar)
		[]
		  [(dtD eq X) and (dtQ eq X)] ->
		    (
		      Q ? newdtQ : Bit [newdtQ ne X]; Qbar ! not (newdtQ);
		      DFlipFlop_BB_Aux_PreClr [D, Preset, Clear, Ck, Q, Qbar]
		       (edge, dtD, dtPreset, dtClear, newdtCk, newdtQ, not (newdtQ))
		    []
		      Qbar ? newdtQbar : Bit [newdtQbar ne X]; Q ! not (newdtQbar);
		      DFlipFlop_BB_Aux_PreClr [D, Preset, Clear, Ck, Q, Qbar]
		       (edge, dtD, dtPreset, dtClear, newdtCk, not (newdtQbar), newdtQbar)
		    )         
		[] 
		  [(dtD eq 0 of bit) and (dtQ ne 0 of bit) ] ->     
		    (
		      (
		        Q ! 0 of bit;
			exit
		      |||
		        Qbar ! 1 of bit;
			exit
		      ) 
		    >>
		      DFlipFlop_BB_Aux_PreClr [D, Preset, Clear, Ck, Q, Qbar] 
		       (edge, dtD, dtPreset, dtClear, newdtCk, 0 of bit, 1 of bit)
		    )
		[]      
		  [(dtD eq 1) and (dtQ ne 1)] ->     
		    (
		      (
		        Q ! 1 of bit;
			exit
		      |||
		        Qbar ! 0 of bit;
			exit
		      ) 
		    >>
		      DFlipFlop_BB_Aux_PreClr [D, Preset, Clear, Ck, Q, Qbar] 
		      (edge, dtD, dtPreset, dtClear, newdtCk, 1 of bit, 0 of bit)
		    )
              )				(* end of preset and clear inactive *)
          []
	    [((dtPreset ne 1) or (dtClear ne 1)) and (dtQ ne X)] -> (* no change *)
	      DFlipFlop_BB_Aux_PreClr [D, Preset, Clear, Ck, Q, Qbar]
	       (edge, dtD, dtPreset, dtClear, newdtCk, dtQ, dtQbar)
          []
            [((dtPreset ne 1) or (dtClear ne 1)) and (dtQ eq X)] ->
	      (
	        Q ? newdtQ : Bit [newdtQ ne X]; Qbar ! not (newdtQ);
	        DFlipFlop_BB_Aux_PreClr [D, Preset, Clear, Ck, Q, Qbar]
	         (edge, dtD, dtPreset, dtClear, dtCk, newdtQ, not (newdtQ)
	      )
          []
	    Qbar ? newdtQbar : Bit [newdtQbar ne X]; Q ! not (newdtQbar);
	    DFlipFlop_BB_Aux_PreClr [D, Preset, Clear, Ck, Q, Qbar]
	     (edge, dtD, dtPreset, dtClear, newdtCk, not (newdtQbar), newdtQbar)
	      )
          )				(* end of appropriate clock transition *)
	[]
	[(dtCk eq edge) or (newdtCk ne edge) or (dtCk eq X)] -> 
	  DFlipFlop_BB_Aux_PreClr [D, Preset, Clear, Ck, Q, Qbar]
	    (edge, dtD, dtPreset, dtClear, newdtCk, dtQ, dtQbar)
      )					(* end of clock transition *)
    []					(* preset *)
      [(dtPreset eq 0 of bit) and (dtClear eq 1)and (dtQ ne 1)] ->          
        (
	  (
	    Q ! 1 of bit;
	    exit
	  |||
	    Qbar ! 0 of bit;
	    exit
	  ) 
	>>
	  DFlipFlop_BB_Aux_PreClr [D, Preset, Clear, Ck, Q, Qbar] 
	   (edge, dtD, dtPreset, dtClear, dtCk, 1 of bit, 0 of bit)) 
    []					(* clear *)
      [(dtPreset eq 1) and (dtClear eq 0 of bit) and (dtQ ne 0 of bit)
       and (dtQbar ne 1)] ->      
        (
	  (
	    Q ! 0 of bit;
	    exit
	  |||
	    Qbar ! 1 of bit;
	    exit
	  ) 
	>>
	  DFlipFlop_BB_Aux_PreClr [D, Preset, Clear, Ck, Q, Qbar] 
	   (edge, dtD, dtPreset, dtClear, dtCk, dtQ, 1 of bit))
    [] (* preset and clear return to inactive simutineously from nonstable state *)
      [(dtPreset eq 1) and (dtClear eq 1) and (dtQ eq 1) and (dtQbar eq 1)] -> 
	(
	  i;
	  Q ! 0 of bit;
	  DFlipFlop_BB_Aux_PreClr [D, Preset, Clear, Ck, Q, Qbar]
	   (edge, dtD, dtPreset, dtClear, dtCk, 0 of bit, dtQbar)
	[]
	  i;
	  Qbar ! 0 of bit;
	  DFlipFlop_BB_Aux_PreClr [D, Preset, Clear, Ck, Q, Qbar]
	   (edge, dtD, dtPreset, dtClear, dtCk, dtQ, 0 of bit)
	)
    endproc (* DFlipFlop_BB_Aux_PreClr *)
  endproc (* DFlipFlop_BB_PreClr *)
')')
