package uk.ac.stir.cs.android.easter;

import android.app.Activity;
import android.app.AlertDialog;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.content.pm.PackageManager.NameNotFoundException;
import android.os.Bundle;
import android.text.SpannableString;
import android.text.util.Linkify;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.View;
import android.view.animation.AnticipateOvershootInterpolator;
import android.widget.TextView;

import java.text.DateFormat;
import java.util.Calendar;
import java.util.Date;

// import kankan.wheel.R;
import kankan.wheel.widget.WheelView;
import kankan.wheel.widget.adapters.NumericWheelAdapter;

/**
  Calculate the date of Easter Sunday for a range of years in the Gregorian
  calendar. The algorithm is taken from "Puzzles and Paradoxes", T. H. O'Beirne,
  1965 (p. 180).

  @author	Kenneth J. Turner
  @version	1.0 (9th August 2011)
  <br>		2.0 (5th October 2022)
		<ul>
		  <li>
		    method "getEaster" now returns a date
		  </li>
		  <li>
		    method "onCreate" now uses the following year if Easter has
		    passed in the current year
		  </li>
		  <li>
		    overall application reworked for Android 13 (API level 33),
		    with small changes such as text colour and button background
		  </li>
		</ul>
*/
public class EasterActivity extends Activity {

  /** Padding for "About Easter" popup */
  private final static int ABOUT_PADDING = 5;

  /** Text size for "About Easter" popup */
  private final static int ABOUT_TEXT_SIZE = 20;

  /** Maximum difference between start and end years */
  private final static int DATES_MAX = 100;

  /** Size of text in wheel */
  private final static int WHEEL_TEXT_SIZE = 28;

  /** Number of visible items in wheel */
  private final static int WHEEL_VISIBLE = 2;

  /** Minimum start year (first Easter of the Gregorian calendar) */
  private final static int YEAR_MIN = 1583;

  /** Maximum start year */
  private final static int YEAR_MAX = 9999;

  /** Current calendar */
  private Calendar currentCalendar;

  /** Current year */
  private int currentYear;

  /** Date format */
  private DateFormat dateFormat;

  /** Date list */
  private TextView dateList;

  /** End year digits */
  private WheelView endYear1, endYear2, endYear3, endYear4;

  /** Start year digits */
  private WheelView startYear1, startYear2, startYear3, startYear4;

  /**
    Create a wheel with digits 0..9.

    @param identifier	wheel widget identifier
    @return		created wheel
  */
  private WheelView createWheel(int identifier) {
    WheelView wheel = (WheelView) findViewById(identifier);
    NumericWheelAdapter wheelAdapter = new NumericWheelAdapter(this, 0, 9);
    wheelAdapter.setTextSize(WHEEL_TEXT_SIZE);
    wheel.setVisibleItems(WHEEL_VISIBLE);
    wheel.setViewAdapter(wheelAdapter);
    wheel.setCyclic(true);
    wheel.setInterpolator(new AnticipateOvershootInterpolator());
    return(wheel);
  }

  /**
    Return whether Easter has already passed for the given year.

    @param year		year for Easter
    @return		whether Easter has already passed for the given year
  */
  private boolean easterPassed(int year) {
    Date now = new Date();
    Date easter = getEaster(year);
    return(now.after(easter));
  }

  /**
    Return Easter Sunday date in localised form for the given year. The
    algorithm is by T. H. O'Beirne.

    @param year		year for Easter
    @return		Easter Sunday date
  */
  @SuppressWarnings("deprecation")
  private Date getEaster(int year) {
    // intermediate values (n = month number, p = day number)
    int a, b, c, d, e, g, h, j, k, l, m, n, p, q;
    b = year / 100;
    c = year % 100;
    a = (5 * b + c) % 19;
    d = (3 * (b + 25)) / 4;
    e = (3 * (b + 25)) % 4;
    g = (8 * (b + 11)) / 25;
    h = (19 * a + d - g) % 30;
    m = (a + 11 * h) / 319;
    j = (60 * (5 - e) + c) / 4;
    k = (60 * (5 - e) + c) % 4;
    l = (2 * j - k - h + m) % 7;
    n = (h - m + l + 110) / 30;
    q = (h - m + l + 110) % 30;
    p = q + 5 - n;

    int day = p;
    int month = n - 1;
    year -= 1900;
    return(new Date(year, month, day));
  }

  /**
    Display list of Easter dates from start to end year. One or other will be
    set to the current year if incorrect (start empty or before first year, end
    empty or before start).

    @param view		current view
  */
  public void listDates(View view) {
    int start1 = startYear1.getCurrentItem();
    int start2 = startYear2.getCurrentItem();
    int start3 = startYear3.getCurrentItem();
    int start4 = startYear4.getCurrentItem();
    int start = start1*1000 + start2*100 + start3*10 + start4;
    if (start < YEAR_MIN)
      start = YEAR_MIN;

    int end1 = endYear1.getCurrentItem();
    int end2 = endYear2.getCurrentItem();
    int end3 = endYear3.getCurrentItem();
    int end4 = endYear4.getCurrentItem();
    int end = end1*1000 + end2*100 + end3*10 + end4;
    if (end < start)
      end = start;
    int yearDifference = end - start;
    if (yearDifference > DATES_MAX)
      end = start + DATES_MAX;
    if (end > YEAR_MAX) {
      start = YEAR_MAX - DATES_MAX;
      end = YEAR_MAX;
    }

    setYears(start, end, true);
    String dates = "";
    for (int year = start; year <= end; year++) {
      Date easter = getEaster(year);
      dates += dateFormat.format(easter) + "\n";
    }
    dateList.setText(dates);
  }

  /**
    React to activity creation by setting up key variables.

    @param savedInstanceState	saved state
  */
  @Override
  public void onCreate(Bundle savedInstanceState) {
    super.onCreate(savedInstanceState);
    setContentView(R.layout.main);
    dateFormat =
      android.text.format.DateFormat.getDateFormat(getApplicationContext());
    currentCalendar = Calendar.getInstance();
    currentYear = currentCalendar.get(Calendar.YEAR);
    int activeYear = easterPassed(currentYear) ? currentYear + 1: currentYear;
    startYear1 = createWheel(R.id.startYear1);
    startYear2 = createWheel(R.id.startYear2);
    startYear3 = createWheel(R.id.startYear3);
    startYear4 = createWheel(R.id.startYear4);
    endYear1 = createWheel(R.id.endYear1);
    endYear2 = createWheel(R.id.endYear2);
    endYear3 = createWheel(R.id.endYear3);
    endYear4 = createWheel(R.id.endYear4);
    dateList = (TextView) findViewById(R.id.dateList);
    setYears(activeYear, activeYear, false);
  }

  /**
    Create options menu.

    @param item		menu item
    @return		menu created (always true)
  */
  @Override
  public boolean onCreateOptionsMenu(Menu menu) {
    MenuInflater inflater = getMenuInflater();
    inflater.inflate(R.menu.menu, menu);
    return(true);
  }

  /**
    Handle menu selection.

    @param item		menu item
    @return		menu item processed (always true)
  */
  @SuppressWarnings("deprecation")
  @Override
  public boolean onOptionsItemSelected(MenuItem item) {
    String packageName = getPackageName();
    PackageInfo packageInfo = null;
    String versionName = "";
    try {
      packageInfo = getPackageManager().getPackageInfo(
	packageName, PackageManager.GET_META_DATA);
      versionName = packageInfo.versionName;
    }
    catch (NameNotFoundException exception) {
      // ignore
    }
    String aboutTitle = getString(R.string.aboutString) + " " +
      getString(R.string.app_name);
    String versionString = getString(R.string.versionString) + " " +
      versionName;
    String aboutText = getString(R.string.aboutText);
    final TextView message = new TextView(this);
    final SpannableString s = new SpannableString(aboutText);
    message.setPadding(
      ABOUT_PADDING, ABOUT_PADDING, ABOUT_PADDING, ABOUT_PADDING);
    message.setTextSize(ABOUT_TEXT_SIZE);
    message.setText(versionString + "\n\n" + s);
    Linkify.addLinks(message, Linkify.ALL);
    AlertDialog.Builder dialogueBuilder = new AlertDialog.Builder(this);
    dialogueBuilder.setTitle(aboutTitle);
    dialogueBuilder.setCancelable(true);
    dialogueBuilder.setIcon(R.drawable.icon);
    dialogueBuilder.setPositiveButton(
      getString(android.R.string.ok), null);
    dialogueBuilder.setView(message);
    AlertDialog alertDialogue = dialogueBuilder.create();
    alertDialogue.show();
    return(true);
  }

  /**
    Reset start and end years to current year.

    @param view		current view
  */
  public void resetYears(View view) {
    setYears(currentYear, currentYear, true);
  }

  /**
    Set start and end year dates.

    @param start	start year
    @param end		end year
    @param animate	whether to animate the change
  */
  private void setYears(int start, int end, boolean animate) {
    startYear1.setCurrentItem(start / 1000, animate);
    startYear2.setCurrentItem((start / 100) % 10, animate);
    startYear3.setCurrentItem((start / 10) % 10, animate);
    startYear4.setCurrentItem(start % 10, animate);
    endYear1.setCurrentItem(end / 1000, animate);
    endYear2.setCurrentItem((end / 100) % 10, animate);
    endYear3.setCurrentItem((end / 10) % 10, animate);
    endYear4.setCurrentItem(end % 10, animate);
  }

}