// Easter.java

import java.io.IOException;

import javax.microedition.lcdui.*;
import javax.microedition.midlet.*;

import com.nokia.mid.ui.FullCanvas;

/**
  MIDlet to calculate the date of Easter Sunday.

  @author	Ken Turner (www.cs.stir.ac.uk/~kjt)
  @version	1.0 (31st December 2006)
*/

public class Easter extends MIDlet implements CommandListener {

  /** Layout for centred item */
  private final static int itemCentred =
    Item.LAYOUT_2 | Item.LAYOUT_CENTER | Item.LAYOUT_NEWLINE_AFTER;

  /** Minimum year */
  private final static int yearMinimum = 1600;

  /** Maximum year */
  private final static int yearMaximum = 9999;

  /** Command to report about program */
  private final static Command commandAbout =
    new Command("About", Command.SCREEN, 1);

  /** Command to compute Easter Sunday */
  private final static Command commandEaster =
    new Command("Easter", Command.SCREEN, 0);

  /** Command to exit program */
  private final static Command commandExit =
    new Command("Exit", Command.EXIT, 1);

  /** Command to accept alert */
  private final static Command commandOK =
    new Command("OK", Command.OK, 1);

  /** Device display */
  private Display display;

  /** Alert about program */
  private final static Alert easterAbout =
    new Alert("About Easter",
      "Author: (C) Kenneth J. Turner\n\n" +
      "URL: www.cs.stir.ac.uk/~kjt/software/pda/easter.html\n\n" +
      "Version: 1.0 (2006-12-31)\n\n" +
      "Licence: GPL (no fee, no warranty)\n\n" +
      "Year: " + yearMinimum + " to " + yearMaximum +
	" (Red if selected, Left or Clear key to delete digits)\n\n" +
      "Algorithm: 'Puzzles and Paradoxes', T. H. O'Beirne, 1965 (p. 180)",
      null, AlertType.INFO);

  /** Field for Easter Sunday value */
  private StringItem easterDate =
    new StringItem("Date: ", "");

  /** Form to collect Easter year and output Easter date */
  private Form easterForm;

  /** Space before easter date */
  private final Spacer easterSpacer = new Spacer(0, 0);

  /** Image for program */
  private static ImageItem imageEaster = null;

  static {
    try {
     imageEaster =
       new ImageItem("", Image.createImage("/EasterLarge.png"),
	 itemCentred, "");
    }
    catch (IOException exception) {
    }
  }

  /** Alert to report year incorrect */
  private final static Alert yearError =
    new Alert("Year Error",
      "Year must be " + yearMinimum + " to " + yearMaximum,
      null, AlertType.ERROR);

  /** Year number */
  private NumberField yearField;

  /** Space before easter year */
  private final Spacer yearSpacer = new Spacer(0, 0);

  /**
    Construct instance, setting up global variables.
  */

  public Easter() {
    display = Display.getDisplay(this);
    easterForm = new Form("Easter Sunday");
    yearField = new NumberField(display, "Enter Date:", 4);

    easterAbout.setTimeout(Alert.FOREVER);
    easterDate.setLayout(itemCentred);
    easterSpacer.setLayout(itemCentred);
    yearError.addCommand(commandOK);
    yearError.setTimeout(5000);
    yearField.setLayout(itemCentred);
    yearSpacer.setLayout(itemCentred);

    easterForm.append(yearSpacer);
    easterForm.append(yearField);
    easterForm.append(imageEaster);
    easterForm.append(easterSpacer);
    easterForm.append(easterDate);

    easterForm.addCommand(commandAbout);
    easterForm.addCommand(commandExit);
    easterForm.addCommand(commandEaster);
    easterForm.setCommandListener(this);
  }

  /**
    Process command.

    @param command	command
    @param displayable	displayable object
  */

  public void commandAction(Command command, Displayable displayable) {
    if (command == commandAbout) {
      display.setCurrent(easterAbout);
    }
    else if (command == commandEaster) {
      String yearString = yearField.getString();
      int year =  yearString != null && yearString.length() > 0 ?
	Integer.parseInt(yearString) : 0;
      yearField.resetNumber();
      if (yearMinimum <= year && year <= yearMaximum)
	setDate(year);
      else
	display.setCurrent(yearError);
      display.setCurrentItem(yearField);
    }
    else if (command == commandExit) {
      destroyApp(false);
      notifyDestroyed();
    }
  }

  /**
    Return ordinal value for day number.

    @param day	day number in month
    @return	day ordinal number
  */

  private String dayOf (int day) {
    String ordinal =
      day == 1 || day == 21 || day == 31 ? "st" :
      day == 2 || day == 22 ? "nd" :
      day == 3 || day == 23 ? "rd" :
      "th";
    return(day + ordinal);
  }

  /**
    Close down MIDlet, taking no action.

    @param unconditional	clean up if true (unused)
  */

  public void destroyApp(boolean unconditional) {
  }

  /**
    Return month name in English.

    @param month	month number
    @return		month name in English
  */

  private String monthOf (int month) {
    String name;				// month name
    switch (month) {				// check month number
      case  1: name = "January"; break;
      case  2: name = "February"; break;
      case  3: name = "March"; break;
      case  4: name = "April"; break;
      case  5: name = "May"; break;
      case  6: name = "June"; break;
      case  7: name = "July"; break;
      case  8: name = "August"; break;
      case  9: name = "September"; break;
      case 10: name = "October"; break;
      case 11: name = "November"; break;
      case 12: name = "December"; break;
      default: name = "Unknown";
    }
    return (name);				// return month name
  }

  /**
    Pause MIDlet, taking no action.
  */

  public void pauseApp() {
  }

  /**
    Set date of Easter Sunday for year. See 'Puzzles and Paradoxes',
    T. H. O'Beirne, 1965 (p. 180) for algorithm.

    @param year	year number
  */

  public void setDate(int year) {
    int a, b, c, d, e, g, h, j, k, l, m, n, p, q;

    b = year / 100;				// compute intermediate values
    c = year % 100;
    a = (5 * b + c) % 19;
    d = (3 * (b + 25)) / 4;
    e = (3 * (b + 25)) % 4;
    g = (8 * (b + 11)) / 25;
    h = (19 * a + d - g) % 30;
    m = (a + 11 * h) / 319;
    j = (60 * (5 - e) + c) / 4;
    k = (60 * (5 - e) + c) % 4;
    l = (2 * j - k - h + m) % 7;
    n = (h - m + l + 110) / 30;			// n = month number
    q = (h - m + l + 110) % 30;
    p = q + 5 - n;				// p = day number

    easterDate.setText(			// set Easter date
      dayOf(p) + " " + monthOf(n) + " " + year);
  }

  /**
    Start MIDlet, setting up single main form.
  */

  public void startApp() {
    display.setCurrentItem(yearField);
  }

}

/**
  Custom item to handle number input.

  @author	Ken Turner (www.cs.stir.ac.uk/~kjt)
  @version	1.0 (31st December 2006)
*/

class NumberField extends CustomItem {

  /** Colour if focus is on field (red) */
  private static final int colourActive = 0x00FF0000;

  /** Colour if focus is not on field (blue) */
  private static final int colourInactive = 0x000000FF;

  private Display numberDisplay;

  /** Minimum height for value, based on default font */
  private static final int minHeight = Font.getDefaultFont().getHeight();

  /** */
  private static final int minWidth = 4 * minHeight;

  /** Sony Ericsson clear key code */
  private static final int SE_CLEAR = -8;

  /** Colour that is current for field */
  private int numberColour;

  /** Maximum digits in number */
  private int numberSize;

  /** Number value */
  private String numberValue;

  /**
    Construct instance, setting up global variables.

    @param label	field label
    @param size		maximum digits in number
  */

  public NumberField(Display display, String label, int size) {
    super(label);
    numberDisplay = display;
    numberSize = size;
    resetNumber();
  }

  /**
    Return minimum height for value.

    @return		minimum height in pixels
  */

  public int getMinContentHeight() {
    return(minHeight);
  }

  /**
    Return minimum width for value.

    @return		minimum width in pixels
  */

  public int getMinContentWidth() {
    return(minWidth);
  }

  /**
    Return preferred height for value.

    @param width	proposed width in pixels
    @return		preferred height in pixels
  */

  public int getPrefContentHeight(int width) {
    return(minHeight);
  }

  /**
    Return preferred width for value.

    @param height	proposed height in pixels
    @return		preferred width in pixels
  */

  public int getPrefContentWidth(int height) {
    return(minWidth);
  }

  /**
    Note key release.

    @param keyCode	code for key
  */

  public void keyReleased(int keyCode) {
    int yearLength = numberValue.length();
    if (yearLength < numberSize &&
	Canvas.KEY_NUM0 <= keyCode && keyCode <= Canvas.KEY_NUM9)
      numberValue += (char) keyCode;
    else if (yearLength > 0 &&
	(keyCode == FullCanvas.KEY_LEFT_ARROW  || keyCode == SE_CLEAR))
      numberValue = numberValue.substring(0, yearLength - 1);
    numberColour = colourActive;
    repaint();
  }

  /**
    Paint field.

    @param graphics	graphics object
    @param width	field width (ignored)
    @param height	field height (ignored)
  */

  public void paint(Graphics graphics, int width, int height) {
    graphics.setColor(numberColour);
    graphics.drawString(numberValue, 0, 0, Graphics.TOP | Graphics.LEFT);
  }

  /**
    Return number value.

    @return		number value as string
  */

  public String getString() {
    return(numberValue);
  }

  /**
    Reset number value to empty and colour to active.

    @param value	number value as string
  */

  public void resetNumber() {
    numberValue = "";
    numberColour = colourActive;
    repaint();
  }

  /**
    Set number value.

    @param value	number value as string
  */

  public void setString(String value) {
    numberValue = value;
    repaint();
  }

  /**
    Note focus has entered field.

    @param dir			direction of traversal
    @param viewportWidth	width of container viewport
    @param viewportHeight	height of container viewport
    @param visibleRectangle	visible rectangle
    @return			true/false if internal/no internal traversal
  */

  public boolean traverse(int dir, int viewportWidth, int viewportHeight,
   int[] visibleRectangle) {
    numberColour = colourActive;
    repaint();
    return(false);
  }

  /**
    Note focus has left field.
  */

  public void traverseOut() {
    numberColour = colourInactive;
    repaint();
  }

}
