%{

/*****************************************************************

esttex.l	K. J. Turner	11/05/02

This "lex" program reads Estelle text (which is assumed to be syntactically
valid) from the standard input, and writes it to the standard output in a
form suitable for processing by LaTeX. The flag "-n" causes output lines to
be numbered at the right.

The program translates initial tab and space characters on a line into LaTeX
horizontal space (\hspace). This assumes that indentation will be by small
(e.g. 2) units of space. Because LaTeX limits the number of tab stops in a
"tabbing" environment to around a dozen, it is not possible to have tab
stops every space or two.

Tabs and multiple spaces embedded in a line are translated into LaTeX tab
commands (\>). This assumes that embedded tabs and multiple spaces are
likely to be used to format comments and the like, for which a coarser
degree of control over spacing is acceptable.

Since the output is assumed to be embedded in a LaTeX "tabbing" environment,
a LaTeX newline command (\\) is appended to each line.

Estelle keywords are converted to bold (\bf) Keywords are recognised only in
lower case or with capitalisation of each word (e.g. "ForOne", "TimeScale").
LaTeX special characters are escaped.

The program recognises "{" or "(*" as beginning comments, "}" or "*)" as
ending comments, and "'" as beginning or ending a string.  Comments may not
be nested. Comment delimiters are treated literally inside strings, and
string delimiters are treated as comments inside comments. Estelle keywords
are not emboldened inside comments or strings. To preserve the appearance of
strings, embedded tabs and multiple spaces are converted into tabs. This is
only a rough approximation because of the proportional spacing.

*****************************************************************/

#define Tab 8		/* number of characters between tab positions */

#define Spc 0.5		/* number "em"s for horizontal space unit */

#define Prog "esttex"	/* program name */

int IpPos = 0;		/* current input position (0 = first) */

int OpPos = 0;		/* current output position (0 = first) */

int TabPos = 0;		/* current tab position (0 = first) */

int InComment = 0;	/* indicates whether inside a comment (1 =
			   yes) */
int InString = 0;       /* indicates whether inside a string (1 =
			   yes) */
int LineNum = 0;	/* current line number - line numbering off by
			   default */

/* output current text */

void echo ();

/*  At the start of a line make good the difference between the input
    and output positions by using "\hspace". Otherwise, round the input
    position up to next tab stop, and output LaTeX tabs to bring the
    output position up to the input position. */

void Input () {
  int NumTabs;

  if (OpPos < IpPos) {				/* positions agree? */
    if (OpPos == 0) {				/* start of line? */
      OpPos = IpPos;
      printf ("\\hspace{%.1fem}", Spc * IpPos);
    }
    else {
	NumTabs = (IpPos - 1) / Tab + 1;	/* next tab stop */
	OpPos = Tab * NumTabs;
	while (TabPos < NumTabs) {
	  printf ("\\>");
	  TabPos++;
	}
    }
  }
}

/* Output current identifier string, updating input and output
   positions. Escape special LaTeX characters as required. */

void OutputId () {
  int Ind;
  char Ch;

  for (Ind = 0; Ind < yyleng; Ind++) {
    Ch = yytext [Ind];
    if (Ch == '#' || Ch == '$' || Ch == '&' || Ch == '%' || Ch == '{' || Ch == '}') {
      putchar ('\\');
      putchar (Ch);
    }
    else if (Ch == '<' || Ch == '>' || Ch == '|' || ((Ch == '-') && !InComment)) {
      putchar ('$');
      putchar (Ch);
      putchar ('$');
    }
    else if (Ch == '~')
      printf ("$\\sim$");
    else if (Ch == '^')
      printf ("\\^{}");
    else if (Ch == '_')
      printf ("\\_\\,");
    else if (Ch == '\\')
      printf ("$\\backslash$");
    else putchar (Ch);
  }
  IpPos += yyleng;
  OpPos += yyleng;
}

/* Output current symbol string, updating input and output positions.
   Escape special LaTeX characters as required, and use special symbols
   where necessary. */

void OutputSym () {
  if ((!strcmp (yytext, "-")) || (!strcmp (yytext, "<")) ||
	(!strcmp (yytext, ">")) || (!strcmp (yytext, "<>")) ||
	(!strcmp (yytext, "<=")) || (!strcmp (yytext, ">=")) ||
	(!strcmp (yytext, "@")))
    printf ("$%s$", yytext);
  else if (!strcmp (yytext, "^"))
    printf ("\\%s{}", yytext);
  else if (!strcmp (yytext, "@"))
    printf ("\\%s", yytext);
  else
    echo ();
  IpPos += yyleng;
  OpPos += yyleng;
}

main (int argc, char *argv[]) {
  if (argc > 1)
    if (argc == 2)
      if (!strcmp (argv[1], "-n"))
        LineNum = 1;				/* turn on line numbering */
      else
        printf ("%s: unknown option %s\n", Prog, argv[1]);
    else
      printf ("%s: too many arguments\n", Prog);
  yylex ();
}

%}

%a 4500						/* increase number of transitions */
%e 1500						/* increase number of nodes */
%n 750						/* increase number of states */
%o 5000						/* increase number of output slots */
%p 5000						/* increase number of positions */

Letter		[a-zA-Z_]
Digit		[0-9]

Identifier      {Letter}({Letter}|{Digit})*

Comment		"{"|"(*"|"}"|"*)"

Symbol1		"+"|"-"|"*"|"/"|"="|"<"|">"|"["|"]"|"."|","
Symbol2		":"|";"|"^"|"("|")"|"<>"|"<="|">="|":="|".."|"'"|"@"
Symbol		{Symbol1}|{Symbol2}

String          "'"

%%

^" "+ |
" "" "+	IpPos += yyleng;

\t+ {
  IpPos = Tab * (IpPos / Tab + yyleng);
}

\n {
  IpPos = 0;
  OpPos = 0;
  TabPos = 0;
  if (LineNum > 0)
    printf ("\\`\\footnotesize%d\\\\\n", LineNum++);
  else
    printf ("\\\\\n");
}

{Comment} {
  Input ();
  InComment = !InComment;
  OutputId ();
}

{String} {
  Input ();
  if (!InComment) InString = !InString;
  OutputId ();
}

{Symbol} {
  Input ();
  OutputSym ();
}

(activity|all|and|any|array|attach|begin|body|by|case|channel|common)	|
(Activity|All|And|Any|Array|Attach|Begin|Body|By|Case|Channel|Common)	|
(connect|const|default|delay|detach|disconnect|div|do|downto|else|end)	|
(Connect|Const|Default|Delay|Detach|Disconnect|Div|Do|Downto|Else|End)	|
(exist|export|external|for|forone|forward|from|function|goto|if|in)	|
(Exist|Export|External|For|ForOne|Forward|From|Function|Goto|If|In)	|
(individual|init|initialize|ip|label|mod|module|modvar|name|nil|not|of)	|
(Individual|Init|Initialize|Ip|Label|Mod|Module|ModVar|Name|Nil|Not|Of)	|
(or|otherwise|output|packed|primitive|priority|procedure|process)	|
(Or|Otherwise|Output|Packed|Primitive|Priority|Procedure|Process)	|
(provided|pure|queue|record|release|repeat|same|set|specification)	|
(Provided|Pure|Queue|Record|Release|Repeat|Same|Set|Specification)	|
(state|stateset|suchthat|systemactivity|systemprocess|terminate|then)	|
(State|StateSet|SuchThat|SystemActivity|SystemProcess|Terminate|Then)	|
(timescale|to|trans|type|until|var|when|while|with)			|
(TimeScale|To|Trans|Type|Until|Var|When|While|With) {
  Input ();
  if (!InComment && !InString) printf ("{\\bf ");
  OutputId ();
  if (!InComment && !InString) printf ("}");
}

{Identifier}|. {
  Input ();
  OutputId ();
}

%%

/* output current text */

void echo () {
  ECHO;
}

