%{

/*****************************************************************

javtex.lex	K. J. Turner	11/05/02

This "lex" program reads Java text (which is assumed to be syntactically
valid) from the standard input, and writes it to the standard output in a
form suitable for processing by LaTeX. The flag "-n" causes output lines to
be numbered at the right.

The program translates initial tab and space characters on a line into LaTeX
horizontal space (\hspace). This assumes that indentation will be by small
(e.g. 2) units of space. Because LaTeX limits the number of tab stops in a
"tabbing" environment to around a dozen, it is not possible to have tab
stops every space or two.

Tabs and multiple spaces embedded in a line are translated into LaTeX tab
commands (\>). This assumes that embedded tabs and multiple spaces are
likely to be used to format comments and the like, for which a coarser
degree of control over spacing is acceptable.

Since the output is assumed to be embedded in a LaTeX "tabbing" environment,
a LaTeX newline command (\\) is appended to each line.

Java keywords are converted to bold (\bf). Keywords are recognised only in
lower case or with capitalisation of each word (e.g. "GoTo",
"InstanceOf"). LaTeX special characters are escaped.

The program recognises "{" or "(*" as beginning comments, "}" or "*)" as
ending comments, and "'" as beginning or ending a string.  Comments may not
be nested. Comment delimiters are treated literally inside strings, and
string delimiters are treated as comments inside comments. Java keywords are
not emboldened inside comments or strings. To preserve the appearance of
strings, embedded tabs and multiple spaces are converted into tabs. This is
only a rough approximation because of the proportional spacing.

*****************************************************************/

#define Tab 8		/* number of characters between tab positions */

#define Spc 0.5		/* number "em"s for horizontal space unit */

#define Prog "javtex"	/* program name */

int IpPos = 0;		/* current input position (0 = first) */

int OpPos = 0;		/* current output position (0 = first) */

int TabPos = 0;		/* current tab position (0 = first) */

int InComment = 0;	/* indicates whether inside a comment (1
			   = yes) */

int InLineComment = 0;	/* indicates whether inside a to-end-of-line comment (1
			   = yes) */

int InString = 0;	/* indicates whether inside a string (1 = yes) */

int LineNum = 0;	/* current line number - line numbering off by
			   default */

/* output current text */

void echo ();

/*  At the start of a line make good the difference between the input
    and output positions by using "\hspace". Otherwise, round the input
    position up to next tab stop, and output LaTeX tabs to bring the
    output position up to the input position. */

void Input () {
  int NumTabs;

  if (OpPos < IpPos) {				/* positions disagree? */
    if (OpPos == 0) {				/* start of line? */
      OpPos = IpPos;
      printf ("\\hspace{%.1fem}", Spc * IpPos);
    }
    else {
	NumTabs = (IpPos - 1) / Tab + 1;	/* next tab stop */
	OpPos = Tab * NumTabs;
	while (TabPos < NumTabs) {
	  printf ("\\>"); TabPos++;
	}
    }
  }
}

main (int argc, char *argv[]) {
  if (argc > 1)
    if (argc == 2)
      if (!strcmp (argv[1], "-n"))
        LineNum = 1;				/* turn on line numbering */
      else
        printf ("%s: unknown option %s\n", Prog, argv[1]);
    else
      printf ("%s: too many arguments\n", Prog);
  yylex ();
}

/* Output current parser string, updating input and output positions */

void Output () {
  echo ();
  IpPos += yyleng;
  OpPos += yyleng;
}

/* Output current identifier string, updating input and output
   positions. Escape special LaTeX characters as required. */

void OutputId () {
  int Ind;
  char Ch;

  for (Ind = 0; Ind < yyleng; Ind++) {
    Ch = yytext [Ind];
    if (Ch == '$') {
      putchar ('\\');
      putchar (Ch);
    }
    else if (Ch == '_') {
      putchar ('\\');
      putchar (Ch);
      if (Ind < yyleng - 1) {
        if (strchr ("fhilrszABEFILMNPXZ", yytext [Ind + 1])) {
	  putchar ('\\');
	  putchar (',');
	}
      }
    }
    else putchar (Ch);
  }
  IpPos += yyleng;
  OpPos += yyleng;
}

%}

%e 1000					/* increase number of nodes */

%n 600					/* increase number of states */

%p 4000					/* increase number of positions */

Letter		[a-zA-Z_\$]
Digit		[0-9]
Special		[#{}\$_&%]

Identifier      {Letter}({Letter}|{Digit})*

Comment		"/*"|"*/"|"//"

String		["]

%%

^" "+ |
" "" "+	IpPos += yyleng;

\t+		IpPos = Tab * (IpPos / Tab + yyleng);

\n {
  IpPos = 0;
  OpPos = 0;
  TabPos = 0;
  if (LineNum > 0)
    printf ("\\`\\footnotesize%d\\\\\n", LineNum++);
  else
    printf ("\\\\\n");
  if (InLineComment) {
    InComment = 0; InLineComment = 0;
  }
}

\<|\>|\| {
  Input ();
  putchar ('$');
  putchar (yytext [0]);
  putchar ('$');
  IpPos = ++OpPos;
}

~ {
  Input ();
  printf ("$\\sim$");
  IpPos = ++OpPos;
}

\^ {
  Input ();
  printf ("\\^{}");
  IpPos = ++OpPos;
}

\\ {
  Input ();
  printf ("$\\backslash$");
  IpPos = ++OpPos;
}

{Special} {
  Input ();
  putchar ('\\');
  putchar (yytext [0]);
  IpPos = ++OpPos;
}

{Comment} {
  Input ();
  if (!InString) {
    if (!strcmp (yytext, "//")) {
      InComment = 1; InLineComment = 1;
    }
    else
      if (!InLineComment)
	InComment = !InComment;
  }
  Output ();
}

{String} {
  Input ();
  if (!InComment)
    InString = !InString;
  Output ();
}

(abstract|boolean|break|byte|case|cast|catch|char)	|
(Abstract|Boolean|Break|Byte|Case|Cast|Catch|Char)	|
(class|const|continue|default|do|double|else|extends)	|
(Class|Const|Continue|Default|Do|Double|Else|Extends)	|
(final|finally|float|for|future|generic|goto|if)	|
(Final|Finally|Float|For|Future|Generic|GoTo|If)	|
(implements|import|inner|instanceof|int|interface)	|
(Implements|Import|Inner|InstanceOf|Int|Interface)	|
(long|native|new|operator|outer|package|private)	|
(Long|Native|New|Operator|Outer|Package|Private)	|
(protected|public|rest|return|short|static|super)	|
(Protected|Public|Rest|Return|Short|Static|Super)	|
(switch|synchronized|this|throw|throws|transient|try)	|
(Switch|Synchronized|This|Throw|Throws|Transient|Try)	|
(var|void|volatile|while)				|
(Var|Void|Volatile|While) {
  Input ();
  if (!InComment && !InString)
    printf ("{\\bf ");
  Output ();
  if (!InComment && !InString)
    printf ("}");
}

{Identifier}|. {
  Input ();
  OutputId ();
}

%%

/* output current text */

void echo () {
  ECHO;
}

