/* auxprocs.c */

/*
 * Copyright (c) 1991-1993  R. Nigel Horspool.
 * All rights reserved.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

/* Auxiliary procedures called by GProc */

#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>

#include "tables.h"
#include "output.h"
#include "lexer.h"
#include "la_mml.h"
#include "main.h"
#include "auxprocs.h"

BOOL firstitem = TRUE;
BOOL filenameprinted = FALSE;

BOOL in_appendix = FALSE;	/* TRUE = reached appendixes */

int list_level = 0;		/* 1/2/3/4 = list level 1/2/3/4 */

static char arg_buffer[MAXLEN+1];


static char *get_string_arg() {
    int i = 0;
    char delim = ' ';

    SKIPWSPACE();
    if (ch == '{') {
	delim = '}';  readch();
    }
    SKIPWSPACE();
    for( ; ; ) {
    	if (i < MAXLEN) arg_buffer[i++] = ch;
    	readch();
    	if (isspace(ch) ||
    		ch == delim || ch == EOF)
    	    break;
    }
    if (delim != ' ') {
	SKIPWSPACE();
	if (ch != delim) {
	    LN();
	    FPR(stderr, "mismatched {\n");		/* } */
	}
	readch();
    }
    arg_buffer[i] = '\0';
    if (i == 0) return NULL;
    return arg_buffer;
}


static ENVPTR get_environment() {
    char *s;
    s = get_string_arg();
    if (s == NULL) return &bad_env;
    return lookup_env(s);
}


/* handles \def command */
TOKEN DefProc( char *cmd ) {
    if (curr_matching_mode == MATH || curr_matching_mode == MATHARRAY) {
	LN();
	FPR(stderr, "\\def command used inside math mode\n");
    }
    SKIPWSPACE();
    if (ch != '\\') {
	LN();
	FPR(stderr, "missing macro name after \\def\n");
	return OTHER;
    }
    do {
	readch();
    } while(ch != '{' && ch != EOF);
    discard_output++;
    match_one_unit();
    discard_output--;
    return OTHER;
}


TOKEN VerbProc( char *dummy ) {
    register int delimiter;
    char *star = NULL;
    int ln = linenum;
    char *savefamily = nextfamily;
    short savefont = nextfont;
    
    if (ch == '*') {
	star = "*";
	readch();
    }

    delimiter = ch;
    if (isspace(delimiter)) {
	LN();
	FPR(stderr, "`\\verb%s' must not be followed by white space\n", star);
	goto EXIT;
    }
    readch();
    nextfamily = COURIERFAMILY;
    nextfont = PLAIN;
    while( ch != EOF ) {
	if (ch == '\n')
	    newline();
	else if (ch == ' ') {
	    if (star != NULL)
		copychar('_');	/* should be visible space char */
	    else
		appendstring("\\x11  ");
	} else
	    copychar(ch);
	readch();
	if (ch == delimiter) {
	    readch();
	    goto EXIT;
	}
    }
    linenum = ln;
    LN();
    FPR(stderr, "unterminated \\verb%s construct\n", star);
EXIT:
    nextfamily = savefamily;
    nextfont = savefont;
    return OTHER;
}


TOKEN VSpaceProc( char *dummy ) {
    char buff[256], *cp;
    float val = 0.0;

    SKIPWSPACE();
    if (ch != '{')
	return GProc("%x%n");
    readch();
    SKIPWSPACE();
    if (isdigit(ch) || ch == '.') {
	for( cp=buff;  ; ) {
	    *cp++ = ch;
	    readch();
	    if (!isdigit(ch) && ch != '.') break;
	}
	*cp = '\0';
	sscanf(buff, "%f", &val);
	SKIPWSPACE();
	cp = buff;
	while(isalpha(ch)) {
	    *cp++ = ch;
	    readch();
	}
	*cp = '\0';
	if (strcmp(buff,"in") == 0 || strcmp(buff,"truein") == 0)
	    val *= 72.0;
	else if (strcmp(buff,"cm") == 0)
	    val *= 28.35;
    }
    while( ch != '}' && ch != EOF )
	readch();
    readch();
    (void)sprintf(buff,
	"%%<B1Text><SpaceBefore %2fpt> \\x09 %%n", val);
    GProc(buff);
    return OTHER;
}


TOKEN EnvProc( char *control ) {
    if (*control == '<') {
	char buffer[16], *bp;

	for( bp = buffer, control++;  *control != '>'; )
	    *bp++ = *control++;
	*bp = '\0';
	handle_environment(lookup_env(buffer), RBRACE);
    } else {
	ENVPTR env1, env2;
	int ln = linenum;

	env1 = get_environment();
	handle_environment(env1, ENDCMD);
	register_tag(curr_env->conttag);

	if (env1->kind != VERBATIM) {
	    env2 = get_environment();
	    if (env1 != env2) {
		LN();  FPR(stderr,
		    "closing env. name (%s) != opening name (%s) on line %d\n",
		    env2->name, env1->name, ln);
	    }
	}
    }
    return OTHER;
}


TOKEN MboxProc( char *dummy ) {
    ACTIONPTR meptr;
    TEXTMODE prev_mode = curr_matching_mode;
    char *savefamily = nextfamily;
    short savefont = nextfont;
    short savesize = nextsize;

    curr_matching_mode = PARA;
    nextfont = PLAIN;  nextfamily = TIMESFAMILY;
    meptr = get_token();
    if (GProc(meptr->code) == LBRACE)
	match_grouping(RBRACE);
    curr_matching_mode = prev_mode;
    nextfont = savefont;  nextfamily = savefamily;  nextsize = savesize;
    return OTHER;
}


TOKEN NewEnvProc( char *num ) {
    int n;
    n = atoi(num);
    while( n-- > 0 ) {
	skip_arg();
    }
    return OTHER;
}


TOKEN ItemProc( char *cmd ) {
    char tag[32];
    
    newline();
    GProc( cmd );
    if ( strcmp(curr_env->name, "enumerate") == 0 ) {
	sprintf(tag, "<LI%dEnumerate>", list_level);
    }
    else if (strcmp(curr_env->name, "itemize") == 0)
	sprintf(tag, "<LI%dBullet>", list_level);
    else
	sprintf(tag, "<LI%dLabel>", list_level);
    register_tag(tag);
    match_opt_arg();
    backup_white_space();
    appendstring("\\t");
    firstitem = FALSE;
    return OTHER;
}


TOKEN ListProc( char *cmd ) {
    if (strcmp(cmd, "+") == 0) list_level++;
    else if (strcmp(cmd, "-") == 0) list_level--;
    return OTHER;
}


TOKEN DocStyleProc( char *dummy ) {
    register int i;
    ACTIONPTR meptr;
    int ind = 0;
    char *code1, *code2, *str;

    SKIPWSPACE();
    /* check optional arguments */
    if (ch == '[') {
	readch();
	for( ; ; ) {
	    if (ch == ']' || ch == EOF) break;
	    if (ch == ',') {
		readch();  continue;
	    }
	    i = 0;
	    do {
	    	if (i < MAXLEN) arg_buffer[i++] = ch;
	    	readch();
	    } while(ch != ',' && ch != ']' && ch != EOF);
	    arg_buffer[i] = '\0';
	    if (strcmp(arg_buffer, "german") == 0)
		german = TRUE;
	}
	if (ch == ']') readch();
    }

    /* process the style argument */
    meptr = get_token();		/* { */
    meptr = get_token();		/* style */
    if (strcmp(docstyle, "") == 0)
      docstyle = tokenbuffer;
    if (strcmp(docstyle, "article") == 0) {
      while ((code1 = commands[ind].code) != NULL) {
	if ((str = strstr(code1, "Heading")) != NULL) {
	  str--;
	  if (isdigit(*str)) {
	    code2 = (char *)malloc(strlen(code1));	/* copy string */
	    strcpy(code2,code1);
	    str = strstr(code2, "Heading");
	    str--;
	    *str = (char) (*str)--;
	    commands[ind].code = code2;
	  }
	}
	ind++;
      }
    }
    meptr = get_token();		/* } */
    return OTHER;
}

TOKEN AccentProc( char *code ) {
    ACTIONPTR meptr;
    char c, base, *cp, buff[32];
    char *trailer=NULL;
    BOOL brace = FALSE, bad = FALSE;

    c = code[0];
    if (curr_env == tabbing_env &&
		(c == '=' || c == '`' || c == '\'')) {
	GProc("%!");
	return OTHER;
    }
    if (c == '?') {	/* \a form of command in tabbing environment */
	c = ch;  readch();
    }
    meptr = get_token();
    cp = meptr->cmd;
    if (cp[0] == '{' && cp[1] == '\0') {
	brace = TRUE;
	meptr = get_token();
	cp = meptr->cmd;
    }
    if (strcmp(cp,"\\imath") == 0 ||
		strcmp(cp,"\\i") == 0) {
	base = 'i';
    } else if (strcmp(cp,"\\jmath") == 0 ||
		strcmp(cp,"\\j") == 0) {
	base = 'j';
    } else {
	if (meptr != &other) { bad = TRUE;  goto EXIT; }
	if (isalpha(tokenbuffer[0])) {
	    base = tokenbuffer[0];
	    trailer = &tokenbuffer[1];	/* trailing letters */
	}
	else goto NOTRANS;
    }
    for( meptr = accent_tab;  meptr->cmd != NULL;  meptr++ ) {
	if (meptr->cmd[0] == base && meptr->cmd[1] == c) {
	    GProc(meptr->code);
	    if (trailer != NULL && trailer[0] != '\0')
		GProc(trailer);
	    goto EXIT;
	}
    }
    /* no suitable translation found */
NOTRANS:
    (void)sprintf(buff, "%%fT\\\\%c{%s%%fT}%%f0", c, tokenbuffer); 
    GProc(buff);
EXIT:
    if (brace) {
	meptr = get_token();
	cp = meptr->cmd;
	if (cp[0] != '}' || cp[1] != '\0')
	    bad = TRUE;
    }
    if (bad) {
	LN();  FPR(stderr, "bad use of accent command \\%c{..}\n", c);
    }
    return OTHER;
}

void read_index_entry() {
    ACTIONPTR meptr;
    int quote = 0;
    int quotech;
    
    meptr = get_token();
    while ((meptr = get_token()), strcmp(tokenbuffer, "}") != 0) {
      if (strcmp(tokenbuffer, "\n") == 0)
	strcpy(tokenbuffer, " ");
      quotech = (strcmp(tokenbuffer, "\"") == 0);
      if (quote) {
	quote = 0;
	if (quotech)
	  appendstring("\"\"");
	else
	  appendstring(tokenbuffer);
      }
      else {
	if (quotech)
	  quote = 1;
	else
	  if (strcmp(tokenbuffer, "!") == 0)
	    appendstring(":");
	  else
	    appendstring(tokenbuffer);
      }
    }
}

TOKEN IndexProc (char *str) {
  int mtype;
  char *mfont;
  char mstr[64];
  
  switch (str[0]) {
    case 'a':
    case 'n':
    case 's':
      mtype = 14; break;
    case 'c':
      mtype = 13; break;
    case 'e':
      mtype = 12; break;
    case 'm':
      mtype = 11; break;
    case ' ':
    case 'o':
    default:
      mtype = 2; break;
  }
  switch (str[1]) {
    case 'b':
      mfont = "\\<Bold\\>"; break;
    case 'i':
      mfont = "\\<Italic\\>"; break;
    case ' ':
    case 'r':
    default:
      mfont = ""; break;
  }
  sprintf(mstr, "<Marker <MType %d> <MText `%s", mtype, mfont);
  appendstring(mstr);
  switch (str[0]) {
    case 'a':
      match_one_unit();
      appendstring("\\<Italic\\>see also\\<Default Para Font\\> ");
      match_one_unit();
      break;
    case 'n':
      match_one_unit();
      appendstring("\\<Italic\\> (");
      match_one_unit();
      appendstring(")");
      break;
    case 's':
      match_one_unit();
      appendstring(", \\<Italic\\>see \\<Default Para Font\\> ");
      match_one_unit();
      break;
    default:
      read_index_entry();
      break;
  }
  appendstring("'> >");
  return OTHER;
}

TOKEN CaptionProc( char *dummy ) {
    if (strcmp(curr_env->name, "figure") == 0  ||
	strcmp(curr_env->name, "figure*") == 0)
	register_tag(in_appendix ? "<CAFigure>" : "<CMFigure>");
    else  /* we need some tag, so no point in further testing */
	register_tag(in_appendix ? "<CATable>" : "<CMTable>");
    read_index_entry();
    return OTHER;
}


TOKEN DelimiterProc( char *str ) {
    switch( str[0] ) {
    case '\\':
	switch( str[1] ) {
	case '[':		/* \[ command */
	    return LSQCMD;
	case ']':		/* \] command */
	    return RSQCMD;
	case '(':		/* \( command */
	    return LPARCMD;
	case ')':		/* \) command */
	    return RPARCMD;
	}
	break;
    case '[':
	return LSQ;
    case ']':
	return RSQ;
    case '{':		/* { command */
	return LBRACE;
    case '}':		/* } command */
	return RBRACE;
    case '$':		/* $ or $$ command */
	if (str[1] != '$')
	    return (curr_matching_mode == MATH)? DOLLAR_END : DOLLAR_START;
	else
	    return (curr_matching_mode ==MATH)? DOLLARDOLLAR_END : DOLLARDOLLAR_START;
    case 'e':		/* \end command */
	return ENDCMD;
    case 'E':		/* end-of-file */
	return EOFCMD;
    }
    return OTHER;
}


/*  Handles \input and \include commands.
    Note: the first argument is unused, it is provided in case
    we ever decide to support the \includeonly feature of LaTeX.  */
TOKEN ReadProc( char *cmd ) {
    char *s, *fn;
    int ln;
    FILE *saved_stream = f;

    s = get_string_arg();
    if (s == NULL) return OTHER;
    ln = strlen(s);
    /* force filename to have the .tex extension */
    if (ln <= 4 || strcmp(&(s[ln-4]), ".tex") != 0) {
	fn = malloc(ln+5);
	MCHECK(fn);
	strcpy(fn, s);
	strcat(fn, ".tex");
    } else {
	fn = malloc(ln+1);
	MCHECK(fn);
	strcpy(fn, s);
    }
    ln = linenum;
    s = filename;
    f = locateTexFile(fn);
    if (f == NULL) {
	LN();  FPR(stderr, "Unable to find file: %s\n", fn);
	numreports++;
    } else {
	checkFile(fn);
	fclose(f);
    }
    filename = s;
    f = saved_stream;
    linenum = ln;
    ch = ' ';		/* bug fix; 12/1/93 */
    free(fn);
    return OTHER;
}

/*  Handles \appendix command by changing M(ain) headings to A(ppendix).
    Note: the first argument is unused.  */
TOKEN AppendixProc( char *cmd ) {
    int ind = 0;
    char *code1, *code2, *str;

    in_appendix = 1;
    while ((code1 = commands[ind].code) != NULL) {
      if ((str = strstr(code1, "Heading")) != NULL) {
        str--; str--;
	if (*str == 'M') {
	  code2 = (char *)malloc(strlen(code1));	/* copy string */
	  strcpy(code2,code1);
	  str = strstr(code2, "Heading");
	  str--; str--;
	  *str = 'A';
	  commands[ind].code = code2;
	}
      }
      ind++;
    }
    return OTHER;
}

