/* la_mml.c */

/*
 * Copyright (c) 1991-1993  R. Nigel Horspool.
 * All rights reserved.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "tables.h"
#include "lexer.h"
#include "output.h"
#include "auxprocs.h"
#include "la_mml.h"



FILE *outputfile;		/* descriptor for output */
char *outputfilename;

int numreports = 0;
BOOL unknowns = FALSE;		/* if true, report "\xxx" commands
				   that are not in the tables (really
				   a debugging option) */

TEXTMODE curr_matching_mode = PARA;
char *paratag = NULL;



void OOPS( char *msg ) {
    if (msg != NULL) perror(msg);
    exit(1);
}


/* Print the current filename and linenumber --
   used just before printing an error message.	*/
void LN() {
    if (!filenameprinted) {
	FPR(stderr, "File %s:\n", filename);
	filenameprinted = TRUE;
    }
    FPR(stderr, "%4d:\t", linenum);
    numreports++;
}


/*  Returns the name of the current text processing mode  */
static char *modename( TEXTMODE mode ) {
    switch( mode ) {
    case MATH:		return "math";
    case PARA:		return "para or lr";
    case MATHARRAY:	return "math array";
    case TABLE:		return "table";
    case VERBATIM:	return "verbatim";
    }
    return "??";
}


void match_opt_arg() {
    SKIPWSPACE();
    if (ch == '[') {
	readch();
	match_grouping(RSQ);
    }
}


void handle_environment( ENVPTR newenv, TOKEN terminator ) {
    ENVPTR prev_env = curr_env;
    TEXTMODE prev_mode = curr_matching_mode;
    BOOL save_first = firstitem;
    ACTIONPTR meptr;
    int ln = linenum;
    char *savedfamily = nextfamily;
    short savedfont = nextfont;
    int savedsize = nextsize;

    curr_env = newenv;
    GProc(newenv->startcmd);
    curr_matching_mode = newenv->kind;
    if (newenv->kind == VERBATIM) {
	for( ; ; ) {
	    readch();
	    while ( ch == '\\') {
		static char *endtext = "\\end{verbatim}";
		register int i, j;

		for( i = 0;  ch == endtext[i];  i++ )
		    readch();
		/* KJT 29/03/08: cast "strlen" result as int to avoid warning */
		if (i >= (int) strlen(endtext)) goto EXIT;
		for( j = 0;  j < i;  j++ )
		    copychar(endtext[j]);
	    }
	    if (ch == EOF) {
		linenum = ln;
		LN();
		FPR(stderr, "unclosed \\begin{verbatim} command\n");
		goto EXIT;
	    }
	    if (ch == '\n')
		newline();
	    else if (ch != ' ')
		copychar(ch);
	    else
		appendstring("\\x11  ");
	}
    } else {
	if (terminator == RBRACE) {	/* special case! */
	    meptr = get_token();
	    if (GProc(meptr->code) != LBRACE) goto EXIT;
	}
	match_grouping(terminator);
    }

EXIT:
    firstitem = save_first;
    GProc( newenv->endcmd );
    nextfamily = savedfamily;  nextfont = savedfont;  nextsize = savedsize;
    if (newenv->conttag != NULL)
	register_tag(prev_env->conttag);
    curr_matching_mode = prev_mode;
    curr_env = prev_env;
}


void match_one_unit() {
    ACTIONPTR meptr;

    meptr = get_token();
    if (GProc(meptr->code) == LBRACE)
	match_grouping(RBRACE);
}


void skip_arg() {
    ACTIONPTR meptr;
    register int nesting = 0;
    register char *cp;

    do {
	meptr = get_token();
	cp = meptr->cmd;
	if (cp[0] == '{' && cp[1] == '\0')
	    nesting++;
	else if (cp[0] == '}' && cp[1] == '\0')
	    nesting--;
    } while( nesting > 0 );
}



void match_grouping( TOKEN terminator ) {
    TOKEN cmd;
    int ln = linenum;
    ACTIONPTR meptr;
    ENVPTR ep;
    short savefont = nextfont;
    short savesize = nextsize;
    char *savefamily = nextfamily;

    for( ; ; ) {
	meptr = (ACTIONPTR)get_token();
	cmd = GProc(meptr->code) ;

	switch( cmd ) {

	case LSQCMD:
	    handle_environment(displaymath_env, RSQCMD);
	    continue;

	case LPARCMD:
	    handle_environment(math_env, RPARCMD);
	    continue;

	case LBRACE:
	    match_grouping(RBRACE);
	    continue;

	case DOLLAR_START:
	    handle_environment(math_env, DOLLAR_END);
	    continue;

	case DOLLARDOLLAR_START:
	    handle_environment(displaymath_env, DOLLARDOLLAR_END);
	    continue;

	case OTHER:
	    continue;

	}

	if (cmd == terminator)
	    goto EXIT;

	if (cmd == RSQ) {
	    copychar(']');
	    continue;
	}

	/*  all correct tokens have been handled and
	    control returned to the top of the loop
	    via a continue.  If control reaches here,
	    there is an error.  */
	if (cmd == EOFCMD) {
	    linenum = ln;
	    LN();
	    FPR(stderr, "construct unclosed when end-of-file reached\n");
	    goto EXIT;
	}
	switch( cmd ) {
	case EOFCMD:
	case RSQCMD:
	case RPARCMD:
	case ENDCMD:
	case RBRACE:
	case DOLLAR_END:
	case DOLLARDOLLAR_END:
	case RSQ:
	    LN();
	    FPR(stderr, "%s mismatched with construct started on line %d\n",
		meptr->cmd, ln);
	    return;
	}
	LN();
	FPR(stderr, "inappropriate use of %s inside %s mode text\n",
	    meptr->cmd, modename(curr_matching_mode));
    }
EXIT:
    nextfont = savefont;  nextfamily = savefamily;  nextsize = savesize;
}


/*  Processes the current input stream  */
void checkFile( char *printName ) {
    BOOL saved_fnp = filenameprinted;

    filename = printName;
    linenum = 0;  ch = '\n';
    filenameprinted = FALSE;
    while(ch != EOF)
	match_grouping(EOFCMD);
    filenameprinted = (filenameprinted)? FALSE : saved_fnp;
}
