/* output.c */

/*
 * Copyright (c) 1991-1993  R. Nigel Horspool.
 * All rights reserved.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
*/

#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "tables.h"
#include "lexer.h"
#include "la_mml.h"
#include "auxprocs.h"
#include "output.h"

#define MAXBUFSIZE	8192
char outputbuf[MAXBUFSIZE+1];
char *obptr = outputbuf;
char *buflimit = outputbuf + MAXBUFSIZE;
int discard_output = 0;

#define store_in_buffer(ch)	if (obptr < buflimit) *obptr++ = ch;	\
				else overflow = TRUE


/* currentfamily == font family (Times/Symbol/Courier/Helvetica) of last text
	to have been sent to the output file;
   nextfamily == font family to use for the next text to be sent.

   Similarly for font styles (which are plain,bold,italic, etc.) and
   point sizes.
*/
char *nextfamily, *currentfamily;
short int nextfont, currentfont;
short int nextsize, currentsize;
short int documentsize = 10;

static char *prevtag;
static int lastch = '\0';

static BOOL overflow = FALSE;


static void divert_to_buffer(int,BOOL), insert_from_buffer(int);
static int bitcnt(int);


static void bufcat( register char *s ) {
    for( ;  *s != '\0';  s++ )
	store_in_buffer(*s);
    lastch = '\0';
}


static PARAFORMAT *get_para_format( char *tag ) {
    PARAFORMAT *pfp;

    if (tag == NULL) return NULL;
    for( pfp = paratable;  pfp->tagname != NULL;  pfp++ ) {
	if (strcmp(pfp->tagname,tag) == 0) {
	    pfp->used = TRUE;
	    return pfp;
	}
    }
    LN();  FPR(stderr, "paragraph tag %s not registered\n", tag);
    return NULL;
}


static CHARFORMAT *get_char_format( PARAFORMAT *pfp ) {
    register CHARFORMAT *chfp;

    if (pfp == NULL) return NULL;
    for( chfp = chfmttable;  chfp->name != NULL; chfp++ ) {
	if (chfp->name == pfp->font) {
	    chfp->used = TRUE;
	    return chfp;
	}
    }
    LN();  FPR(stderr, "char tag %s not registered\n", pfp->font);
    return NULL;
}


void register_tag( char *tag ) {
    register PARAFORMAT *pfp;
    register CHARFORMAT *chfp;

    pfp = get_para_format(tag);
    if (pfp == NULL) return;
    prevtag = paratag = pfp->tagname;
    chfp = get_char_format(pfp);
    if (chfp == NULL) return;
    currentfont = nextfont = chfp->style;
    currentfamily = nextfamily = chfp->family;
    currentsize = nextsize = chfp->size;
}


/* routine needed by \Large and similar commands */
static void set_font_size( char *tag ) {
    register CHARFORMAT *chfp;

    nextfamily = TIMESFAMILY;  nextfont = PLAIN;
    for( chfp = chfmttable;  chfp->name != NULL; chfp++ ) {
	if (strcmp(chfp->name,tag) == 0) {
	    nextsize = chfp->size;
	    return;
	}
    }
    LN();  FPR(stderr, "bad char format tag %s\n", tag);
}


static void check_font_and_family() {
    register CHARFORMAT *chfp;
    CHARFORMAT *altfp = NULL;
    int best;

    if (nextfamily == currentfamily && nextfont == currentfont
	&& nextsize == currentsize) return;

    /* look for a suitable character format tag */
    best = 0;
    for( chfp = chfmttable;  chfp->name != NULL; chfp++ ) {
	int similarity = 0;
	if (nextfamily == chfp->family) similarity++;
	if (nextfont == chfp->style) similarity++;
	if (nextsize == chfp->size) similarity++;
	if (similarity > best) {
	    best = similarity;
	    altfp = chfp;
	    if (similarity == 3) break;
	}
    }
    chfp = altfp;
    if (chfp != NULL) {	/* output the tag */
	bufcat(chfp->name);
	currentfamily = chfp->family;
	currentfont = chfp->style;
	currentsize = chfp->size;
	chfp->used = TRUE;
    }
    /* now make adjustments as required */
    if (nextfamily != currentfamily) {
	bufcat(nextfamily);
	currentfamily = nextfamily;
    }
    if (nextfont != currentfont) {
	short diffs = (nextfont ^ currentfont) & STYLEMASK;
	if (bitcnt(diffs) > bitcnt(nextfont & STYLEMASK)) {
	    bufcat("<Plain>");
	    diffs = nextfont & STYLEMASK;
	}
	if (diffs & ITALIC)
	    bufcat(nextfont & ITALIC? "<Italic>" : "<NoItalic>");
	if (diffs & BOLD)
	    bufcat(nextfont & BOLD? "<Bold>" : "<NoBold>");
	if (diffs & UNDERLINE)
	    bufcat(nextfont & UNDERLINE? "<Underline>" : "<NoUnderline>");
	if (diffs & STRIKE)
	    bufcat(nextfont & STRIKE? "<Strike>" : "<NoStrike>");
	if (diffs & OBLIQUE)
	    bufcat(nextfont & OBLIQUE? "<Oblique>" : "<NoOblique>");
	if ((nextfont ^ currentfont) & SCRIPTMASK) {
	    if (nextfont & SUBSCRIPT)
		bufcat("<Subscript>");
	    else if (nextfont & SUPERSCRIPT)
		bufcat("<Superscript>");
	    else
		bufcat("<Normal>");
	}
	currentfont = nextfont;
    }
    if (nextsize != currentsize) {
	char buff[16];
	sprintf(buff, "<pts %d>", nextsize);
	bufcat(buff);
	currentsize = nextsize;
    }
}


void copychar( char ch ) {
    if (discard_output > 0) return;
    if (ch == ' ' && curr_matching_mode != VERBATIM) {
	if (obptr == outputbuf) return;
	if (lastch == ' ' && obptr >= outputbuf+6) {
	    if (obptr[-1] == ' ' && obptr[-4] == 'x' && obptr[-5] == '\\'
			&& obptr[-6] != '\\')
		return;
	}
    }
    check_font_and_family();
    lastch = ch;
    if (ch == '<' || ch == '>' || ch == '\\')
	store_in_buffer('\\');
    store_in_buffer(ch);
}


void copystring( register char *str ) {
    while(*str != '\0')
	copychar( *str++ );
}


void appendchar( char ch ) {
    if (discard_output > 0) return;
    check_font_and_family();
    store_in_buffer(ch);
    lastch = '\0';
}


void appendstring( char *str ) {
    if (discard_output > 0) return;
    check_font_and_family();
    while( *str != '\0' )
	store_in_buffer(*str++);
    lastch = '\0';
}

void backup_white_space() {
    while(obptr > outputbuf && obptr[-1] == ' ') {
	if (obptr > outputbuf+5 && obptr[-5] == '\\' && obptr[-4] == 'x')
	    break;
	obptr--;
    }
}


void newline() {
    register char *cp;
    static int nlcnt = 0;

    if (discard_output > 0 && obptr == outputbuf) return;
    *obptr = '\0';
    if (curr_env->kind == VERBATIM) {
	FPR(OF, "%s%s\n", paratag == NULL? "\n" : paratag, outputbuf);
	goto EXIT;
    }
    for( cp = outputbuf;  ;  cp++ ) {
	if (*cp == '\0') {	/* an empty line */
	    if (nlcnt == 0 && paratag == NULL && curr_env->nlmode == 'N')
		register_tag(curr_env->conttag);
	    nlcnt++;
	    goto EXIT;
	}
	if (!isspace(*cp)) break;
    }
    nlcnt = 0;
    if (paratag != NULL) {
	FPR(OF, "\n%s\n", paratag);
	prevtag = paratag;
	paratag = NULL;
    }
    FPR(OF, "%s\n", cp, OF );
    lastch = '\0';
    if (curr_env->nlmode == 'P')
	register_tag(curr_env->conttag);
EXIT:
    obptr = outputbuf;
    if (overflow) {
	LN();  FPR(stderr, "The output buffer overflowed\n");
	overflow = FALSE;
    }
}


/* format controls handled by GProc:
	%b	interpolate tokenbuffer in current font & style
		(If tokenbuffer holds alphabetic characters and current mode
		is MATH, the standard font becomes TIMES-ITALIC.)
	%fX	switch to font X; X = A for slant (\sl), B for bold (\bf),
		C for courier (\tt), I for italic (\em or \it),
		S for symbol, T for Times (\rm), V for helvetica (\sf),
		0 to revert to font/family/size at beginning of format string
	%n	terminate current output line, equiv. to \n in input
	%o	discard optional argument, but still interpret it
	%p	interpolate next argument from the input in PARA mode
	%sX	switch to style X; X = H for superscript, L for subscript,
		N for normal (not sub/super),
		U for underline, P for plain (not underlined).
	%x	discard next argument from the input, but still interpret it
	%z	discard next argument from the input, without interpreting it
	%Dn	divert output to buffer number n (1 <= n <= 9)
	%D+n	append output to buffer number n (1 <= n <= 9)
	%D0	restore output to the output file
	%In	insert contents of buffer n (1 <= n <= 9)
	%P<c>	set size as for character tag <c>, but family=Times, style=Plain
	%T	set paragraph tag to conttag of current environment
	%X	kill current output line (\kill)
	%space	ignore white space in input
	%\b	remove trailing blanks in output buffer
	%&	set paratag to previous paratag, as needed for \\ command
	%#	insert today's date into the output
	%*	discard '*' in input
	%+	suppress mml output (this construct is nestable)
	%-	re-enable mml output
	%!	reset font & family back to standard defaults for environment
	%?X	verify that mode == TABLE/MATHARRAY if X=a,
		MATH/MATHARRAY if x=m,  !(MATH/MATHARRAY) if X=p
	%=X	force mode to PARA if X=p, MATH if X=m, TABLE if X=t, ...
	%@	revert to para mode at beginning of format string
	%(X)PP	call a special handling function, passing parameter string PP.
		Possibilities for X are:
			A AccentProc		B IndexProc
			D DelimiterProc		E EnvProc
			F DefProc		I ItemProc
			L ListProc
			N NewEnvProc		R ReadProc
			S VSpaceProc		V VerbProc
			X AppendixProc		Z DocStyleProc
	%<ttt>	set paratag to <ttt>, and currentfont/currentfamily/currentsize
		to defaults for this paragraph type.
	%%	output a %
*/
TOKEN GProc( register char *fmt ) {
    char *tp, tag[64];
    char *savefamily = nextfamily;
    short savefont = nextfont;
    short savesize = nextsize;
    ENVPTR saveenv = curr_env;
    TEXTMODE savemode = curr_matching_mode;
    static time_t now;

    for( ;  *fmt != '\0'; fmt++ ) {
	if (*fmt == '%') {
	    switch( *++fmt ) {
	    case 'b':
		if ((curr_matching_mode == MATH || curr_matching_mode == MATHARRAY)
			&& isalpha(tokenbuffer[0])) {
		    nextfont |= ITALIC;  nextfamily = TIMESFAMILY;
		}
		copystring(tokenbuffer);
		nextfont = savefont;  nextfamily = savefamily;
		break;
	    case 'f':
		switch( *++fmt ) {
		case 'A':  nextfamily = TIMESFAMILY; nextfont = OBLIQUE;  break;
		case 'B':  nextfamily = TIMESFAMILY; nextfont = BOLD;  break;
		case 'C':  nextfamily = COURIERFAMILY; nextfont = PLAIN;  break;
		case 'I':  nextfamily = TIMESFAMILY; nextfont = ITALIC;  break;
		case 'S':  nextfamily = SYMBOLFAMILY; nextfont = PLAIN;  break;
		case 'T':  nextfamily = TIMESFAMILY; nextfont = PLAIN;  break;
		case 'V':  nextfamily = HELVETICAFAMILY; nextfont = PLAIN;  break;
		case '0':  nextfont = savefont;  nextfamily = savefamily;
			   nextsize = savesize;  break;
		default:   LN(); FPR(stderr, "bad font code %c\n", *fmt);
		}
		break;
	    case 'n':
		newline();
		break;
	    case 'o':
		discard_output++;
		match_opt_arg();
		discard_output--;
		break;
	    case 'p':
		match_one_unit();
		break;
	    case 's':
		switch( *++fmt ) {
		case 'H':
		    nextfont &= (~SCRIPTMASK);  nextfont |= SUPERSCRIPT;  break;
		case 'L':
		    nextfont &= (~SCRIPTMASK);  nextfont |= SUBSCRIPT;  break;
		case 'N':
		    nextfont &= (~SCRIPTMASK);  break;
		case 'P':
		    nextfont = PLAIN;  break;
		case 'U':
		    nextfont |= UNDERLINE;  break;
		default:
		    LN(); FPR(stderr, "unknown style code %c\n", *fmt);
		}
		break;
	    case 'x':
		discard_output++;
		match_one_unit();
		discard_output--;
		break;
	    case 'z':
		skip_arg();
		break;
	    case 'D':
		if (*++fmt == '+')
		    divert_to_buffer(*++fmt - '0', TRUE);
		else
		    divert_to_buffer(*fmt - '0', FALSE);
		break;
	    case 'I':
		insert_from_buffer( *++fmt - '0' );
		break;
	    case 'P':
		for( fmt++,tp = tag;  (*tp++ = *fmt) != '>';  fmt++ )
		    ;
		*tp = '\0';
		set_font_size(tag);
		break;
	    case 'T':
		newline();
		register_tag(curr_env->conttag);
		break;
	    case 'X':
		obptr = outputbuf;
		break;
	    case '\b':
		backup_white_space();
		break;
	    case ' ':
		while(isspace(ch)) readch();
		break;
	    case '&':
		if (curr_env->slmode == 'P') {
		    register_tag(prevtag);
		    if (curr_env->kind != TABLE) {
			/* preserve current font for next line */
			nextfont = savefont;
			nextfamily = savefamily;
			nextsize = savesize;
		    }
		    /*	Note: \\ in table environments resets
			the font */
		} else if (curr_env->slmode == 'H')
		    appendstring("\\x09 \n");	/* a HardReturn */
		break;
	    case '#':
		time(&now);
		tp = asctime(localtime(&now));
		strcpy(tag, " %fI");
		strcat(tag, tp);
		for( tp=tag;  *tp!='\0' && *tp!='\n';  tp++ )
		    ;
		strcpy(tp, "%f0 ");
		GProc(tag);
		break;
	    case '*':
		if (ch == '*') readch();
		break;
	    case '<':
		newline();
		for( tp = tag;  (*tp++ = *fmt) != '>';  fmt++ )
		    ;
		*tp = '\0';
		register_tag(tag);
		break;
	    case '+':
		discard_output++;
		break;
	    case '?':
		switch( *++fmt ) {
		case 'a':
		    if (curr_matching_mode != MATHARRAY && curr_matching_mode != TABLE) {
		      /* ignored to allow commands to include tables/tabbing
			LN();
			FPR(stderr, "%s used outside array or table environment\n",
				tokenbuffer);
		      */
		    }
		    break;
		case 'm':
		    if (curr_matching_mode != MATH && curr_matching_mode != MATHARRAY) {
			LN();
			FPR(stderr, "%s used outside math mode text\n", tokenbuffer);
		    }
		    break;
		case 'p':
		    if (curr_matching_mode == MATH || curr_matching_mode == MATHARRAY) {
			LN();
			FPR(stderr, "%s used inside math mode text\n", tokenbuffer);
		    }
		    break;
		default:
		    LN(); FPR(stderr, "unknown code %c for matching mode\n", *fmt);
		    break;
		}
		break;
	    case '=':
		switch( *++fmt ) {
		case 'p':  curr_matching_mode = PARA;  break;
		case 'm':  curr_matching_mode = MATH;  break;
		case 't':  curr_matching_mode = TABLE;  break;
		default:   LN();  FPR(stderr,
				"unsupported code %c for matching mode\n", *fmt);
			   break;
		}
		break;
	    case '@':
		curr_matching_mode = savemode;
		break;
	    case '!':
		{ PARAFORMAT *pfp;  CHARFORMAT *chfp;
		  pfp = get_para_format(curr_env->conttag);
		  if (pfp == NULL) break;
		  chfp = get_char_format(pfp);
		  if (chfp == NULL) break;
		  nextfont = chfp->style;  nextfamily = chfp->family;
		  nextsize = chfp->size;
		}
		break;
	    case '-':
		discard_output--;
		break;
	    case '(':
		switch( *++fmt ) {
		case 'A':	return AccentProc(fmt+2);
		case 'B':	return IndexProc(fmt+2);
		case 'C':	return CaptionProc(fmt+2);
		case 'D':	return DelimiterProc(fmt+2);
		case 'E':	return EnvProc(fmt+2);
		case 'F':	return DefProc(fmt+2);
		case 'I':	return ItemProc(fmt+2);
		case 'L':	return ListProc(fmt+2);
		case 'N':	return NewEnvProc(fmt+2);
		case 'R':	return ReadProc(fmt+2);
		case 'S':	return VSpaceProc(fmt+2);
		case 'V':	return VerbProc(fmt+2);
		case 'X':	return AppendixProc(fmt+2);
		case 'Z':	return DocStyleProc(fmt+2);
		}
		FPR(stderr, "Proc code %%(%c) unknown\n", *fmt);
		return OTHER;
	    default:
		FPR(stderr, "Control code %c unknown\n", *fmt);
		break;
	    }
	    continue;
	}
	appendchar(*fmt);
    }
    return OTHER;
}


/* counts number of bits set in argument n */
static int bitcnt( register int n ) {
    register int cnt = 0;
    while( n != 0 ) {
	n &= (n - 1);
	cnt++;
    }
    return cnt;
}


static FILE *savedOF = NULL;
static FILE *tempfile;

typedef struct dbuf {
	    long	start, end;
	    struct dbuf	*next;
	} FILEREC, *FILERECPTR;

#define MAXDLEVEL	16
static FILERECPTR currbuffptr = NULL;
static FILERECPTR outbuffstack[MAXDLEVEL], diversion[10];
int divlevel = 0;


static void divert_to_buffer( int n, BOOL append ) {
    if (n < 0 || n > 9) {
	LN(); FPR(stderr, "bad buffer number %d for %%D control\n", n);
	return;
    }
    if (currbuffptr != NULL)
	currbuffptr->end = ftell(OF);
    if (n == 0) {	/* close current diversion */
	if (divlevel <= 0) {
	    LN(); FPR(stderr, "unpaired use of %%D0\n");
	    return;
	}
	currbuffptr = outbuffstack[--divlevel];
	if (currbuffptr == NULL)	/* no longer a diversion in effect */
	    OF = savedOF;
	else {
	    currbuffptr->next = (FILERECPTR)malloc(sizeof(FILEREC));
	    currbuffptr = currbuffptr->next;
	    currbuffptr->start = ftell(OF);
	    currbuffptr->next = NULL;
	}
	return;
    }
    if (tempfile == NULL) {
	/* KJT 29/03/08: called "open_tempfile" instead of "tmpfile" */
	tempfile = open_tempfile();
	if (tempfile == NULL) OOPS("cannot create temp file");
	savedOF = OF;
    }
    if (divlevel >= MAXDLEVEL) {
	LN();  FPR(stderr, "diversion stack overflow\n");
	return;
    }
    OF = tempfile;
    fseek(OF, 0L, 2);	/* position to end of file */
    outbuffstack[divlevel++] = currbuffptr;
    currbuffptr = diversion[n];
    if (append && currbuffptr != NULL) {
	while(currbuffptr->next != NULL)
	    currbuffptr = currbuffptr->next;
	currbuffptr->next = (FILERECPTR)malloc(sizeof(FILEREC));
	currbuffptr = currbuffptr->next;
    } else
	if (currbuffptr == NULL) {
	    currbuffptr = (FILERECPTR)malloc(sizeof(FILEREC));
	    diversion[n] = currbuffptr;
	}
    currbuffptr->start = ftell(OF);
    currbuffptr->next = NULL;
}


static void insert_from_buffer( int n ) {
    register int cnt, ch;
    FILERECPTR tempptr;

    if (n < 1 || n > 9) {
	LN(); FPR(stderr, "bad buffer number %d for %%I control\n", n);
	return;
    }
    if (tempfile == NULL)  return;
    if (currbuffptr != NULL)
	currbuffptr->end = ftell(OF);
    for( tempptr = diversion[n];  tempptr != NULL;  tempptr = tempptr->next) {
	cnt = tempptr->end - tempptr->start;
	if (cnt <= 0) continue;
	fseek(tempfile, tempptr->start, 0);
	while(cnt-- > 0) {
	    ch = getc(tempfile);
	    putc(ch,OF);
	}
    }
}

/* KJT 29/03/08:
   added "open_tempfile" as "tmpfile" tries to write to the root directory on
   Windows
*/

FILE* open_tempfile() {
    char* name = (char *) tempnam(NULL,"la_mml");
    FILE* tempfile = fopen(name, "wb+");
    return tempfile;
}

void close_tempfile() {
    if (tempfile == NULL) return;
    fclose(tempfile);
    tempfile = NULL;
    currbuffptr = NULL;
    divlevel = 0;
}

