%{

/*****************************************************************

lottex.l	K. J. Turner	11/05/02

This "lex" program reads LOTOS text (which is assumed to be syntactically
valid) from the standard input, and writes it to the standard output in a
form suitable for processing by LaTeX. The flag "-n" causes output lines to
be numbered at the right.

The program translates initial tab and space characters on a line into LaTeX
horizontal space (\hspace). This assumes that indentation will be by small
(e.g. 2) units of space. Because LaTeX limits the number of tab stops in a
"tabbing" environment to around a dozen, it is not possible to have tab
stops every space or two.

Tabs and multiple spaces embedded in a line are translated into LaTeX tab
commands (\>). This assumes that embedded tabs and multiple spaces are
likely to be used to format comments and the like, for which a coarser
degree of control over spacing is acceptable.

Since the output is assumed to be embedded in a LaTeX "tabbing" environment,
a LaTeX newline command (\\) is appended to each line.

LOTOS keywords are converted to bold (\bf). Keywords are recognised only in
lower case or with capitalisation of each word (e.g. "OfSort", "RenamedBy").
LaTeX special characters are escaped. Special characters are used for some
symbols; this requires the "lotos" environment.

The program recognises "(*" as beginning comments and "*)" as ending
comments.  Comments may not be nested. LOTOS keywords are not emboldened
inside comments.

*****************************************************************/

#define Tab 8		/* number of characters between tab positions */

#define Spc 0.5		/* number "em"s for horizontal space unit */

#define Prog "lottex"	/* program name */

int IpPos = 0;		/* current input position (0 = first) */

int OpPos = 0;		/* current output position (0 = first) */

int TabPos = 0;		/* current tab position (0 = first) */

int InComment = 0;	/* indicates whether inside a comment (1
			   = yes) */
int LineNum = 0;	/* current line number - line numbering off by
			   default */

/* output current text */

void echo ();

/*  At the start of a line make good the difference between the input
    and output positions by using "\hspace". Otherwise, round the input
    position up to next tab stop, and output LaTeX tabs to bring the
    output position up to the input position. */

void Input () {
  int NumTabs;

  if (OpPos < IpPos) {				/* positions disagree? */
    if (OpPos == 0) {				/* start of line? */
      OpPos = IpPos;
      printf ("\\hspace{%.1fem}", Spc * IpPos);
    }
    else {
	NumTabs = (IpPos - 1) / Tab + 1;	/* next tab stop */
	OpPos = Tab * NumTabs;
	while (TabPos < NumTabs) {
	  printf ("\\>"); TabPos++;
	}
    }
  }
}

/* Output current identifier string, updating input and output
   positions. Escape special LaTeX characters as required. */

void OutputId () {
  int Ind;
  char Ch;

  for (Ind = 0; Ind < yyleng; Ind++) {
    Ch = yytext [Ind];
    if (Ch == '#' || Ch == '$' || Ch == '&' || Ch == '%' || Ch == '{' || Ch == '}') {
      putchar ('\\');
      putchar (Ch);
    }
    else if (Ch == '<' || Ch == '>' || Ch == '|' || ((Ch == '-') && !InComment)) {
      putchar ('$');
      putchar (Ch);
      putchar ('$');
    }
    else if (Ch == '~')
      printf ("$\\sim$");
    else if (Ch == '^')
      printf ("\\^{}");
    else if (Ch == '_')
      printf ("\\_\\,");
    else if (Ch == '\\')
      printf ("$\\backslash$");
    else putchar (Ch);
  }
  IpPos += yyleng;
  OpPos += yyleng;
}

/* Output current symbol string, updating input and output positions.
   Escape special LaTeX characters as required, and use special symbols
   where necessary. */

void OutputSym () {
  if (!strcmp (yytext, ":="))
    printf ("{\\bec}");
  else if (!strcmp (yytext, "=>"))
    printf ("{\\imp}");
  else if (!strcmp (yytext, ">>"))
    printf ("{\\ena}");
  else if (!strcmp (yytext, "[>"))
    printf ("{\\dis}");
  else if (!strcmp (yytext, "||"))
    printf ("$||$");
  else if (!strcmp (yytext, "|||"))
    printf ("$|||$");
  else if (!strcmp (yytext, "]|"))
    printf ("$]|$");
  else if (!strcmp (yytext, "|["))
    printf ("$|[$");
  else if (!strcmp (yytext, "[]"))
    printf ("{\\cho}");
  else if (!strcmp (yytext, "->"))
    printf ("{\\map}");
  else if (!strcmp (yytext, "_"))
    printf ("\\_\\,");
  else
    echo ();
  IpPos += yyleng;
  OpPos += yyleng;
}

main (int argc, char *argv[]) {
  if (argc > 1)
    if (argc == 2)
      if (!strcmp (argv[1], "-n"))
        LineNum = 1;				/* turn on line numbering */
      else
        printf ("%s: unknown option %s\n", Prog, argv[1]);
    else
      printf ("%s: too many arguments\n", Prog);
  yylex ();
}

%}

%p 3500						/* increase number of positions */

Comment		"(*"|"*)"

Symbol		"="|"=>"|":="|">>"|"[>"|"||"|"|||"|"|["|"]|"|"[]"|"->"|";"|","|":"|"["|"]"|"("|")"|"?"|"!"|"_"

Letter		[a-zA-Z]
Digit		[0-9]
NormChar	[a-zA-Z0-9]
SpecChar	[#%&*+-./<=>@\\\^~{}]
UnderScore	_

NormIdent	{Letter}(({NormChar}|{UnderScore})*{NormChar})?
SpecIdent	({SpecChar})+|({Digit}(({NormChar}|{UnderScore})*{NormChar})?)
Identifier	{NormIdent}|{SpecIdent}

%%

^" "+ |
" "" "+	IpPos += yyleng;

\t+ {
  IpPos = Tab * (IpPos / Tab + yyleng);
}

\n {
  IpPos = 0;
  OpPos = 0;
  TabPos = 0;
  if (LineNum > 0)
    printf ("\\`\\footnotesize%d\\\\\n", LineNum++);
  else
    printf ("\\\\\n");
}

{Comment} {
  Input ();
  InComment = !InComment;
  OutputId ();
}

{Symbol} {
  Input ();
  OutputSym ();
}

(accept|actualizedby|any|behavior|behaviour|choice|endlib)	|
(Accept|ActualizedBy|Any|Behavior|Behaviour|Choice|EndLib)	|
(endproc|endspec|endtype|eqns|exit|for|forall|formaleqns)	|
(EndProc|EndSpec|EndType|Eqns|Exit|For|ForAll|FormalEqns)	|
(formalopns|formalsorts|hide|i|in|is|let|library|noexit|of)	|
(FormalOpns|FormalSorts|Hide|I|In|Is|Let|Library|NoExit|Of)	|
(ofsort|opnnames|opns|par|process|renamedby|sortnames|sorts)	|
(OfSort|OpnNames|Opns|Par|Process|RenamedBy|SortNames|Sorts)	|
(specification|stop|type|using|where)				|
(Specification|Stop|Type|Using|Where) {
  Input ();
  if (!InComment) printf ("{\\bf ");
  OutputId ();
  if (!InComment) printf ("}");
}

{Identifier}|. {
  char Ch;
  if (yytext[yyleng-1] == '*') {
    Ch = input ();
    unput(Ch);
    if (Ch == ')') {
      REJECT;
    }
    else {
      Input ();
      OutputId ();
    }
  }
  else {
    Input ();
    OutputId ();
  }
}

%%

/* output current text */

void echo () {
  ECHO;
}

