%{

/*****************************************************************************

  "mifparse.y"	P. C. Hsu, K. J. Turner	30th November 2022

  This is a yacc program written to convert a FrameMaker document to a LaTeX
  document. The "mifch_la" utility is used at the end of the program to
  convert all special characters.

  The following information is lost in the translation:

    o page layout, including page numbering
    o the document date
    o frames, including figure contents and mathematics expressions
    o markers, including index entries
    o table formatting

  The following information is fixed in the translation:

    o cross-references
    o variable values

  The following information can be recreated by running LaTeX on the result:

    o section numbers
    o table of contents, list of figures, list of tables

  Version 1.0, K. J. Turner and Felicia P. C. Hsu, 27th September 1994

    o First public version

  Version 1.1, K. J. Turner, 4th May 1997

    o LaTeX2e document class commands added

  Version 1.2, K. J. Turner, 11th February 1998

    o "eurodate.sty" style file included in the distribution

  Version 1.3, K. J. Turner, 29th May 2000

    o made "res_char" static in "esc_reserved_char"

  Version 1.4, K. J. Turner, 30th November 2022

    o function prototypes defined for various external and interal functions to
    avoid implicit declarations

    o return types defined for "usage" and "main"

*****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <sys/types.h>

extern FILE *yyin, *yyout;
extern int yyparse();
extern int yylex (void);

/* KJT 30/11/22: added the following function prototypes */

extern void yyerror(char *s);
extern pid_t getpid();

void output_preamble(char trans_type);
void store_table(char store_field_type, char *store_field);
void end_para(int  end_block);
void retrieve_footnote(char *footnote_id);
void store_footnote(char store_field_type, char *store_field);
void convert_to_LaTeX_font_cmd(char *font_name);
void convert_to_LaTeX_para_cmd(char *para_name);
char *esc_reserved_char(char *text);
void retrieve_table(char *table_id);
void store_variable(char store_field_type, char *store_field);
void retrieve_variable(char *variable_id);
int strend(char *str1, char *str2);

int lineno;

#define PROG			"mif_la"

#define CAPTION_SIZE		256		/* max. caption */
#define FNOTE_SIZE		512		/* max. footnote */
#define RESCHAR_SIZE		4096		/* max. reserved chars */
#define TABLE_SIZE		4096		/* max. table */

#define MIFEXT1			".mif"
#define MIFEXT2			".framemif"
#define TEXEXT			".tex"

/* Codes for dealing with footnotes and tables */

#define STORE_DETAILS	'd'
#define STORE_ID	'i'
#define STORE_STR	's'
#define STORE_FONT	'f'
#define STORE_TAG	't'

/* Codes for end of selected paragraph formats */

#define END_OF_PGFCATALOG	1
#define END_OF_FONTCATALOG	2
#define END_OF_TEXTFLOW		3
#define END_OF_NOTES		4
#define END_OF_FNOTE		5
#define END_OF_PARA		6

char  trans_type = '0';		/* 'a'(rticle), 'b'(ook), 'r'(eport) */

int arr_id;
int level_cnt = 0;		/* level of list env. not terminated */

int pgf_found; 			/* paragraph tag found */
int font_found; 		/* character tag found */

int ignore_text = 1; 		/* paragraph text is to be ignored */

int in_caption = 0;		/* "TblTitleContent" paragraph found */
int in_notes = 0;		/* "Notes" paragraph found */
int in_para = 0;		/* "Para" paragraph found */
int in_symbol = 0;		/* "Symbol" font found */
int in_table = 0;		/* "Tbl" paragraph found */
int in_var = 0;			/* "VariableFormat" paragraph found */

int first_abstract = 1; 	/* first abstract paragraph not yet found */
int first_appendix = 1; 	/* first appendix not yet found */
int first_cell;   		/* first row cell not yet found */
int first_footnote = 1;   	/* first footnote not yet found */
int first_heading_para = 1;   	/* first heading's paragraph not yet found */
int first_para_line = 0;   	/* first paragraph's line not yet found */
int first_row;   		/* first table row not yet found */
int first_string = 1;   	/* first string not yet found */
int first_table = 1;   		/* first table not yet found */
int first_var = 1;   		/* first variable not yet found */

int is_figure;   		/* in LaTeX figure */
int is_table;   		/* in LaTeX table */
int num_cols;   		/* number of table columns */


char  *para_flag         = "";	/* current paragraph */
char  *para_end_tag      = "";	/* command to end the para */
char  *prev_list_tag     = "";	/* previous list env. */
char  *prev_para_name    = "";	/* previous paragraph */
char  *prev_ftag_end_str = "";	/* command to end previous font */

char  *curr_fnote_end_str;	/* command to end the footnote */
char  *curr_fnote_id;		/* current footnote id */
char  *curr_fnote_str;		/* current footnote text */

char  *curr_caption_str;	/* current table caption string */
char  *curr_caption_end_str;	/* command to end the caption */

char  *curr_table_end_str;	/* command to end the table */
char  *curr_table_id;		/* current table id */
char  *curr_table_str;		/* current table text */

char  *curr_var_id;		/* current variable id */
char  *curr_var_str;		/* current variable text */

char  list_table[15][15];	/* list env. not terminated yet */

struct id_str{   		/* identifier-string associative list */
  char *id;			/* for storing id */
  char *str;			/* for storing text */
  struct id_str *next;		/* pointer to next entry */
};

struct id_str *first_fnote_ptr = NULL;
struct id_str *curr_fnote_ptr  = NULL;
struct id_str *next_fnote_ptr  = NULL;

struct id_str *first_table_ptr = NULL;
struct id_str *curr_table_ptr  = NULL;
struct id_str *next_table_ptr  = NULL;

struct id_str *first_var_ptr = NULL;
struct id_str *curr_var_ptr  = NULL;
struct id_str *next_var_ptr  = NULL;

struct rep_para_struct {	/* for report format template */
       char *para_tag;		/* paragraph format found in FrameMaker */
       char *para_beg_str;	/* LaTeX command to begin paragraph format */
       char *para_end_str;	/* LaTeX command to end paragraph format */
} special_rep_para[] = {
  { "A1Heading",		"\n\\chapter{",			"}\n" },
  { "A2Heading",		"\n\\section{",			"}\n" },
  { "A3Heading",		"\n\\subsection{",		"}\n" },
  { "A4Heading",		"\n\\subsubsection{",		"}\n" },
  { "A5Heading",		"\n\\paragraph{",		"}\n" },
  { "M1Heading",		"\n\\chapter{",			"}\n" },
  { "M2Heading",		"\n\\section{",			"}\n" },
  { "M3Heading",		"\n\\subsection{",		"}\n" },
  { "M4Heading",		"\n\\subsubsection{",		"}\n" },
  { "M5Heading",		"\n\\paragraph{",		"}\n" },
  { "U1Heading",		"\n\\chapter*{",		"}\n" },
  { "U2Heading",		"\n\\section*{",		"}\n" },
  { "U3Heading",		"\n\\subsection*{",		"}\n" },
  { "U4Heading",		"\n\\subsubsection*{",		"}\n" },
  { "U5Heading",		"\n\\paragraph*{",		"}\n" },
  { "Abstract",			"\n\\begin{abstract}\n",	""    } ,
  { "AText",			"\n",				"\n"  },
  { "Author",			"\n\\author{",			"}\n" },
  { "B1Text",			"\n",				"\n"  },
  { "B2Text",			"\n",				"\n"  },
  { "Date",			"\n\\begin{document}\n\n\\maketitle\n",	""    },
  { "DText",			"",				""    },
  { "Footnote",			"\\footnote{",			"}\n" },
  { "Index Entries",		"",				""    },
  { "LI1Bullet",		"\n\\item\n",			"\n"  },
  { "LI1Enumerate",		"\n\\item\n",			"\n"  },
  { "LI1Label",			"\n\\item\n",			"\n"  },
  { "LI2Bullet",		"\n\\item\n",			"\n"  },
  { "LI2Enumerate",		"\n\\item\n",			"\n"  },
  { "LI2Label",			"\n\\item\n",			"\n"  },
  { "LI3Bullet",		"\n\\item\n",			"\n"  },
  { "LI3Enumerate",		"\n\\item\n",			"\n"  },
  { "LI3Label",			"\n\\item\n",			"\n"  },
  { "LI4Bullet",		"\n\\item\n",			"\n"  },
  { "LI4Enumerate",		"\n\\item\n",			"\n"  },
  { "LI4Label",			"\n\\item\n",			"\n"  },
  { "List of Figures",		"\n\\listoffigures\n",		""    },
  { "List of Tables",		"\n\\listoftables\n",		""    },
  { "LM1",			"\n",				"\n"  },
  { "LM2",			"\n",				"\n"  },
  { "LM3",			"\n",				"\n"  },
  { "LM4",			"\n",				"\n"  },
  { "LT",			"",				""    },
  { "NewPage",			"\n\\clearpage\n",		""    },
  { "Part",			"\n\\part{",			"}\n" },
  { "Program",			"\n\\program{\n",		"\n}\n" },
  { "Reference",		"\n\\reference{",		"}\n" },
  { "Table of Contents",	"\n\\tableofcontents\n",	""    },
  { "TFootnote",		"\\footnote{",			"}\n" },
  { "THeading",			"\\bf{",			"}\n" },
  { "Title",			"\n\\title{",			"}\n" },
  { "UPart",			"\n\\part*{",			"}\n" },
  { NULL,			NULL,				NULL  }
};

struct art_para_struct {	/* for article format template */
       char *para_tag;		/* paragraph format found in FrameMaker */
       char *para_beg_str;	/* LaTeX command to begin paragraph format */
       char *para_end_str;	/* LaTeX command to end paragraph format */
} special_art_para[] = {
  { "A1Heading",		"\n\\section{",			"}\n" },
  { "A2Heading",		"\n\\subsection{",		"}\n" },
  { "A3Heading",		"\n\\subsubsection{",		"}\n" },
  { "A4Heading",		"\n\\paragraph{",		"}\n" },
  { "A5Heading",		"\n\\subparagraph{",		"}\n" },
  { "M1Heading",		"\n\\section{",			"}\n" },
  { "M2Heading",		"\n\\subsection{",		"}\n" },
  { "M3Heading",		"\n\\subsubsection{",		"}\n" },
  { "M4Heading",		"\n\\paragraph{",		"}\n" },
  { "M5Heading",		"\n\\subparagraph{",		"}\n" },
  { "U1Heading",		"\n\\section*{",		"}\n" },
  { "U2Heading",		"\n\\subsection*{",		"}\n" },
  { "U3Heading",		"\n\\subsubsection*{",		"}\n" },
  { "U4Heading",		"\n\\paragraph*{",		"}\n" },
  { "U5Heading",		"\n\\subparagraph*{",		"}\n" },
  { NULL,			NULL,				NULL  }
};

struct bk_para_struct {		/* for book format template */
       char *para_tag;		/* paragraph format found in FrameMaker */
       char *para_beg_str;	/* LaTeX command to begin paragraph format */
       char *para_end_str;	/* LaTeX command to end paragraph format */
} special_bk_para[] = {
  { NULL,			NULL,				NULL  }
};

struct font_struct {
       char *font_tag;		/* character format defined in FrameMaker */
       char *font_in_tex;	/* equivalent LaTeX command */
       char *font_end_str;	/* LaTeX command to end character format */
} special_font[] = {
  { "Bold",		"{\\bf ",		"}"    },
  { "BoldItalic",	"{\\bi ",		"}"    },
  { "Emphasis",		"{\\em ",		"}"    },
  { "Footnote",		"{\\footnotesize ",	"}"    },
  { "Huge",		"{\\Huge ",		"}"    },
  { "huge",		"{\\huge ",		"}"    },
  { "LARGE",		"{\\LARGE ",		"}"    },
  { "Large",		"{\\Large ",		"}"    },
  { "large",		"{\\large ",		"}"    },
  { "Italic",		"{\\it ",		"}"    },
  { "Normal",		"{\\normalsize ",	"}"    },
  { "Roman",		"{\\rm ",		"}"    },
  { "Script",		"{\\scriptsize ",	"}"    },
  { "Small",		"{\\small ",		"}"    },
  { "Symbol",		"",			""     },
  { "Tiny",		"{\\tiny ",		"}"    },
  { "SansSerif",	"{\\sf ",		"}"    },
  { "Subscript",	"$_{",			"}$ "  },
  { "Superscript",	"$^{",			"}$ "  },
  { "None",		"",			""     },
  { "SymbolLarge",	"",			""     },
  { "SymbolSmall",	"",			""     },
  { NULL,		NULL,			NULL   }
};

%}

%union { /* to define the possible symbol types (used in yacc) */
  char *txt;
};

%token <txt> ATBL
%token <txt> CELL
%token <txt> CHAR
%token <txt> FNOTE
%token <txt> FONTCATALOG
%token <txt> FTAG
%token <txt> ID
%token <txt> IDENT
%token <txt> MIF
%token <txt> NOTES
%token <txt> NUM
%token <txt> PARA
%token <txt> PARALINE
%token <txt> PGFCATALOG
%token <txt> PGFTAG
%token <txt> ROW
%token <txt> STR
%token <txt> STRING
%token <txt> TBL
%token <txt> TBLID
%token <txt> TBLNUMCOLUMNS;
%token <txt> TBLTAG
%token <txt> TBLTITLECONTENT;
%token <txt> TEXTFLOW
%token <txt> VARIABLEFORMAT;
%token <txt> VARIABLEDEF;
%token <txt> VARIABLENAME;

%start mifs

%%

mifs:		 /* empty */
		| mifs mif
		;

mif:		mif_id
		| char
		| cell
		| fnote
		| fnote_id
		| fontcatalog
		| ftag
		| notes
		| pgfcatalog
		| textflow
		| para
		| paraline
		| pgftag
		| row
		| string
		| table
		| table_caption
		| table_cols
		| table_def
		| table_id
		| table_tag
		| var_def
		| var_fmt
		| var_name
		| other_statements
		;

char:		'<' CHAR IDENT {
		    if ((strcmp(para_flag, "textflow") == 0) &&
			!in_notes && in_para)
		      if (!ignore_text)
			fprintf(yyout, "\\%s ", $3);
		} '>'
		;

mif_id:		'<' MIF NUM '>' {
		  output_preamble(trans_type);
		}
		;

cell:		'<' CELL {
		  if (in_table && is_table && !first_cell)
		    store_table(STORE_STR, "	\\And ");
		  first_cell = 0;
		} mifs '>' {
		  strncat(curr_table_str, curr_table_end_str,
		    strlen(curr_table_end_str));
		  curr_table_end_str = "";
		}
		;

fnote:		'<' FNOTE mifs '>' {
		  end_para(END_OF_FNOTE);
		}
		| '<' FNOTE NUM {
		  if ((strcmp(para_flag, "textflow") == 0) &&
		      in_para && !in_notes) {
		      retrieve_footnote($3);
		    }
		} '>'
		;

fnote_id:	'<' ID NUM {
		  if ((strcmp(para_flag, "textflow") == 0) && in_notes)
		    store_footnote(STORE_ID, $3);
		} '>'
		;

fontcatalog:	'<' FONTCATALOG mifs '>' {
		    end_para(END_OF_FONTCATALOG);
		}
		;

ftag:		'<' FTAG STR {
		  if ($3 != "") {
		    if (strcmp(para_flag, "textflow") == 0) {
		      if (in_notes)
			store_footnote(STORE_FONT, $3);
		      else if (in_para && !ignore_text)
			convert_to_LaTeX_font_cmd($3);
		    }
		    else if (in_table)
		      store_table(STORE_FONT, $3);
		  }
		} '>'
		;

notes:		'<' NOTES {
		  in_notes = 1;
		} mifs '>' { end_para(END_OF_NOTES); }
		;

pgfcatalog:	'<' PGFCATALOG  mifs '>' {
		  end_para(END_OF_PGFCATALOG);
		}
		;

para:           '<' PARA {
		    if (strcmp(para_flag, "textflow") == 0) {
		      first_para_line = 1;
		      in_para = 1;
		    }
		} mifs '>' {
		  if (in_para)
		    end_para(END_OF_PARA);
		}
		;

paraline:	'<' PARALINE {
		  if (first_para_line) {
		    first_para_line = 0;
		    convert_to_LaTeX_para_cmd(prev_para_name);
		  }
		} mifs '>'
		;

pgftag:		'<' PGFTAG STR {
		  if ((strcmp(para_flag, "textflow") == 0) &&
		      in_para && !in_notes) {
		    prev_para_name = (char *)malloc(strlen($3));
		    strcpy(prev_para_name, $3);
		    }
		} '>'
		;

row:		'<' ROW {
		  if (in_table) {
		    first_cell = 1;
		    if (first_row) {
		      first_row = 0;
		      if (is_table) {
			strcat (curr_caption_str, "{|"); /* temp string */
			while (num_cols-- > 0)
			  strcat (curr_caption_str, " l |");
			strcat (curr_caption_str, "} \\hline\n");
			store_table(STORE_STR, curr_caption_str);
			*curr_caption_str = '\0';
		      }
		      else
			store_table(STORE_STR, "\n");
		    }
		    else
		      store_table(STORE_STR, " \\\\ \\hline\n");
		  }
		} mifs '>'
		;

string:		'<' STRING STR {
		  if (strcmp(para_flag, "textflow") == 0) {
		    first_string = 0;
		    if (in_notes)
			store_footnote(STORE_STR, esc_reserved_char($3));
		    else if (in_para && !ignore_text)
			fprintf(yyout, "%s", esc_reserved_char($3));
		  }
		  if (in_table)
		    store_table(STORE_STR, esc_reserved_char($3));
		} '>'
		;

table:		'<' ATBL NUM {
		  if ((strcmp(para_flag, "textflow") == 0) && in_para)
		    retrieve_table($3);
		} '>'
		;

table_caption:	'<' TBLTITLECONTENT {
		  in_caption = 1;
		} mifs '>' {
		  in_caption = 0;
		}
		;

table_cols:	'<' TBLNUMCOLUMNS NUM {
		  num_cols = atoi ($3);
		} '>'
		;

table_def:	'<' TBL {
		  in_table = 1;
		} mifs '>' {
		  in_table = 0;
		  store_table(STORE_DETAILS, NULL);
		}
		;

table_id:	'<' TBLID NUM {
		  store_table(STORE_ID, $3);
		} '>'
		;

table_tag:	'<' TBLTAG STR {
		  if (in_table)
		    store_table(STORE_TAG, $3);
		} '>'
		;

textflow:	'<' TEXTFLOW {
		  para_flag = "textflow";
		} mifs '>' {
		  end_para(END_OF_TEXTFLOW);
		}
		;

var_def:	'<' VARIABLEDEF STR {
		  store_variable(STORE_STR, esc_reserved_char($3));
		} '>'
		;

var_fmt:	'<' VARIABLEFORMAT {
		  in_var = 1;
		} mifs '>' {
		  in_var = 0;
		  store_variable(STORE_DETAILS, NULL);
		}
		;

var_name:	'<' VARIABLENAME STR {
		  if (in_var)
		    store_variable(STORE_ID, $3);
		  else if (in_para && !first_string && !ignore_text)
		    retrieve_variable($3);
		} '>'
		;

other_statements:	'<' IDENT others '>'
			;

others:			/* empty */
			| others more_statements
			;

more_statements:	mif
			| NUM
			| IDENT
			| STR
			| NUM '"'
			| NUM '%'
			;

%%

/***************************************************************/
/* to display the line number when an syntax error is          */
/* encountered                                                 */
/***************************************************************/

void yyerror(s)
char *s;
{
  fprintf(stderr,"%s: %s at line number %d\n", PROG, s, lineno);
} /* end of yyerror()  */

/***************************************************************/
/* to output the end document command                          */
/***************************************************************/

void yyaccpt()
{
  fprintf(yyout, "\n\\end{document}\n");
} /* end of yyaccpt()  */

/***************************************************************/
/* to output the document preamble commands                    */
/***************************************************************/
/* KJT 04/05/97: LaTeX2e document class commands added         */
/***************************************************************/

void output_preamble(char trans_type)
{
  char doc_class [16];

  switch(trans_type) {
    case 'a':
	strcpy (doc_class, "article");
	break;
    case 'b':
	strcpy (doc_class, "book");
	break;
    case 'r':
    default:
	strcpy (doc_class, "report");
	break;
  }
  fprintf(yyout, "%%%% Adjust the following if you use LaTeX2e\n\n");
  fprintf(yyout, "\\documentstyle[mif__la]{%s}\n\n", doc_class);
  fprintf(yyout, "%%%% \\documentclass{%s}\n", doc_class);
  fprintf(yyout, "%%%% \\usepackage{mif__la}\n");
} /* end of output_preamble(trans_type) */

/***************************************************************/
/* to allocate an area to store the footnote id so that the    */
/* value of the footnote id does not change with the pointer   */
/***************************************************************/

char *alloc_storage(str_ptr)
char *str_ptr;
{
 int  str_length;
 char *s;

 str_length = strlen(str_ptr);
 s = (char *)malloc(str_length);
 strncpy(s, str_ptr, str_length);
 s[str_length] = '\0';
 return s;
}

/***************************************************************/
/* to output the equivalent LaTeX commands for the list-making */
/* environment                                                 */
/***************************************************************/

void write_list_para(item_para)
char *item_para;
{
 int  str_length, list_level;
 int found_list = 0;
 char *list_tag;

 switch(item_para[3]) {
   case 'B':  /* LI1Bullet, LI2Bullet, LI3Bullet or LI4Bullet */
	      list_tag = "itemize";
	      break;
   case 'E':  /* LI1Enumerate, LI2Enumerate,
		 LI3Enumerate or LI4Enumerate */
	      list_tag = "enumerate";
	      break;
   case 'L':  /* LI1Label, LI2Label, LI3Label or LI4Label */
	      list_tag = "description";
	      break;
   default:   break;
  } /* end of switch */

 /* if the current list paragraph is different from the previous */
 if (strcmp(item_para, prev_list_tag) != 0) {
   if (level_cnt != 0) { /* if the previous list(s) has not ended
			    with an "\end{}" statement */
     for (list_level = level_cnt - 1;
	  list_level >= 0 && !found_list;
	  list_level--) {
      if (strcmp(list_table[list_level], item_para) == 0) {
	found_list = 1;
	for (level_cnt--; level_cnt != list_level; level_cnt--) {
	  switch(list_table[level_cnt][3]) {
	   case 'B':  /* LI1Bullet, LI2Bullet, LI3Bullet or LI4Bullet */
		      list_tag = "itemize";
		      break;
	   case 'E':  /* LI1Enumerate, LI2Enumerate,
			 LI3Enumerate or LI4Enumerate */
		      list_tag = "enumerate";
		      break;
	   case 'L':  /* LI1Label, LI2Label, LI3Label or LI4Label */
		      list_tag = "description";
		      break;
	   default:   break;
	   } /* end of switch */
	  fprintf(yyout, "\n\\end{%s}\n", list_tag);
	 } /* end of for */
	} /* end of if */
       } /* end of for */
    } /* end of if */

  if (!found_list)
    {
     fprintf(yyout, "\n\\begin{%s}\n", list_tag);
     strcpy(list_table[level_cnt], item_para);
    }

  level_cnt++;

  /* to store the previous list environment */
  prev_list_tag = alloc_storage(item_para);

  } /* end of if */

} /* end of write_list_para */


/***************************************************************/
/* output the LaTeX commands to end the list-making            */
/* environment                                                 */
/***************************************************************/

void write_end_list(pgf_tag)
char *pgf_tag;
{
 int  list_level;
 char *list_tag;

 for (list_level = level_cnt - 1; list_level >= 0 ; list_level--) {
   switch(list_table[list_level][3]) {
     case 'B':  /* LI1Bullet, LI2Bullet, LI3Bullet or LI4Bullet */
		list_tag = "itemize";
		break;
     case 'E':  /* LI1Enumerate, LI2Enumerate, LI3Enumerate or LI4Enumerate */
		list_tag = "enumerate";
		break;
     case 'L':  /* LI1Label, LI2Label, LI3Label or LI4Label */
		list_tag = "description";
		break;
     default:   break;
    }
   fprintf(yyout, "\n\\end{%s}\n", list_tag);
  }

 /* initialisation */
 level_cnt = 0;
 prev_list_tag = "";
 for (list_level = 0; list_level < 15; list_level++) {
    list_table[list_level][0] = '\0';
  }

} /* end of write_end_list */

/***************************************************************/
/* LaTeX does not accept # % & _ ~ ^ \ { } $ as ordinary       */
/* printing symbol. They can only be printed by adding a       */
/* backslash or by changing them to math mode. The first seven */
/* characters are converted using "mifch_la". The               */
/* last three characters are frequently used in LaTeX commands */
/* hence they cannot be changed from the output file.          */
/***************************************************************/

char *esc_reserved_char(text)
char *text;
{
  static char res_char[RESCHAR_SIZE];
  char ch;
  char *res = res_char;
  int bslash = 0;

  do {
    ch = *text++;
    if (bslash) {
      bslash = 0;
      if (ch == '\\')
	ch = '*';		/* mark second backslash */
      else if (ch == 'x' && in_symbol)
	ch = 'X';		/* mark hex character in symbol font */
    }
    else if (ch == '\\')
      bslash = 1;
    else if (ch == '{' || ch == '}' || ch == '$')
      *res++ = '\\';
    *res++ = ch;
  } while (ch != '\0');

  if (strlen (res_char) > RESCHAR_SIZE)
    fprintf (stderr, "%s: conversion overflowed, line %d\n", PROG, lineno);
  return res_char;
} /* end of esc_reserved_char */

/***************************************************************/
/* this function is to ensure that special LaTeX environments  */
/* are started or terminated properly.                         */
/***************************************************************/

void check_special_pgf(para_name)
char *para_name;
{
  if ((strcmp(para_name, "Table of Contents") == 0) ||
      (strcmp(para_name, "List of Figures")  == 0) ||
      (strcmp(para_name, "List of Tables") == 0) ||
      (strcmp(para_name, "Index Entries") == 0) ||
      (strcmp(para_name, "Date") == 0)) {
    ignore_text = 1;
  }
  else {
    /* check if to ignore paragraphs till first string */
    if (!first_string)
      ignore_text = 0;
    /* check if to wait for first paragraph after heading */
    if ((strcmp(para_name, "Part") == 0) || strend (para_name, "Heading"))
      first_heading_para = 1;
    else if (strcmp (para_name, "B1Text") == 0) {
      if (first_heading_para)
	first_heading_para = 0;
      else if (!ignore_text)
	fprintf (yyout, "\n\\noindent");
    }
  }

  /* output "\appendix" for first "A1Heading" */
  if (first_appendix && strcmp(para_name, "A1Heading") == 0) {
      fprintf(yyout, "\n\\appendix\n");
      first_appendix = 0;
  }

  /* output "\end{abstract}" for first non-"AText" paragraph */
  if (first_abstract == 1 && strcmp(para_name, "AText") == 0) {
    first_abstract = 0;
  }
  else if (first_abstract == 0 && strcmp(para_name, "AText") != 0) {
    first_abstract = -1;
    fprintf(yyout, "\n\\end{abstract}\n");
  }

} /* end of check_special_pgf */

/****************************************************************************/
/* this function is to ensure that special fonts are treated appropriately  */
/****************************************************************************/

void check_special_font(font_name)
char *font_name;
{
  if ((strcmp(font_name, "Symbol") == 0)) {
    in_symbol = 1;
  }
  else
    in_symbol = 0;

} /* end of check_special_font */

/***************************************************************/
/* this function is to handle the leveling in the list         */
/* environment.                                                */
/***************************************************************/

void handle_list_env(para_name)
char *para_name;
{
 switch(para_name[0]) {
   case 'L':  /* if list-making environment is required */
	      switch(para_name[1]) {
		case 'I': write_list_para(para_name);
			  break;

		case 'M': break;

		default:  /* if the previous paragraph is a   */
			  /* list, output the "\end" commands */
			  if (level_cnt > 0) {
			     write_end_list(para_name);
			   }
			  break;
	       }
	      break;

   default:   /* if the previous paragraph is a   */
	      /* list, output the "\end" commands */
	      if (level_cnt > 0) {
		write_end_list(para_name);
	       }
	      break;
  } /* end of switch */

} /* end of handle_list_env */

/***************************************************************/
/* Check if "str1" ends with "str2".                           */
/***************************************************************/

int strend(str1, str2)
char *str1, *str2;
{
  char *strptr;

  if ((strptr = strstr (str1, str2)) != NULL)
    return *(strptr + strlen (str2)) == '\0';
  else
    return 0;
}

/***************************************************************/
/* convert the FrameMaker paragraph format to its equivalent   */
/* commands in LaTeX.                                          */
/***************************************************************/

void convert_to_LaTeX_para_cmd(para_name)
char *para_name;
{
  int pgf_found = 0;

  for (arr_id = 0;
	(!pgf_found && special_rep_para[arr_id].para_tag != NULL);
	  arr_id++) {
    if  (strcmp(special_rep_para[arr_id].para_tag, para_name) == 0) {
      fprintf(yyout, "%s", prev_ftag_end_str);
      prev_ftag_end_str = "";
      handle_list_env(para_name);
      check_special_pgf(para_name);
      if (!first_string) {
	fprintf(yyout, "%s", special_rep_para[arr_id].para_beg_str);
	para_end_tag = special_rep_para[arr_id].para_end_str;
      }
      pgf_found = 1;
    } /* end of if */
  } /* end of for */

 if (!pgf_found) {
   /* ignore paragraph names beginning Header or Footer, or ending with
      TOC, LOF, LOT or IX */
   if ((strncmp (para_name, "Header", 6) == 0) ||
       (strncmp (para_name, "Footer", 6) == 0) ||
       strend(para_name, "TOC") || strend(para_name, "LOF") ||
       strend(para_name, "LOT") || strend(para_name, "IX"))
     ignore_text = 1;
   else
     fprintf(stderr, "%s: untranslated paragraph format %s\n", PROG,
	     para_name);
  }

} /* end of convert_to_LaTeX_para_cmd */

/***************************************************************/
/* convert the FrameMaker font format to its equivalent        */
/* command(s) in LaTeX.                                        */
/***************************************************************/

void convert_to_LaTeX_font_cmd(font_name)
char *font_name;
{
 /* output the necessary commands to end the previous font format */
 fprintf(yyout, "%s", prev_ftag_end_str);
 prev_ftag_end_str = "";

 if (strcmp(font_name, "") != 0) {
   font_found = 0;
   /* search for the equivalent LaTeX  command for the current font */
   for (arr_id = 0; (!font_found &&
       (special_font[arr_id].font_tag != NULL)); arr_id ++)
    if (strcmp(special_font[arr_id].font_tag, font_name) == 0) {
      font_found = 1;
      check_special_font (font_name);
      fprintf(yyout, "%s", special_font[arr_id].font_in_tex);
      prev_ftag_end_str = special_font[arr_id].font_end_str;
     } /* end of if */
  } /* end of if */

 if (!font_found && (strcmp(font_name, "") != 0)) {
   fprintf(stderr, "%s: untranslated character format %s\n", PROG, font_name);
  }

} /* end of convert_to_LaTeX_font_cmd */

/***************************************************************/
/* to store the footnote id and footnote text                  */
/* store field type = 'i' - store the footnote id              */
/*                    's' - concatenate all the lines in one   */
/*                          footnote                           */
/*                    'd' - store the text of the footnote     */
/*                    'f' - store the font of the footnote     */
/***************************************************************/

void store_footnote(store_field_type, store_field)
char store_field_type;	/* 'i', 's', 'd' or 'f' */
char *store_field;	/* footnote id, text to concat, footnote or font */
{
 size_t str_length;

 switch(store_field_type) {
  case STORE_ID:
    curr_fnote_id = alloc_storage(store_field);
    curr_fnote_str = (char *)malloc(FNOTE_SIZE);
    *curr_fnote_str = '\0';
    curr_fnote_end_str = "";
    if (first_footnote) {
      first_fnote_ptr =
	(struct id_str *) malloc(sizeof(struct id_str));
      first_fnote_ptr->id = curr_fnote_id;
      first_fnote_ptr->next = NULL;
    }
    else {
    for (curr_fnote_ptr = first_fnote_ptr;
	curr_fnote_ptr->next != NULL;
	curr_fnote_ptr = curr_fnote_ptr->next)
      ;  /* to find the next position */
    next_fnote_ptr =
      (struct id_str *) malloc(sizeof(struct id_str));
    next_fnote_ptr->id = curr_fnote_id;
    next_fnote_ptr->next = NULL;
    curr_fnote_ptr->next = next_fnote_ptr;
    }
    break;

  case STORE_STR:
    str_length = strlen(store_field);
    if ((strlen (curr_fnote_str) + str_length) < FNOTE_SIZE)
      strncat(curr_fnote_str, store_field, str_length);
    else
      fprintf(stderr,
	"%s: ignoring excessive footnote text \"%s\"\n", PROG,
	  store_field);
    break;

  case STORE_DETAILS:
    str_length = strlen(curr_fnote_end_str);
    strncat(curr_fnote_str, curr_fnote_end_str, str_length);
    if (first_footnote) {
      first_fnote_ptr->str = curr_fnote_str;
      first_fnote_ptr->next = NULL;
      first_footnote = 0;
    }
    else {
      for (curr_fnote_ptr = first_fnote_ptr;
	  curr_fnote_ptr->next != NULL;
	  curr_fnote_ptr = curr_fnote_ptr->next)
	;  /* to find the next position */
      curr_fnote_ptr->str = curr_fnote_str;
    }
    break;

  case STORE_FONT:
    font_found = 0;
    str_length = strlen(curr_fnote_end_str);
    strncat(curr_fnote_str, curr_fnote_end_str, str_length);
    curr_fnote_end_str = "";
    for (arr_id = 0;
	(!font_found && (special_font[arr_id].font_tag != NULL));
	arr_id ++) { /* search for the equivalent
		      LaTeX command for the current font */
      if  (strcmp(special_font[arr_id].font_tag, store_field) == 0) {
	font_found = 1;
	check_special_font (store_field);
	str_length = strlen(special_font[arr_id].font_in_tex);
	strncat(curr_fnote_str, special_font[arr_id].font_in_tex, str_length);
	str_length = strlen(special_font[arr_id].font_end_str);
	curr_fnote_end_str = special_font[arr_id].font_end_str;
      } /* end of if */
    } /* end of for */
    /* report erroneous character format */
    if (!font_found && (strcmp(store_field, "") != 0)) {
      fprintf(stderr, "%s: unknown character format %s\n", PROG, store_field);
    }
    break;

   default:    break;
  } /* end of switch */
} /* end of store_footnote */

/***************************************************************/
/* to retrieve the text for a footnote based on the fnote id   */
/***************************************************************/

void retrieve_footnote(footnote_id)
char *footnote_id;
{
 int found_footnote = 0;

 for (curr_fnote_ptr = first_fnote_ptr;
      ((curr_fnote_ptr->next != NULL) && !found_footnote);
      curr_fnote_ptr = curr_fnote_ptr->next) {
  if (strcmp(curr_fnote_ptr->id, footnote_id) == 0) {
      found_footnote = 1;
      fprintf(yyout, "\\footnote{%s}", curr_fnote_ptr->str);
     } /* end of if */
 } /* end of for */

 if (!found_footnote) {
   if (strcmp(curr_fnote_ptr->id, footnote_id) == 0)
     fprintf(yyout, "\\footnote{%s}", curr_fnote_ptr->str);
  } /* end of if */

} /* end of retrieve_footnote */

/***************************************************************/
/* to store the table id and table text                        */
/* store field type = 'i' - store the table id                 */
/*                    's' - concatenate all the lines in one   */
/*                          table                              */
/*                    'd' - store the text of the table        */
/*                    'f' - store the font of the table        */
/*                    't' - store the tag of the table         */
/***************************************************************/

void store_table(store_field_type, store_field)
char store_field_type;	/* 'i', 's', 'd', 'f' or 't' */
char *store_field;  	/* table id, text to concat, table or font */
{
 size_t str_length;

 switch(store_field_type) {
  case STORE_ID:
    curr_table_id = alloc_storage(store_field);
    curr_table_str = (char *) malloc(TABLE_SIZE);
    *curr_table_str = '\0';
    curr_table_end_str = "";
    curr_caption_str = (char *) malloc(CAPTION_SIZE);
    *curr_caption_str = '\0';
    curr_caption_end_str = "";
    if (first_table) {
      first_table_ptr =
	(struct id_str *) malloc(sizeof(struct id_str));
      first_table_ptr->id = curr_table_id;
      first_table_ptr->next = NULL;
    }
    else {
      for (curr_table_ptr = first_table_ptr;
	  curr_table_ptr->next != NULL;
	  curr_table_ptr = curr_table_ptr->next)
	;  /* to find the next position */
      next_table_ptr =
	(struct id_str *) malloc(sizeof(struct id_str));
      next_table_ptr->id = curr_table_id;
      next_table_ptr->next = NULL;
      curr_table_ptr->next = next_table_ptr;
    }
    first_row = 1; num_cols = 0;
    break;

  case STORE_TAG:
    is_table = 0; is_figure = 0;
    if (strend (store_field, "Table")) {
      is_table = 1;
      strcat(curr_table_str, "\n\\begin{table}\n");
      strcat(curr_table_str, "\n\\begin{center}\n");
      strcat(curr_table_str, "\n\\begin{tabular}");
    }
    else if (strend (store_field, "Figure")) {
      is_figure = 1;
      strcat(curr_table_str, "\n\\begin{figure}");
    }
    else
      fprintf(stderr, "%s: unknown table type %s\n", PROG,
	store_field);
    break;

  case STORE_STR:
    if (in_caption) {
      str_length = strlen(store_field);
      if ((strlen (curr_caption_str) + str_length) <
	  CAPTION_SIZE)
	strncat(curr_caption_str, store_field, str_length);
      else
	fprintf(stderr,
	  "%s: ignoring excessive caption text \"%s\"\n", PROG,
	    store_field);
    }
    else {
      str_length = strlen(store_field);
      if ((strlen (curr_table_str) + str_length) < TABLE_SIZE)
	strncat(curr_table_str, store_field, str_length);
      else
	fprintf(stderr,
	  "%s: ignoring excessive table text \"%s\"\n", PROG,
	    store_field);
    }
    break;

  case STORE_DETAILS:
    str_length = strlen(curr_table_end_str);
    strncat(curr_table_str, curr_table_end_str, str_length);
    str_length = strlen(curr_caption_end_str);
    strncat(curr_caption_str, curr_caption_end_str, str_length);
    if (is_table) {
      strcat(curr_table_str, " \\\\ \\hline\n\\end{tabular}\n");
      strcat(curr_table_str, " \n\\end{center}\n");
      strcat(curr_table_str, "\n\\caption{");
      strcat(curr_table_str, curr_caption_str);
      strcat(curr_table_str, "}\n");
    strcat(curr_table_str, "\n\\end{table}");
    }
    else if (is_figure) {
      strcat(curr_table_str, "\n\\caption{");
      strcat(curr_table_str, curr_caption_str);
      strcat(curr_table_str, "}\n");
      strcat(curr_table_str, "\n\\end{figure}");
    }
    if ((strlen (curr_table_str)) >= TABLE_SIZE)
      fprintf(stderr, "%s: table overflowed, line %d\n", PROG, lineno);
    if (first_table) {
      first_table_ptr->str = curr_table_str;
      first_table_ptr->next = NULL;
      first_table = 0;
    }
    else {
      for (curr_table_ptr = first_table_ptr;
	  curr_table_ptr->next != NULL;
	  curr_table_ptr = curr_table_ptr->next)
	;  /* to find the next position */
      curr_table_ptr->str = curr_table_str;
    }
    break;

  case STORE_FONT:
    font_found = 0;
    if (in_caption) {
      str_length = strlen(curr_caption_end_str);
      strncat(curr_caption_str, curr_caption_end_str, str_length);
      curr_caption_end_str = "";
    }
    else {
      str_length = strlen(curr_table_end_str);
      strncat(curr_table_str, curr_table_end_str, str_length);
      curr_table_end_str = "";
    }
    for (arr_id = 0; (!font_found && (special_font[arr_id].font_tag != NULL));
	 arr_id ++) { /* search for equivalent font LaTeX command */
      if  (strcmp(special_font[arr_id].font_tag, store_field) == 0) {
	font_found = 1;
	check_special_font (store_field);
	str_length = strlen(special_font[arr_id].font_in_tex);
	if (in_caption) {
	  strncat(curr_caption_str,
	    special_font[arr_id].font_in_tex, str_length);
	  curr_caption_end_str = special_font[arr_id].font_end_str;
	}
	else {
	  strncat(curr_table_str,
	    special_font[arr_id].font_in_tex, str_length);
	  curr_table_end_str = special_font[arr_id].font_end_str;
	}
      } /* end of if */
    } /* end of for */
    /* report erroneous character format */
    if (!font_found && (strcmp(store_field, "") != 0)) {
      fprintf(stderr, "%s: unknown character format %s\n", PROG, store_field);
    }
    break;

  default:
    break;
  } /* end of switch */
} /* end of store_table */

/***************************************************************/
/* to retrieve the text for a table based on the table id     */
/***************************************************************/

void retrieve_table(table_id)
char *table_id;
{
 int found_table = 0;

 for (curr_table_ptr = first_table_ptr;
      ((curr_table_ptr->next != NULL) && !found_table);
      curr_table_ptr = curr_table_ptr->next) {
  if (strcmp(curr_table_ptr->id, table_id) == 0) {
      found_table = 1;
      fprintf(yyout, "\n%s", curr_table_ptr->str);
     } /* end of if */
 } /* end of for */

 if (!found_table) {
   if (strcmp(curr_table_ptr->id, table_id) == 0)
      fprintf(yyout, "\n%s", curr_table_ptr->str);
  } /* end of if */

} /* end of retrieve_table */

/***************************************************************/
/* to store the variable id and variable value                 */
/* store field type = 'i' - store the variable id              */
/*                    's' - store the variable value           */
/*                    'd' - store the text of the variable     */
/***************************************************************/

void store_variable(store_field_type, store_field)
char store_field_type;	/* 'i', 's' or 'd' */
char *store_field;	/* variable id, text to concat, variable or font */
{
 size_t str_length;

 switch(store_field_type) {
   case STORE_ID:
    curr_var_id = alloc_storage(store_field);
    if (first_var) {
      first_var_ptr =
	(struct id_str *) malloc(sizeof(struct id_str));
      first_var_ptr->id = curr_var_id;
      first_var_ptr->next = NULL;
    }
    else {
      for (curr_var_ptr = first_var_ptr;
	  curr_var_ptr->next != NULL;
	  curr_var_ptr = curr_var_ptr->next)
	;  /* to find the next position */
      next_var_ptr =
	(struct id_str *) malloc(sizeof(struct id_str));
      next_var_ptr->id = curr_var_id;
      next_var_ptr->next = NULL;
      curr_var_ptr->next = next_var_ptr;
    }
    break;

  case STORE_STR:
    curr_var_str = alloc_storage(store_field);
    break;

  case STORE_DETAILS:
    if (first_var) {
      first_var_ptr->str = curr_var_str;
      first_var_ptr->next = NULL;
      first_var = 0;
    }
    else {
      for (curr_var_ptr = first_var_ptr;
	  curr_var_ptr->next != NULL;
	  curr_var_ptr = curr_var_ptr->next)
	;  /* to find the next position */
      curr_var_ptr->str = curr_var_str;
    }
    break;

  default:
    break;
  } /* end of switch */
} /* end of store_variable */

/***************************************************************/
/* to retrieve the text for a variable based on the var id     */
/***************************************************************/

void retrieve_variable(variable_id)
char *variable_id;
{
 int found_variable = 0;

 for (curr_var_ptr = first_var_ptr;
      ((curr_var_ptr->next != NULL) && !found_variable);
      curr_var_ptr = curr_var_ptr->next) {
  if (strcmp(curr_var_ptr->id, variable_id) == 0) {
      found_variable = 1;
      fprintf(yyout, "%s", curr_var_ptr->str);
     } /* end of if */
 } /* end of for */

 if (!found_variable) {
   if (strcmp(curr_var_ptr->id, variable_id) == 0)
     fprintf(yyout, "%s", curr_var_ptr->str);
  } /* end of if */

} /* end of retrieve_variable */

/***************************************************************/
/* to handle end of different paragraphs                       */
/***************************************************************/

void end_para(end_block)
int  end_block;
{
  switch(end_block) {
    case(END_OF_FONTCATALOG):
    case(END_OF_PGFCATALOG):
    case(END_OF_TEXTFLOW):
      para_flag = ""; break;
    case(END_OF_NOTES):
      in_notes = 0; break;
    case(END_OF_FNOTE):
      store_footnote(STORE_DETAILS, NULL); break;
    case(END_OF_PARA):
      /* to end LaTeX commands */
      fprintf(yyout, "%s", para_end_tag);
      in_para = 0;
      para_end_tag = "";
      break;
  } /* end of switch */
} /* end of end_para */

/***************************************************************/
/* usage information                                           */
/***************************************************************/

/* KJT 30/11/22: added return type */

void usage()
{
  fprintf (stderr,"usage: %s [-a|-b|-r] file[.mif|.framemif]\n", PROG);
  exit (1);
}

/***************************************************************/
/* main program                                                */
/***************************************************************/

/* KJT 30/11/22: added return type */

int main(argc, argv)
int argc;
char *argv[];
{
  extern FILE   *yyin, *yyout;
  register char *ap;
  register int  i;
  int has_ext;
  char input_arg[80];
  char input_file[90];
  char output_file[90];
  char temp_file[30];
  char temp_str[200];

  lineno = 1;

  for (i = 1; i < argc; i++) {
    ap = argv[i];
    if (ap[0] == '-') {
      switch(ap[1]) {
	case 'a':
	  /* translate to article format */
	  trans_type = ap[1];
	  for (arr_id = 0; special_art_para[arr_id].para_tag != NULL;
	       arr_id++) {
	    special_rep_para[arr_id].para_tag =
	      special_art_para[arr_id].para_tag;
	    special_rep_para[arr_id].para_beg_str =
	      special_art_para[arr_id].para_beg_str;
	    special_rep_para[arr_id].para_end_str =
	      special_art_para[arr_id].para_end_str;
	  } /* end of for */
	  break;
	case 'b':
	  /* translate to book format */
	  trans_type = ap[1];
	  for (arr_id = 0; special_bk_para[arr_id].para_tag != NULL;
	       arr_id++) {
	    special_rep_para[arr_id].para_tag =
	      special_bk_para[arr_id].para_tag;
	    special_rep_para[arr_id].para_beg_str =
	      special_bk_para[arr_id].para_beg_str;
	    special_rep_para[arr_id].para_end_str =
	      special_bk_para[arr_id].para_end_str;
	  } /* end of for */
	  break;
	case 'r':
	  /* translate to report format */
	  trans_type = ap[1];
	  break;
	default:
	  fprintf(stderr, "%s: unknown option %s\n", PROG, ap);
	  usage();
	} /* end of switch */
      } /* end of if */
  } /* end of for */

  switch(trans_type) {
    case 'a':
    case 'b':
    case 'r': {
      if (argc != 3) {
	fprintf(stderr,"%s: input filename needed\n", PROG);
	usage();
      }
      else
	strcpy(input_arg, argv[2]);
    }
    break;

    default:
	if (argc != 2) { /* if the translation type is not specified */
	  fprintf(stderr,"%s: input filename needed\n", PROG);
	  usage();
	}
	else {
	  strcpy(input_arg, argv[1]);
	  trans_type = 'r';   /* default is report format */
	} /* end of else */
	break;

  } /* end of switch */

  strcpy(input_file, input_arg);
  strcpy(output_file, input_arg);

  if (strend (input_arg, MIFEXT1)) {		/* has one MIF extension */
    yyin = fopen(input_file, "r");
    *(output_file + strlen (output_file) - strlen (MIFEXT1)) = '\0';
    strcat(output_file, TEXEXT);
  }
  else if (strend (input_arg, MIFEXT2)) {	/* has other MIF extension */
    yyin = fopen(input_file, "r");
    *(output_file + strlen (output_file) - strlen (MIFEXT2)) = '\0';
    strcat(output_file, TEXEXT);
  }
  else {					/* has no MIF extension */
    strcat(input_file, MIFEXT1);		/* try one MIF extension */
    if ((yyin = fopen(input_file, "r")) == NULL) {
      strcpy(input_file, input_arg);		/* try other MIF extension */
      strcat(input_file, MIFEXT2);
      yyin = fopen(input_file, "r");
    }
    strcat(output_file, TEXEXT);
  }

  if (yyin == NULL) {
    fprintf(stderr,"%s: cannot read %s\n", PROG, input_arg);
    usage ();
  }

  sprintf (temp_file, "/tmp/%s%d", PROG, getpid());
						/* make temporary filename */

  if ((yyout = fopen(temp_file, "w")) == NULL) {
    fprintf (stderr,"%s: cannot write %s\n", PROG, temp_file);
    usage ();
  }

  /* initialise the table that stores the different levels of lists */
  for (arr_id = 0; arr_id < 15; arr_id++)
    list_table[arr_id][0] = '\0';

  /* normal interaction on yyin and yyout */
  yyparse();

  yyaccpt();

  /* close files */
  fclose(yyout);
  fclose(yyin);

  /* convert special characters. */
  sprintf (temp_str, "mifch_la < %s > %s", temp_file, output_file);
  system(temp_str);
  sprintf (temp_str, "rm %s", temp_file);
  system(temp_str);

} /* end of main */
