%{

/*****************************************************************

msctex.l	K. J. Turner	11/05/02

This "lex" program reads MSC/PR text (which is assumed to be syntactically
valid) from the standard input, and writes it to the standard output in a
form suitable for processing by LaTeX. The flag "-n" causes output lines to
be numbered at the right.

The program translates initial tab and space characters on a line into LaTeX
horizontal space (\hspace). This assumes that indentation will be by small
(e.g. 2) units of space. Because LaTeX limits the number of tab stops in a
"tabbing" environment to around a dozen, it is not possible to have tab
stops every space or two.

Tabs and multiple spaces embedded in a line are translated into LaTeX tab
commands (\>). This assumes that embedded tabs and multiple spaces are
likely to be used to format comments and the like, for which a coarser
degree of control over spacing is acceptable.

Since the output is assumed to be embedded in a LaTeX "tabbing" environment,
a LaTeX newline command (\\) is appended to each line.

MSC 96 keywords are converted to bold (\bf). Keywords are recognised only in
lower case or with capitalisation of each word (e.g. "EndMSC", "InLine").
LaTeX special characters are escaped. (The keywords "end" and "route" are
accepted on their own.)

The program recognises "/" followed by "*" as beginning comments, "*"
followed by "/" as ending comments, and "'" as beginning or ending a
string.  Comments may not be nested. Comment delimiters are treated
literally inside strings, and string delimiters are treated as comments
inside comments. MSC keywords are not emboldened inside comments or strings.
To preserve the appearance of strings, embedded tabs and multiple spaces are
converted into tabs. This is only a rough approximation because of the
proportional spacing.

*****************************************************************/

#define Tab 8		/* number of characters between tab positions */

#define Spc 0.5		/* number "em"s for horizontal space unit */

#define Prog "msctex"	/* program name */

int IpPos = 0;		/* current input position (0 = first) */

int OpPos = 0;		/* current output position (0 = first) */

int TabPos = 0;		/* current tab position (0 = first) */

int InComment = 0;	/* indicates whether inside a comment (1 =
			   yes) */
int InString = 0;       /* indicates whether inside a string (1 =
			   yes) */
int LineNum = 0;	/* current line number - line numbering off by
			   default */

/* output current text */

void echo ();

/*  At the start of a line make good the difference between the input
    and output positions by using "\hspace". Otherwise, round the input
    position up to next tab stop, and output LaTeX tabs to bring the
    output position up to the input position. */

void Input () {
  int NumTabs;

  if (OpPos < IpPos) {				/* positions disagree? */
    if (OpPos == 0) {				/* start of line? */
      OpPos = IpPos;
      printf ("\\hspace{%.1fem}", Spc * IpPos);
    }
    else {
	NumTabs = (IpPos - 1) / Tab + 1;	/* next tab stop */
	OpPos = Tab * NumTabs;
	while (TabPos < NumTabs) {
	  printf ("\\>"); TabPos++;
	}
    }
  }
}

/* Output current identifier string, updating input and output
   positions. Escape special LaTeX characters as required. */

void OutputId () {
  int Ind;
  char Ch;

  for (Ind = 0; Ind < yyleng; Ind++) {
    Ch = yytext [Ind];
    if (Ch == '#' || Ch == '$' || Ch == '&' || Ch == '%' || Ch == '{' || Ch == '}') {
      putchar ('\\');
      putchar (Ch);
    }
    else if (Ch == '<' || Ch == '>' || Ch == '|' || ((Ch == '-') && !InComment)) {
      putchar ('$');
      putchar (Ch);
      putchar ('$');
    }
    else if (Ch == '~')
      printf ("$\\sim$");
    else if (Ch == '^')
      printf ("\\^{}");
    else if (Ch == '_')
      printf ("\\_\\,");
    else if (Ch == '\\')
      printf ("$\\backslash$");
    else putchar (Ch);
  }
  IpPos += yyleng;
  OpPos += yyleng;
}

/* Output current symbol string, updating input and output positions.
   Escape special LaTeX characters as required, and use special symbols
   where necessary. */

void OutputSym () {
  if ((!strcmp (yytext, "-")) || (!strcmp (yytext, "<")) ||
	(!strcmp (yytext, ">")) || (!strcmp (yytext, "<>")) ||
	(!strcmp (yytext, "<=")) || (!strcmp (yytext, ">=")) ||
	(!strcmp (yytext, "@")))
    printf ("$%s$", yytext);
  else if (!strcmp (yytext, "^"))
    printf ("\\%s{}", yytext);
  else if (!strcmp (yytext, "@"))
    printf ("\\%s", yytext);
  else
    echo ();
  IpPos += yyleng;
  OpPos += yyleng;
}

main (int argc, char *argv[]) {
  if (argc > 1)
    if (argc == 2)
      if (!strcmp (argv[1], "-n"))
        LineNum = 1;				/* turn on line numbering */
      else
        printf ("%s: unknown option %s\n", Prog, argv[1]);
    else
      printf ("%s: too many arguments\n", Prog);
  yylex ();
}

%}

%e 1000						/* increase number of nodes */

%n 500						/* increase number of states */

%p 4000						/* increase number of positions */

Letter		[a-zA-Z_]
Digit		[0-9]

Identifier      {Letter}({Letter}|{Digit})*

Comment		"\/*"|"*\/"

Symbol1		"+"|"-"|"*"|"/"|"="|"<"|">"|"["|"]"|"."|","|":"|";"
Symbol2		"^"|"("|")"|"!"
Symbol		{Symbol1}|{Symbol2}

String          "'"

%%

^" "+ |
" "" "+	IpPos += yyleng;

\t+ {
  IpPos = Tab * (IpPos / Tab + yyleng);
}

\n {
  IpPos = 0;
  OpPos = 0;
  TabPos = 0;
  if (LineNum > 0)
    printf ("\\`\\footnotesize%d\\\\\n", LineNum++);
  else
    printf ("\\\\\n");
}

{Comment} {
  Input ();
  InComment = !InComment;
  OutputId ();
}

{String} {
  Input ();
  if (!InComment)
    InString = !InString;
  OutputId ();
}

{Symbol} {
  Input ();
  OutputSym ();
}

(action|all|alt|as|before|begin|block|by|comment|concurrent)	|
(Action|All|Alt|As|Before|Begin|Block|By|Comment|Concurrent)	|
(condition|connect|create|decomposed|empty|end|endconcurrent)	|
(Condition|Connect|Create|Decomposed|Empty|End|EndConcurrent)	|
(endmsc|endexpr|env|exc|expr|external|found|from|gate|in|inf)	|
(EndMSC|EndExpr|Env|Exc|Expr|External|Found|From|Gate|In|Inf)	|
(inline|inst|instance|loop|lost|msc|mscdocument|msg|opt|order)	|
(InLine|Inst|Instance|Loop|Lost|MSC|MSCDocument|Msg|Opt|Order)	|
(par|process|reference|related|reset|service|seq|set|shared)	|
(Par|Process|Reference|Related|Reset|Service|Seq|Set|Shared)	|
(stop|subst|system|text|timeout|to|via)				|
(Stop|Subst|System|Text|Timeout|To|Via) {
  Input ();
  if (!InComment && !InString)
    printf ("{\\bf ");
  OutputId ();
  if (!InComment && !InString)
    printf ("}");
}

{Identifier}|. {
  Input ();
  OutputId ();
}

%%

/* output current text */

void echo () {
  ECHO;
}

