%{

/*****************************************************************

pastex.lex	K. J. Turner	11/05/02

This "lex" program reads Pascal text (which is assumed to be syntactically
valid) from the standard input, and writes it to the standard output in a
form suitable for processing by LaTeX. The flag "-n" causes output lines to
be numbered at the right.

The program translates initial tab and space characters on a line into LaTeX
horizontal space (\hspace). This assumes that indentation will be by small
(e.g. 2) units of space. Because LaTeX limits the number of tab stops in a
"tabbing" environment to around a dozen, it is not possible to have tab
stops every space or two.

Tabs and multiple spaces embedded in a line are translated into LaTeX tab
commands (\>). This assumes that embedded tabs and multiple spaces are
likely to be used to format comments and the like, for which a coarser
degree of control over spacing is acceptable.

Since the output is assumed to be embedded in a LaTeX "tabbing" environment,
a LaTeX newline command (\\) is appended to each line.

Pascal keywords are converted to bold (\bf). Keywords are recognised only in
lower case or with capitalisation of each word (e.g. "DownTo", "GoTo").
LaTeX special characters are escaped.

The program recognises "{" or "(*" as beginning comments, "}" or "*)" as
ending comments, and "'" as beginning or ending a string.  Comments may not
be nested. Comment delimiters are treated literally inside strings, and
string delimiters are treated as comments inside comments. Pascal keywords
are not emboldened inside comments or strings. To preserve the appearance of
strings, embedded tabs and multiple spaces are converted into tabs. This is
only a rough approximation because of the proportional spacing.

*****************************************************************/

#define Tab 8		/* number of characters between tab positions */

#define Spc 0.5		/* number "em"s for horizontal space unit */

#define Prog "pastex"	/* program name */

int IpPos = 0;		/* current input position (0 = first) */

int OpPos = 0;		/* current output position (0 = first) */

int TabPos = 0;		/* current tab position (0 = first) */

int InComment = 0;	/* indicates whether inside a comment (1
			   = yes) */

int InString = 0;	/* indicates whether inside a string (1 = yes) */

int LineNum = 0;	/* current line number - line numbering off by
			   default */

/* output current text */

void echo ();

/*  At the start of a line make good the difference between the input
    and output positions by using "\hspace". Otherwise, round the input
    position up to next tab stop, and output LaTeX tabs to bring the
    output position up to the input position. */

void Input () {
  int NumTabs;

  if (OpPos < IpPos) {				/* positions disagree? */
    if (OpPos == 0) {				/* start of line? */
      OpPos = IpPos;
      printf ("\\hspace{%.1fem}", Spc * IpPos);
    }
    else {
	NumTabs = (IpPos - 1) / Tab + 1;	/* next tab stop */
	OpPos = Tab * NumTabs;
	while (TabPos < NumTabs) {
	  printf ("\\>"); TabPos++;
	}
    }
  }
}

/* Output current parser string, updating input and output positions */

void Output () {
  echo ();
  IpPos += yyleng;
  OpPos += yyleng;
}

main (int argc, char *argv[]) {
  if (argc > 1)
    if (argc == 2)
      if (!strcmp (argv[1], "-n"))
        LineNum = 1;				/* turn on line numbering */
      else
        printf ("%s: unknown option %s\n", Prog, argv[1]);
    else
      printf ("%s: too many arguments\n", Prog);
  yylex ();
}

%}

Identifier	[a-zA-Z][a-zA-Z0-9]*
Comment		("{"|"(*"|"}"|"*)")
String		[']

%%

^" "+ |
" "" "+	IpPos += yyleng;

\t+		IpPos = Tab * (IpPos / Tab + yyleng);

\n {
  IpPos = 0; OpPos = 0; TabPos = 0;
  if (LineNum > 0)
    printf ("\\`\\footnotesize%d\\\\\n", LineNum++);
  else
    printf ("\\\\\n");
}

#|\$|_|\&|\% {
  Input ();
  putchar ('\\');
  putchar (yytext [0]);
  IpPos = ++OpPos;
}

\<|\> {
  Input ();
  putchar ('$');
  putchar (yytext [0]);
  putchar ('$');
  IpPos = ++OpPos;
}

~ {
  Input ();
  printf ("$\\sim$");
  IpPos = ++OpPos;
}

\^ {
  Input ();
  printf ("$\\uparrow$");
  IpPos = ++OpPos;
}

\\ {
  Input ();
  printf ("$\\backslash$");
  IpPos = ++OpPos;
}


{Comment} {
  Input ();
  if ((yytext[0] == '{') || (yytext[0] == '}'))
    putchar ('\\');
  if (!InString) InComment = !InComment;
  Output ();
}

{String} {
  Input ();
  if (!InComment) InString = !InString;
  Output ();
}

(and|array|begin|case|const|div|do|downto|else|end)	|
(And|Array|Begin|Case|Const|Div|Do|DownTo|Else|End)	|
(file|for|function|goto|if|in|label|mod|nil|not)	|
(File|For|Function|GoTo|If|In|Label|Mod|Nil|Not)	|
(of|or|packed|procedure|program|record|repeat|set)	|
(Of|Or|Packed|Procedure|Program|Record|Repeat|Set)	|
(then|to|type|until|var|while|with)			|
(Then|To|Type|Until|Var|While|With) {
  Input ();
  if (!InComment && !InString) printf ("{\\bf ");
  Output ();
  if (!InComment && !InString) printf ("}");
}

{Identifier}|. {
  Input ();
  Output ();
}

%%

/* output current text */

void echo () {
  ECHO;
}

