/**************************************************************************

  "put_pi.c"	K. J. Turner <kjt@cs.stir.ac.uk>	02/05/01

  This program converts to Pilot format the files given on the command line:
  
    -a	"<category>.addr" files are combined/converted to "AddressDB.pdb"
		    
    -d	"Datebook" is converted to "Datebook.pdb"
    
    -e	"<category>.exp" files are combined/converted to "ExpenseDB.pdb"
        (not yet implemented)
    
    -h	Print usage help
    
    -m	"<title>.memo[bpu[s]]" files are combined/converted to "MemoDB.pdb"
  
    -t	"ToDo" is converted to "ToDoDB.pdb"
  
  All files must be in the CURRENT directory. Any existing address, etc.
  database files will be OVERWRITTEN.
  
  To handle different formats from the ones generated by this program,
  customise the following functions:
  
    Address	set_addr_ent
    
    Datebook	set_appt_ents
    
    Expense	set_exp_ent
    
    Memo	set_memo_ent
    
    ToDo	set_todo_ent
  
  Inspired by "pilot-xfer" and "pilot-file" utilities by Kenneth Albanowski
  <kjahds@kjahds.com> and Pace Willisson <pace@blitz.com>.

  This is free software, distributed under the GNU Public License V2.

**************************************************************************/
          
/********************************* Globals *******************************/

#include "piconv.h"

/* appointment line fields */

char descr[1024];				/* description */
int day, month, year;				/* DD, MM, YYYY */
int beg_hour, beg_min;				/* begin hour/minute */
int end_hour, end_min;				/* end hour/minute */
int alm_hour, alm_min;				/* alarm hour/minute */

/******************************* Address Book *****************************/

/*** set up address app file information ***/

void set_addr_db_info (struct pi_file * pf, struct DBInfo * info)
{
  strcpy (info->name, "AddressDB");		/* set database name */
  info->flags = 0X00;				/* set flags */
  info->miscFlags = 0X00;			/* set miscellaneous flags */
  info->type = makelong ("DATA");		/* set data type */
  info->creator = makelong ("addr");		/* set data creator */
  info->version = 0;				/* set version (value?) */
  info->modnum = 0;				/* set modif. no. (value?) */
  info->createDate = time (NULL);		/* set created date */
  info->modifyDate = time (NULL);		/* set modified date */
  info->backupDate = -1;			/* set backup date */
  info->index = 0;				/* set index (value?) */
  if (pi_file_set_info (pf, info))		/* set database info */
    fatal ("cannot set address file info");
}

/*** set address category index from name ***/

int set_addr_cat (struct AddressAppInfo * addr_info, char * cat_name)
{
  int cat = 0;
  
  while ((cat < CAT_SIZE) &&			/* no category name match? */
   (strcmp (addr_info->category.name[cat], cat_name)))
    cat++;					/* to next category */
  if (cat == CAT_SIZE)				/* got to end of list? */
    cat = -1;					/* set invalid index */
  return (cat);					/* return category index */
}

/*** set up address application information ***/

void set_addr_info (struct pi_file * pf, struct AddressAppInfo * addr_info)
{
  DIR * dir;					/* current directory */
  struct dirent * dirent;			/* current dir entry */
  int len;					/* buffer length */
  int fld_ind;					/* category index */
  int i, j;					/* counters */
  char cat_name[CAT_SIZE];			/* category name */
  unsigned char buff[BUFF_SIZE];		/* data buffer */
  char * cp1, * cp2;				/* character pointers */
  char ch;					/* current character */
  
  for (i = 0; i < CAT_SIZE; i++)		/* initialise cat names */
    addr_info->category.renamed[i] = 1;		/* set new cat flag */
  for (i = 0; i < CAT_SIZE; i++)		/* initialise cat names */
    for (j = 0; j < CAT_SIZE; j++)
      addr_info->category.name[i][j] = 0;
  strcpy (addr_info->category.name[0], "Unfiled"); /* set unfiled category */
  fld_ind = 1;					/* initialise cat. index */
  if (dir = opendir (".")) {			/* open current directory? */
    while (dirent = readdir (dir)) {		/* read directory entries */
      cp1 = dirent->d_name;			/* set filename */
      if ((cp2 = strstr (cp1, ".addr")) &&	/* address file? */
           (*(cp1 + strlen (cp1) - 1) != '~')) { /* not "~" (backup) end? */
	i = cp2 - cp1;				/* set cat. name length */
	if (i >= (CAT_SIZE - 1))		/* category name too long? */
	  i = CAT_SIZE - 1;			/* truncate name */
	for (j = 0; j < i; j++) {		/* copy category name */
	  ch = *(cp1++);			/* get category character */
	  if (ch == '_')			/* underscore to space */
	    ch = ' ';
	  cat_name[j] = ch; 			/* copy character */
	}
	cat_name[j] = '\0';			/* terminate name */
	if (set_addr_cat (addr_info, cat_name) == -1) { /* cat. name new? */
	  if (fld_ind < CAT_SIZE) {		/* not too many categories? */
	    strcpy (addr_info->category.name[fld_ind], cat_name);
	    fld_ind++;				/* to next category */
	  }
	  else {				/* too many categories */
	    sprintf (message, "more than %d address categories", fld_ind);
	    fatal (message);
	  }
	}
      }
    }
  }
  else						/* cannot open current dir */
    fatal ("cannot read current directory");
  for (i = 0; i < CAT_SIZE; i++)		/* set cat ids */
    addr_info->category.ID[i] = i;
  addr_info->category.lastUniqueID = CAT_SIZE - 1; /* set last cat id */
  for (i = 0; i < ADDR_LAB_COUNT; i++)		/* initialise field labels */
    for (j = 0; j <= (CAT_SIZE - 1); j++)
      addr_info->labels[i][j] = 0;
  strcpy (addr_info->labels[0],  "Last name");	last_fld = 0;
  strcpy (addr_info->labels[1],  "First name");	first_fld = 1;
  strcpy (addr_info->labels[2],  "Company");	comp_fld = 2;
  strcpy (addr_info->labels[3],  "Work");	workph_fld = 3;
  strcpy (addr_info->labels[4],  "Home");	homeph_fld = 4;
  strcpy (addr_info->labels[5],  "Fax");	fax_fld = 5;
  strcpy (addr_info->labels[6], "Mobile");	mobile_fld = 6;
  strcpy (addr_info->labels[7],  "E-mail");	email1_fld = 7;
  strcpy (addr_info->labels[8],  "Address");	workad_fld = 8;
  strcpy (addr_info->labels[9],  "Town");	town_fld = 9;
  strcpy (addr_info->labels[10], "County");	county_fld = 10;
  strcpy (addr_info->labels[11], "Post Code");	code_fld = 11;
  strcpy (addr_info->labels[12], "Country");	country_fld = 12;
  strcpy (addr_info->labels[13], "Title");	title_fld = 13;
  strcpy (addr_info->labels[14], "E-mail 2");	email2_fld = 14;
  strcpy (addr_info->labels[15], "Web");	web_fld = 15;
  strcpy (addr_info->labels[16], "Links");	link_fld = 16;
  strcpy (addr_info->labels[17], "Home Addr");	homead_fld = 17;
  strcpy (addr_info->labels[18], "Note");	note_fld = 18;
  strcpy (addr_info->labels[19], "Main");	main_fld = 19;
  strcpy (addr_info->labels[20], "Pager");	pager_fld = 20;
  strcpy (addr_info->labels[21],  "Other");	other_fld = 21;
  for (i = 0; i < ADDR_LAB_COUNT; i++)		/* initialise labels */
    addr_info->labelRenamed[i] = 1;		/* set new label flag */
  for (i = 0; i < 8; i++)			/* initialise phone labels */
    strcpy (addr_info->phoneLabels[i], "\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0");
  addr_info->country = 0;			/* set country (value?) */
  addr_info->sortByCompany = 0;			/* set sort option (value?) */
  len = pack_AddressAppInfo (addr_info, buff, BUFF_SIZE); /* pack app info */
  if (pi_file_set_app_info (pf, buff, len))	/* set app info? */
    fatal ("cannot set addr app info");
}

/*** set up address sorting information ***/

void set_addr_sort_info (struct pi_file * pf)
{
    pi_file_set_sort_info (pf, NULL, 0);	/* set sort info */
}

/*** set address field, replacing "~" by newline (except in web address)
     and "^" by space (in first/last name) ***/

void set_addr_fld (int fld_ind, char * field, struct Address * addr)
{
  char * cp1, * cp2;				/* character pointers */
  char ch;					/* current character */

  if ((fld_ind >= 0) && strlen (field)) {	/* valid field? */
    addr->entry[fld_ind] =			/* make space for field */
      malloc (strlen (field) + 1);
    cp1 = addr->entry[fld_ind]; cp2 = field;	/* set initial pointers */
    while (ch = *(cp2++)) {			/* get character till NUL */
      if ((ch == '^') &&			/* caret in first/last name */
           ((fld_ind == first_fld) || (fld_ind == last_fld)))
        ch = ' ';				/* change to space */
      if ((ch == '~') && (fld_ind != web_fld))	/* tilde not in web addr? */
        ch = '\n';				/* change to newline */
      *(cp1++) = ch;				/* store character */
    }
    *cp1 = '\0';				/* terminate entry string */
  }
}

/*** set address entry in preferred format ***/

void set_addr_ent (char * addr_line, struct Address * addr)
{
  int i;					/* counter */
  char field[FIELD_SIZE], field1[FIELD_SIZE],	/* address fields */
    field2[FIELD_SIZE], field3[FIELD_SIZE];
  char tag;					/* address field type */
  char * cp1, * cp2, * cp3;			/* character pointers */

  for (i = 0; i < 5; i++)			/* set phone labels */
    addr->phoneLabel[i] = i;
  for (i = 0; i < ADDR_FLD_COUNT; i++)		/* init address entries */
    addr->entry[i] = "";
  cp1 = addr_line;				/* set current position */
  do {						/* repeatedly read fields */
    tag = *(cp1++);				/* set address field type */
    cp2 = field;				/* set field pointer */
    while ((*cp1 != '\t') && (*cp1 != '\n'))
      *(cp2++) = *(cp1++);			/* copy character */
    *cp2 = '\0';				/* terminate string */
    switch (tag) {
      case 'T':					/* title */
	set_addr_fld (title_fld, field, addr); break;
      case 'p':					/* prename */
	set_addr_fld (first_fld, field, addr); break;
      case 's':					/* surname */
	set_addr_fld (last_fld, field, addr); break;
      case 't':					/* phones work~home~fax~mob */
	cp2 = field;				/* copy field pointer */
	cp3 = field1;				/* init work pointer */
	while (*cp2 != '~')			/* copy till tilde */
	  *(cp3++) = *(cp2++);			/* copy work character */
	cp2++;					/* go past tilde */
	*cp3 = '\0';				/* terminate work */
	cp3 = field2;				/* init home pointer */
	while (*cp2 != '~')			/* copy till tilde */
	  *(cp3++) = *(cp2++);			/* copy home character */
	cp2++;					/* go past tilde */
	*cp3 = '\0';				/* terminate home */
	cp3 = field3;				/* init fax pointer */
	while (*cp2 != '~')			/* copy till tilde */
	  *(cp3++) = *(cp2++);			/* copy fax character */
	*cp3 = '\0';				/* terminate fax */
	cp2++;					/* past tilde to mobile */
	set_addr_fld (workph_fld, field1, addr); /* store work phone */
	set_addr_fld (homeph_fld, field2, addr); /* store home phone */
	set_addr_fld (fax_fld, field3, addr);	/* store fax phone */
	set_addr_fld (mobile_fld, cp2, addr);	/* store mobile phone */
	if (strcmp (field1, ""))		/* business phone known? */
	  addr->showPhone = 0;			/* show business no. */
    else if (strcmp (field2, ""))		/* home phone known? */
	  addr->showPhone = 1;			/* show home no. */
    else if (strcmp (field3, ""))		/* fax phone known? */
	  addr->showPhone = 2;			/* show fax no. */
    else if (strcmp (cp2, ""))			/* mobile phone known? */
	  addr->showPhone = 3;			/* show mobile no. */
    else
	  addr->showPhone = 0;			/* business by default */
	break;
      case 'a':					/* work address */
	set_addr_fld (workad_fld, field, addr); break;
      case 'A':					/* home address */
	set_addr_fld (homead_fld, field, addr); break;
      case 'e':					/* email addresses */
	cp2 = field;				/* copy field pointer */
	cp3 = field1;				/* init email 1 pointer */
	while (*cp2 != '~')			/* copy till tilde */
	  *(cp3++) = *(cp2++);			/* copy character */
	cp2++;					/* go past tilde */
	*cp3 = '\0';				/* terminate email 1 */
	cp3 = field2;				/* init email 2 pointer */
	while (*cp2 != '~')			/* copy till tilde */
	  *(cp3++) = *(cp2++);			/* copy character */
	*cp3 = '\0';				/* terminate email 2 */
	cp2++;					/* go past tilde */
	set_addr_fld (email1_fld, field1, addr); /* store email 1 */
	set_addr_fld (email2_fld, field2, addr); /* store email 2 */
	set_addr_fld (web_fld, cp2, addr);	/* store web address */
	break;
      case 'l':					/* links */
	set_addr_fld (link_fld, field, addr); break;
      default:					/* unknown */
	break;					/* ignore */
    }
  } while (*(++cp1));				/* to next field unless NUL */
}

/*** set address entries in preferred format ***/

void set_addr_ents (struct pi_file * pf, struct DBInfo * info,
  struct AddressAppInfo * addr_info)
{
  DIR * dir;					/* current directory */
  FILE * af;					/* input address file */
  struct Address addr;				/* Pilot address */
  struct dirent * dirent;			/* current dir entry */
  pi_uid_t uid;					/* user id */
  int cat;					/* record category */
  int attrs;					/* attributes */
  int len;					/* buffer length */
  int i, j;					/* counters */
  unsigned char buff[BUFF_SIZE];		/* data buffer */
  char addr_line[ADDR_SIZE];			/* input address line */
  char cat_name[CAT_SIZE + 1];			/* category name */
  char * cp1, * cp2;				/* character pointers */
  char ch;					/* current character */

  if (dir = opendir (".")) {			/* open current directory? */
    while (dirent = readdir (dir)) {		/* read directory entries */
      cp1 = dirent->d_name;			/* set filename */
      if ((cp2 = strstr (cp1, ".addr")) &&	/* address file? */
           (*(cp1 + strlen (cp1) - 1) != '~')) { /* not "~" (backup) end? */
	if (af = fopen (cp1, "r")) {		/* open file? */
	  i = cp2 - cp1;			/* set category length */
	  if (i >= (CAT_SIZE -1))		/* category name too long? */
	    i = CAT_SIZE - 1;			/* truncate name */
	  for (j = 0; j < i; j++) {		/* copy category name */
	    ch = *(cp1++);			/* get category character */
	    if (ch == '_')			/* underscore to space */
	      ch = ' ';
	    cat_name[j] = ch;			/* copy character */
	  }
	  cat_name [j] = '\0';			/* terminate name */
	  cat = set_addr_cat (addr_info, cat_name); /* set category */
	  attrs = dlpRecAttrDirty;		/* set record attributes */
	  uid = 0;				/* set user id (value?) */
	  while (fgets (addr_line, ADDR_SIZE, af)) { /* read addr line */
	    set_addr_ent (addr_line, &addr);	/* set address entry */
	    len = pack_Address (&addr, buff, BUFF_SIZE); /* pack address */
	    pi_file_append_record (pf, buff, len, attrs, cat, uid);
	  }
	  fclose (af);				/* close file */
	}
	else {					/* cannot open file */
          sprintf (message, "cannot read %s", cp1);
	  warning (message);
	}
      }
    }
  }
  else						/* cannot open current dir */
    fatal ("cannot read current directory");
}

/*** parse and write addresses ***/

void to_address ()
{
  struct pi_file *pf;				/* pilot file handle */
  struct DBInfo info;				/* database info */
  struct AddressAppInfo addr_info;		/* address app info */
  struct Address addr;				/* address */
  char * name;					/* pilot file name */

  printf ("%s: converting to Pilot address book\n", progname);
  name = "AddressDB.pdb";			/* set pilot file name */
  info.flags = 0X00;				/* set flags */
  if (pf = pi_file_create (name, &info)) {	/* open pilot file */
    set_addr_db_info (pf, &info);		/* set database info */
    set_addr_info (pf, &addr_info);		/* set app info */
    set_addr_sort_info (pf);			/* set sort info */
    set_addr_ents (pf, &info, &addr_info);	/* set address entires */
    if (pi_file_close (pf)) {			/* close pilot file? */
      sprintf (message, "cannot close %s", name);
      warning (message);
    }
  }
  else {					/* cannot open pilot file */
    sprintf (message, "cannot open %s", name);
    fatal (message);
  }
}

/******************************* Date Book *****************************/

/*** set up date book application file information ***/

void set_appt_db_info (struct pi_file * pf, struct DBInfo * info)
{
  strcpy (info->name, "DatebookDB");		/* set database name */
  info->flags = 0X00;				/* set flags */
  info->miscFlags = 0X00;			/* set miscellaneous flags */
  info->type = makelong ("DATA");		/* set data type */
  info->creator = makelong ("date");		/* set data creator */
  info->version = 0;				/* set version (value?) */
  info->modnum = 0;				/* set modif. no. (value?) */
  info->createDate = time (NULL);		/* set created date */
  info->modifyDate = time (NULL);		/* set modified date */
  info->backupDate = -1;			/* set backup date */
  info->index = 0;				/* set index (value?) */
  if (pi_file_set_info (pf, info))		/* set database info */
    fatal ("cannot set date book file info");
}

/*** set up date book application information ***/

void set_appt_info (struct pi_file * pf,
  struct AppointmentAppInfo * appt_info)
{
  unsigned char buff[BUFF_SIZE];		/* data buffer */
  int len;					/* buffer length */
  int i, j;					/* counters */
  
  for (i = 0; i < CAT_SIZE; i++)		/* initialise cat names */
    appt_info->category.renamed[i] = 1;		/* set new cat flag */
  for (i = 0; i < CAT_SIZE; i++)		/* initialise cat names */
    for (j = 0; j < CAT_SIZE; j++)
      appt_info->category.name[i][j] = 0;
  strcpy (appt_info->category.name[0], "");
  strcpy (appt_info->category.name[1], "");
  strcpy (appt_info->category.name[2], "");
  strcpy (appt_info->category.name[3], "");
  for (i = 0; i < CAT_SIZE; i++)		/* set cat ids */
    appt_info->category.ID[i] = 0;
  appt_info->category.lastUniqueID = CAT_SIZE - 1; /* set last cat id */
  appt_info->startOfWeek = 0;			/* set start day (value?) */
  len = pack_AppointmentAppInfo (appt_info, buff, BUFF_SIZE); /* pack app info */
  if (pi_file_set_app_info (pf, buff, len))	/* set app info? */
    fatal ("cannot set date book app info");
}

/*** set up appointment sorting information ***/

void set_appt_sort_info (struct pi_file * pf)
{
    pi_file_set_sort_info (pf, NULL, 0);	/* set sort info */
}

/*** roughly check appointment fields, returning 0/1 for OK/wrong ***/

int chk_appt_ent ()
{
  if ((1 <= day) && (day <= 31) &&		/* day in range? */
      (((0 <= beg_hour) && (beg_hour <= 23) &&	/* begin hour/min in range? */
        (0 <= beg_min) && (beg_min <= 59)) ||
	((beg_hour == -1) && (beg_min == -1))) &&
      (((0 <= end_hour) && (end_hour <= 23) &&	/* end hour/min in range? */
        (0 <= end_min) && (end_min <= 59)) ||
	((end_hour == -1) && (end_min == -1))) &&
      strlen (descr) > 0 &&			/* description non-empty? */
      (((0 <= alm_hour) && (alm_hour <= 23) &&	/* alarm hour/min in range? */
        (0 <= alm_min) && (alm_min <= 59)) ||
	((alm_hour == -1) && (alm_min == -1))))
    return (0);					/* checks are OK */
  else
    return (1);					/* checks fail */
}

/*** set up appointment structure ***/

void set_appt_ent (struct Appointment * appt)
{
  int adv;					/* alarm advance */
  
  if (beg_hour != -1)				/* begin time given? */
    appt->event = 0;				/* set as timed */
  else						/* no begin time */
    appt->event = 1;				/* set as untimed */
  appt->begin.tm_year = year;			/* set begin time ... */
  appt->begin.tm_mon = month;
  appt->begin.tm_mday = day;
  appt->begin.tm_hour = beg_hour;
  appt->begin.tm_min = beg_min;
  appt->begin.tm_sec = 0;
  appt->end.tm_year = year;			/* set end time ... */
  appt->end.tm_mon = month;
  appt->end.tm_mday = day;
  appt->end.tm_hour = end_hour;
  appt->end.tm_min = end_min;
  appt->end.tm_sec = 0;
  if (alm_hour != -1) {				/* alarm time given? */
    appt->alarm = 1;				/* set as alarmed */
    adv =					/* advance = begin - alarm */
      (beg_hour - alm_hour) * 60 + (beg_min - alm_min);
    if (adv < 100) {				/* alarm OK up to 99 mins. */
      appt->advance = adv;
      appt->advanceUnits = advMinutes;		/* alarm in minutes */
    }
    else {					/* convert alarm to hours */
      appt->advance = (adv + 30) / 60;		/* round to nearest hour */
      appt->advanceUnits = advHours;		/* alarm in hours */
    }
  }
  else {					/* no alarm time */
    appt->alarm = 0;				/* set as unalarmed */
    appt->advance = 0;				/* no advance */
    appt->advanceUnits = advMinutes;		/* alarm in minutes */
  }
  appt->repeatType = repeatNone;		/* no repetition */
  appt->repeatForever = 0;			/* not forever */
  appt->repeatFrequency = 0;				/* no repeat frequency */
  appt->repeatDay = 0;				/* no repeat on item */
  appt->repeatWeekstart = 0;			/* no repeat week start */
  appt->exceptions = 0;				/* no repeat exceptions */
  appt->description = descr;			/* set description */
  appt->note = NULL;				/* set note */
}

/*** set appointment entries from preferred format ***/

void set_appt_ents (struct pi_file * pf, struct DBInfo * info,
  struct AppointmentAppInfo * appt_info)
{
  FILE *af;					/* input appointment file */
  struct Appointment appt;			/* Pilot appointment */
  unsigned char buff[BUFF_SIZE];		/* data buffer */
  char appt_line[APPT_SIZE];			/* appointment line */
  char month_name[LINE_SIZE];			/* month name */
  char * cp1, * cp2;				/* character pointers */
  int len;					/* buffer length */
  int i;					/* counter */
  int attrs;					/* attributes */
  int cat;					/* record category */
  pi_uid_t uid;					/* user id */
  
  char cat_name[CAT_SIZE];			/* category name */
  char field[FIELD_SIZE], field1[FIELD_SIZE],	/* appointment fields */
    field2[FIELD_SIZE];
  char tag;					/* appointment field type */

  cp1 = "Datebook";				/* set filename */
  if (af = fopen (cp1, "r")) {			/* open file? */
    cat = 0;					/* set record category */
    uid = 0;					/* set user id (value?) */
    /* set up appointment records from file */
    while (fgets (appt_line, APPT_SIZE, af)) { /* read appointment line */
      cp1 = appt_line;				/* initialise appt line ptr */
      if (strcmp (cp1, "\n")) {			/* appt line not empty? */
	attrs = dlpRecAttrDirty;		/* set record attributes */
	sscanf (cp1, "%s %d %d\t", &month_name,	/* read MMM DD YYYY */
	  &day, &year);
	year -= 1900;				/* set years since 1900 */
	month = 0;				/* search for month name */
	while ((month < 12) && strcmp (month_name, mon_name[month]))
	  month++;
	if (month >= 12) {			/* name not found? */
	  sprintf (message, "unrecognised month %s", month_name);
	  warning (message);
	  continue;				/* to next appt line */
	}
	cp1 += 11;				/* to end of MMM DD YYYY */
	if (*(++cp1) != '\t') {			/* begin time follows? */
	  sscanf (cp1, "%d.%d", &beg_hour, &beg_min);
	  cp1 += 5;				/* to end of begin time */
	  if (*cp1 == '-') {			/* end time follows? */
	    sscanf (++cp1, "%d.%d", &end_hour, &end_min);
	    cp1 += 5;				/* to end of end time */
	  }
	  else {				/* no end time follows */
	    end_hour = beg_hour; end_min = beg_min;
	    if (*(++cp1) != '\t') {		/* not followed by tab? */
	      sprintf (message, "format error after time in \"%s\"",
	        appt_line);
	      warning (message);
	      continue;				/* to next appt line */
	    }
	  }
	}
	else {					/* no begin/end time */
	  beg_hour = -1; beg_min = -1;
	  end_hour = -1; end_min = -1;
	  if (*(++cp1) != '\t') { /* not two tabs? */
	    sprintf (message, "format error after date in \"%s\"",
	      appt_line);
	    warning (message);
	    continue;				/* to next appt line */
	  }
	}
	cp2 = strchr (++cp1, '\t');		/* skip tab, find next tab */
	if (cp2) {				/* alarm time present? */
	  len = cp2 - cp1;			/* set description length */
	  sscanf (++cp2, "%d.%d", &alm_hour, &alm_min);
	}
	else {					/* no alarm time */
	  len = strlen (cp1) - 1;		/* set description length */
	  alm_hour = -1; alm_min = -1;
	}
	if (*cp1 == '@') {			/* event is private? */
	  attrs |= dlpRecAttrSecret;		/* mark as private */
	  cp1++;				/* skip past marker */
	  len--;				/* decrement length */
	}
	cp2 = descr;				/* set destination pointer */
	while (len--) {				/* copy description */
	  *(cp2++) = *(cp1++);
	}
	*cp2 = '\0';				/* terminate description */
	if (!chk_appt_ent ()) {			/* check appt fields */
	  set_appt_ent (&appt);			/* set appt structure */
	  len = pack_Appointment (&appt, buff, BUFF_SIZE); /* pack appt */
	  pi_file_append_record (pf, buff, len, attrs, cat, uid);
	}
	else {					/* field range error */
	  sprintf (message, "field range error in \"%s\"", appt_line);
	  warning (message);
	}
      }
    }
    
    fclose (af);				/* close file */
  }
  else {					/* cannot open file */
      sprintf (message, "cannot read %s", cp1);
      fatal (message);
  }
}

/*** parse and write appointments ***/

void to_datebook ()
{
  struct pi_file *pf;				/* pilot file handle */
  struct DBInfo info;				/* database info */
  struct AppointmentAppInfo appt_info;		/* date app info */
  struct Appointment date;			/* appointment */
  char * name;					/* pilot file name */

  printf ("%s: converting to Pilot date book\n", progname);
  name = "DatebookDB.pdb";			/* set pilot file name */
  info.flags = 0X00;				/* set flags */
  if (pf = pi_file_create (name, &info)) {	/* open pilot file */
    set_appt_db_info (pf, &info);		/* set database info */
    set_appt_info (pf, &appt_info);		/* set app info */
    set_appt_sort_info (pf);			/* set sort info */
    set_appt_ents (pf, &info, &appt_info);	/* set date entires */
    if (pi_file_close (pf)) {			/* close pilot file? */
      sprintf (message, "cannot close %s", name);
      warning (message);
    }
  }
  else {					/* cannot open pilot file */
    sprintf (message, "cannot open %s", name);
    fatal (message);
  }
}

/******************************* Expenses *****************************/

/*** parse and write expenses ***/

void to_expense ()
{
  printf ("%s: converting to Pilot expenses", progname);
  printf (" (sorry - not yet implemented)\n");
}

/******************************* Memo Pad *****************************/

/*** set up memo app file information ***/

void set_memo_db_info (struct pi_file * pf, struct DBInfo * info)
{
  strcpy (info->name, "MemoDB");		/* set database name */
  info->flags = 0X00;				/* set flags */
  info->miscFlags = 0X00;			/* set miscellaneous flags */
  info->type = makelong ("DATA");		/* set data type */
  info->creator = makelong ("memo");		/* set data creator */
  info->version = 0;				/* set version (value?) */
  info->modnum = 0;				/* set modif. no. (value?) */
  info->createDate = time (NULL);		/* set created date */
  info->modifyDate = time (NULL);		/* set modified date */
  info->backupDate = -1;			/* set backup date */
  info->index = 0;				/* set index (value?) */
  if (pi_file_set_info (pf, info))		/* set database info */
    fatal ("cannot set memo file info");
}

/*** set memo category index from name ***/

int set_memo_cat (struct MemoAppInfo * memo_info, char * cat_name)
{
  int cat = 0;
  
  while ((cat < CAT_SIZE) &&			/* no category name match? */
   (strcmp (memo_info->category.name[cat], cat_name)))
    cat++;					/* to next category */
  if (cat == CAT_SIZE)				/* got to end of list? */
    cat = -1;					/* set invalid index */
  return (cat);					/* return category index */
}

/*** set up memo application information ***/

void set_memo_info (struct pi_file * pf, struct MemoAppInfo * memo_info)
{
  DIR * dir;					/* current directory */
  struct dirent * dirent;			/* current dir entry */
  int fld_ind;					/* category index */
  int len;					/* buffer length */
  int i, j;					/* counters */
  unsigned char buff[BUFF_SIZE];		/* data buffer */
  char cat_name[CAT_SIZE];			/* category name */
  char * cp1, * cp2;				/* character pointers */
  char ch;					/* current character */
  
  for (i = 0; i < CAT_SIZE; i++)		/* initialise cat names */
    memo_info->category.renamed[i] = 1;		/* set new cat flag */
  for (i = 0; i < CAT_SIZE; i++)		/* initialise cat names */
    for (j = 0; j < CAT_SIZE; j++)
      memo_info->category.name[i][j] = 0;
  strcpy (memo_info->category.name[0], "Unfiled"); /* set unfiled category */
  fld_ind = 1;					/* initialise cat. index */
  if (dir = opendir (".")) {			/* open current directory? */
    while (dirent = readdir (dir)) {		/* read directory entries */
      cp1 = dirent->d_name;			/* set filename */
      if (strstr (cp1, ".memo") &&		/* memo file? */
	    (*(cp1 + strlen (cp1) - 1) != '~')) { /* not "~" (backup) end? */
	if (cp2 = strchr (cp1, '-')) {		/* dash present? */
	  i = cp2 - cp1;			/* set cat. name length */
	  if (i >= (CAT_SIZE - 1))		/* category name too long? */
	    i = CAT_SIZE - 1;			/* truncate name */
	  for (j = 0; j < i; j++) {		/* copy category name */
	    ch = *(cp1++);			/* get category character */
	    if (ch == '_')			/* underscore to space */
	      ch = ' ';
	    cat_name[j] = ch;			/* copy character */
	  }
	  cat_name[j] = '\0';			/* terminate name */
	}
	else					/* no dash, default bus. */
	  strcpy (cat_name, "Business");
	if (set_memo_cat (memo_info, cat_name) == -1) { /* cat. name new? */
	  if (fld_ind < CAT_SIZE) {		/* not too many categories? */
	    strcpy (memo_info->category.name[fld_ind], cat_name);
	    fld_ind++;				/* to next category */
	  }
	  else {				/* too many categories */
	    sprintf (message, "more than %d memo categories", fld_ind);
	    fatal (message);
	  }
	}
      }
    }
  }
  else						/* cannot open current dir */
    fatal ("cannot read current directory");
  for (i = 0; i < CAT_SIZE; i++)		/* set cat ids */
    memo_info->category.ID[i] = i;
  memo_info->category.lastUniqueID = 6;		/* set last cat id */
  memo_info->sortByAlpha = 1;			/* alphabetical order */
  len = pack_MemoAppInfo (memo_info, buff, BUFF_SIZE); /* pack app info */
  if (pi_file_set_app_info (pf, buff, len))	/* set app info? */
    fatal ("cannot set memo app info");
}

/*** set up memo sorting information ***/

void set_memo_sort_info (struct pi_file * pf)
{
    pi_file_set_sort_info (pf, NULL, 0);	/* set sort info */
}

/*** set memo entry in preferred format ***/

void set_memo_ent (FILE * mf, struct Memo * memo, char * memo_name)
{
  int len;					/* memo length */
  static char memo_text[MEMO_SIZE];		/* memo line */
  char * cp;					/* character pointer */

  cp = memo_text;				/* initialise pointer */
  len = 0;					/* initialise memo length */
  while ((len++ < MEMO_SIZE) &&			/* memo size limit OK? */
    (*(cp++) = fgetc (mf)) != EOF) 		/* read memo character */
    ;
  *(--cp) = '\0';				/* replace EOF by NUL */
  if (len >= MEMO_SIZE) {
    sprintf (message, "%s exceeds %d characters",
      memo_name, MEMO_SIZE - 1);
    warning (message);
  }
  memo->text = memo_text;			/* copy memo text pointer */
}

/*** set memo entries in preferred format ***/

void set_memo_ents (struct pi_file * pf, struct DBInfo * info,
  struct MemoAppInfo * memo_info)
{
  DIR * dir;					/* current directory */
  FILE * mf;					/* input memo file */
  struct dirent * dirent;			/* current dir entry */
  struct Memo memo;				/* Pilot memo */
  int cat;					/* record category */
  int i, j;					/* counters */
  int len;					/* buffer length */
  int attrs;					/* attributes */
  pi_uid_t uid;					/* user id */
  unsigned char buff[BUFF_SIZE];		/* data buffer */
  char cat_name[CAT_SIZE];			/* category name */
  char * cp1, * cp2, * cp3;			/* character pointers */
  char ch;					/* current character */

  if (dir = opendir ("."))	{		/* open current directory? */
    while (dirent = readdir (dir)) {		/* read directory entries */
      cp1 = dirent->d_name;			/* set filename */
      if ((cp2 = strstr (cp1, ".memo")) && 	/* open memo file? */
           (*(cp1 + strlen (cp1) - 1) != '~')) { /* not "~" (backup) end? */
	if (mf = fopen (cp1, "r")) 	{	/* open file? */
	  attrs = dlpRecAttrDirty;		/* set record attributes */
          uid = 0;				/* set user id (value?) */
	  if (cp3 = strchr (cp1, '-')) {	/* dash present? */
	    i = cp3 - cp1;			/* set cat. name length */
	    if (i >= (CAT_SIZE - 1))		/* category name too long? */
	      i = CAT_SIZE - 1;			/* truncate name */
	    for (j = 0; j < i; j++) {		/* copy category name */
	      ch = *(cp1++);			/* get category character */
	      if (ch == '_')			/* underscore to space */
		ch = ' ';
	      cat_name[j] = ch; /* copy character */
	    }
	    cat_name[j] = '\0';			/* terminate name */
	  }
	  else					/* no dash, default bus. */
	    strcpy (cat_name, "Business");
	  cat = set_memo_cat (memo_info, cat_name);
	  set_memo_ent (mf, &memo, cp1); 		/* set memo entry */
	  cp2 += 5;				/* skip over ".memo" */
	  if (*cp2 == 's')			/* private memo? */
	    attrs |= dlpRecAttrSecret;		/* set secret flag */
	  len = pack_Memo (&memo, buff, BUFF_SIZE); /* pack memo */
	  pi_file_append_record (pf, buff, len, attrs, cat, uid);
	  fclose (mf);				/* close file */
	}
	else {					/* cannot open file */
            sprintf (message, "cannot read %s", cp1);
	    warning (message);
	}
      }
    }
  }
  else						/* cannot open current dir */
    fatal ("cannot read current directory");
}

/*** parse and write memos ***/

void to_memo ()
{
  struct pi_file *pf;				/* pilot file handle */
  struct DBInfo info;				/* database info */
  struct MemoAppInfo memo_info;			/* memo pad app info */
  struct Memo date;				/* memo */
  char * name;					/* pilot file name */

  printf ("%s: converting to Pilot memo pad\n", progname);
  name = "MemoDB.pdb";				/* set pilot file name */
  info.flags = 0X00;				/* set flags */
  if (pf = pi_file_create (name, &info)) {	/* open pilot file */
    set_memo_db_info (pf, &info);		/* set database info */
    set_memo_info (pf, &memo_info);		/* set app info */
    set_memo_sort_info (pf);			/* set sort info */
    set_memo_ents (pf, &info, &memo_info);	/* set memo entires */
    if (pi_file_close (pf)) {			/* close pilot file? */
      sprintf (message, "cannot close %s", name);
      warning (message);
    }
  }
  else {					/* cannot open pilot file */
    sprintf (message, "cannot open %s", name);
    fatal (message);
  }
}

/******************************* To Dos *****************************/

/*** set up todo app file information ***/

void set_todo_db_info (struct pi_file * pf, struct DBInfo * info)
{
  strcpy (info->name, "ToDoDB");		/* set database name */
  info->flags = 0X00;				/* set flags */
  info->miscFlags = 0X00;			/* set miscellaneous flags */
  info->type = makelong ("DATA");		/* set data type */
  info->creator = makelong ("todo");		/* set data creator */
  info->version = 0;				/* set version (value?) */
  info->modnum = 0;				/* set modif. no. (value?) */
  info->createDate = time (NULL);		/* set created date */
  info->modifyDate = time (NULL);		/* set modified date */
  info->backupDate = -1;			/* set backup date */
  info->index = 0;				/* set index (value?) */
  if (pi_file_set_info (pf, info))		/* set database info */
    fatal ("cannot set todo file info");
}

/*** set todo category index from name ***/

int set_todo_cat (struct ToDoAppInfo * todo_info, char * cat_name)
{
  int cat = 0;
  
  while ((cat < CAT_SIZE) &&			/* no category name match? */
   (strcmp (todo_info->category.name[cat], cat_name)))
    cat++;					/* to next category */
  if (cat == CAT_SIZE)				/* got to end of list? */
    cat = -1;					/* set invalid index */
  return (cat);					/* return category index */
}

/*** set up todo application information ***/

void set_todo_info (struct pi_file * pf, struct ToDoAppInfo * todo_info)
{
  unsigned char buff[BUFF_SIZE];		/* data buffer */
  int len;					/* buffer length */
  int i, j;					/* counters */
  
  for (i = 0; i < CAT_SIZE; i++)		/* initialise cat names */
    todo_info->category.renamed[i] = 1;		/* set new cat flag */
  for (i = 0; i < CAT_SIZE; i++)		/* initialise cat names */
    for (j = 0; j < CAT_SIZE; j++)
      todo_info->category.name[i][j] = 0;
  strcpy (todo_info->category.name[0], "Unfiled");
  strcpy (todo_info->category.name[1], "Business");
  strcpy (todo_info->category.name[2], "Personal");
  for (i = 0; i < CAT_SIZE; i++)		/* set cat ids */
    todo_info->category.ID[i] = i;
  todo_info->category.lastUniqueID = CAT_SIZE - 1; /* set last cat id */
  todo_info->dirty = 0XFFFF;			/* dirty flags */
  todo_info->sortByPriority = 0;		/* sort by priority */
  len = pack_ToDoAppInfo (todo_info, buff, BUFF_SIZE); /* pack app info */
  if (pi_file_set_app_info (pf, buff, len))	/* set app info? */
    fatal ("cannot set todo app info");
}

/*** set up todo sorting information ***/

void set_todo_sort_info (struct pi_file * pf)
{
    pi_file_set_sort_info (pf, NULL, 0);	/* set sort info */
}

/*** roughly check todo fields, returning 0/1 for OK/wrong ***/

int chk_todo_ent ()
{
  if ((1 <= day) &&				/* day in range? */
      strlen (descr))				/* description non-empty? */
    return (0);					/* checks are OK */
  else
    return (1);					/* checks fail */
}

/*** set todo entry in preferred format, returning 1 if OK ***/

int set_todo_ent (FILE * tf, struct ToDoAppInfo * todo_info,
  struct ToDo * todo, char * todo_line, int * cat, int * attrs)
{
  char todo_text[TODO_SIZE];			/* todo description */
  char month_name[LINE_SIZE];			/* month name */
  char * cp1, * cp2;				/* character pointers */
  char ch;					/* current character */

  if (*cat == -1) {				/* no record seen yet? */
    if (!strcmp (todo_line, "\n")) {		/* empty line? */
      *cat = 0;					/* start reading records */
    }
    return (0);					/* note todo record invalid */
  }
  if (!strcmp (todo_line, "\n"))		/* empty line? */
    return (0);					/* note todo record invalid */
  *cat = set_todo_cat (todo_info, "Business");	/* business cat. by default */
  todo->complete = 0;				/* not done by default */
  cp1 = todo_line;				/* initialise line pointer */
  while ((ch = *(cp1++)) != '\t') {		/* get todo status till tab */
    switch (tolower (ch)) {			/* check todo status */
      case '*':					/* business todo */
	todo->complete = 1;			/* set completed */
	break;
      case 'b':					/* business todo */
	*cat = set_todo_cat (todo_info, "Business"); /* set business category */
	break;
      case 'p':					/* personal todo */
	*cat = set_todo_cat (todo_info, "Personal"); /* set personal category */
	break;
      case 'u':					/* unfiled todo */
	*cat = set_todo_cat (todo_info, "Unfiled"); /* set unfiled category */
	break;
      case 's':					/* unfiled todo */
	*attrs |= dlpRecAttrSecret;		/* set secret flag */
	break;
      default:					/* unknown todo type */
	sprintf (message, "unknown todo status in", todo_line);
	warning (message);
    }
  }
  sscanf (cp1, "%d", &todo->priority);		/* get priority */
  cp1++;					/* to end of priority digit */
  if (*(cp1++) != '\t') {			/* not tab after priority? */
    sprintf (message, "format error after status in %s", todo_line);
    warning (message);
    return (0);					/* note todo record invalid */
  }
  if (*cp1 != '\t') {				/* due date present? */
    todo->indefinite = 0;			/* set has due date */
    sscanf (cp1, "%s %d %d",			/* read MMM DD YYYY */
      &month_name, &day, &year);
    cp1 += 11;					/* to end of MMM DD YYYY */
    month = 0;					/* search for month name */
    while ((month < 12) && strcmp (month_name, mon_name[month]))
      month++;
    if (month >= 12) {				/* name not found? */
      sprintf (message, "unrecognised month %s", month_name);
      warning (message);
      return (0);				/* note todo record invalid */
    }
    year -= 1900;				/* set years since 1900 */
    todo->due.tm_year = year;			/* set due date ... */
    todo->due.tm_mon = month;
    todo->due.tm_mday = day;
  }
  else {					/* no due date */
    todo->indefinite = 1;			/* set indefinite */
    cp1++;					/* skip tab */
  }
  cp1++;					/* skip tab */
  cp2 = todo_text;				/* initialise descr pointer */
  while ((ch = *(cp1++)) != '\n') {		/* get character till NL */
    if (ch == '~')				/* is tilde? */
      ch = '\n';				/* change to newline */
    *(cp2++) = ch;				/* store character */
  }
  *cp2 = '\0';					/* terminate descr string */
  todo->description = todo_text;		/* set description pointer */
  todo->note = NULL;				/* set note pointer */
  return (1);					/* note todo record valid */
}

/*** set todo entries in preferred format ***/

void set_todo_ents (struct pi_file * pf, struct DBInfo * info,
  struct ToDoAppInfo * todo_info)
{
  FILE * tf;					/* input todo file */
  struct ToDo todo;				/* Pilot todo */
  pi_uid_t uid;					/* user id */
  int attrs;					/* attributes */
  int cat;					/* record category */
  int i;					/* counter */
  int len;					/* buffer length */
  char todo_line[TODO_SIZE];			/* todo line */
  unsigned char buff[BUFF_SIZE];		/* data buffer */
  char * cp;					/* character pointer */

  cp = "ToDo";					/* set todo input filename */
  cat = -1;					/* mark no record seen yet */
  if (tf = fopen (cp, "r")) {			/* open todo file? */
    while (fgets (todo_line, TODO_SIZE, tf)) {	/* read till EOF */
      attrs = dlpRecAttrDirty;			/* set record attributes */
      uid = 0;					/* set user id (value?) */
      if (set_todo_ent (tf, todo_info, &todo,	/* set todo entry */
        todo_line, &cat, &attrs)) {
	len = pack_ToDo (&todo, buff, BUFF_SIZE); /* pack todo */
	pi_file_append_record (pf, buff, len, attrs, cat, uid);
      }
    }
    fclose (tf);				/* close file */
 }
  else {					/* cannot open file */
      sprintf (message, "cannot read %s", cp);
      warning (message);
  }
}

/*** parse and write to do list ***/

void to_todo ()
{
  struct pi_file *pf;				/* pilot file handle */
  struct DBInfo info;				/* database info */
  struct ToDoAppInfo todo_info;			/* todo pad app info */
  struct ToDo date;				/* todo */
  char * name;					/* pilot file name */

  printf ("%s: converting to Pilot todo list\n", progname);
  name = "ToDoDB.pdb";				/* set pilot file name */
  info.flags = 0X00;				/* set flags */
  if (pf = pi_file_create (name, &info)) {	/* open pilot file */
    set_todo_db_info (pf, &info);		/* set database info */
    set_todo_info (pf, &todo_info);		/* set app info */
    set_todo_sort_info (pf);			/* set sort info */
    set_todo_ents (pf, &info, &todo_info);	/* set todo entires */
    if (pi_file_close (pf)) {			/* close pilot file? */
      sprintf (message, "cannot close %s", name);
      warning (message);
    }
  }
  else {					/* cannot open pilot file */
    sprintf (message, "cannot open %s", name);
    fatal (message);
  }
}

/******************************* Main Program *****************************/

int main (int argc, char **argv)
{
  char * name;					/* local file name */
  struct DBInfo info;				/* database information */
  int arg_ind;					/* argument number */
  int opt;					/* option index */
  
  progname = argv[0];				/* get program name/options */
  while ((opt = getopt (argc, argv, "adehmt")) != EOF) {
    switch (opt) {
      case 'a':					/* address book */
        to_address ();				/* convert address book */
	break;
      case 'd':					/* date book */
        to_datebook ();				/* convert date book */
	break;
      case 'h':					/* help */
	usage ();
	break;
      case 'e':					/* expenses */
        to_expense ();				/* convert expenses */
	break;
      case 'm':					/* memo pad */
        to_memo ();				/* convert memos */
	break;
      case 't':					/* todo list */
        to_todo ();				/* convert todos */
	break;
      default:					/* unrecognised option */
	usage ();
    }
  }
  if (optind != argc) {				/* excess arguments? */
    warning ("only options may be given");
    usage ();
  }
  return (0);
}

/*************************************************************************/
