package uk.ac.stir.cs.PlugwiseDriver;

import java.text.SimpleDateFormat;

import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.io.IOException;

import java.util.Calendar;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Properties;
import java.util.TimeZone;

import javax.comm.*;

import org.osgi.framework.BundleActivator;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.framework.ServiceRegistration;

import org.osgi.service.event.EventAdmin;
import org.osgi.service.event.EventConstants;
import org.osgi.service.event.EventHandler;

import org.osgi.util.tracker.ServiceTracker;

import uk.ac.stir.cs.device.serial.SerialService;

/**
  Activator for PlugwiseDriver.

  @author	Kenneth J. Turner
  @version	1.0: 17/05/11 (KJT)
*/
public class Activator implements BundleActivator {

  /* ****************************** Constants ******************************* */

  /** Bundle properties file name */
  private static String BUNDLE_PROPERTIES = "PlugwiseDriver.properties";

  /** Policy action event topics */
  private final static String[] DEVICE_TOPICS = new String[] {
    "uk/ac/stir/cs/accent/device_out"
  };

  /* ****************************** Variables ******************************* */

  /** Action mapping (entity name/instance to Circle address) */
  private Hashtable<DeviceParameters, String> actionMapping =
    new Hashtable<DeviceParameters, String>();

  /** Policy action registration */
  private ServiceRegistration actionRegistration;

  /** Circle+ address (second part only) */
  private String controllerAddress = null;

  /** Interval (minutes) between polls for recent energy consumption */
  private int energyInterval;

  /** Whether an error is found during bundle start */
  private static boolean errorFound = false;

  /** Port entity */
  private String portEntity;

  /** Port name */
  private String portName;

  /** Interval (minutes) between polls for instantaneous power consumption */
  private int powerInterval;

  /** Event posting service */
  private PostEventService postEventService;

  /** Plugwise receiver instance */
  private PlugwiseTransceiver plugwiseTransceiver;

  /** The number of attempts allowed at sending a message */
  private int retryLimit;

  /** Serial port */
  private SerialPort serialPort;

  /** Sensor mapping (Circle address to entity name/instance) */
  private Hashtable<String, DeviceParameters> sensorMapping =
    new Hashtable<String, DeviceParameters>();

  /** First (and fixed) part of Plugwise addresses */
  private String startAddress;

  /** Service tracker */
  private ServiceTracker trackerEventAdmin;

  /* ******************************* Methods ******************************** */

  /**
    Return the current time as String yyyy-MM-dd HH:mm:ss.

    @return	time in format yyyy-MM-dd HH:mm:ss
  */
  public static String getTime() {
    Calendar cal = Calendar.getInstance(TimeZone.getDefault());
    String df = "yyyy-MM-dd HH:mm:ss";
    SimpleDateFormat sdf = new SimpleDateFormat(df);
    sdf.setTimeZone(TimeZone.getDefault());
    return sdf.format(cal.getTime());
  }

  /**
    Append error message to policy log.

    @param source	source of message
    @param message	message to log
  */
  public static void logError(String source, String message) {
    errorFound = true;				// note error found
    String text =
      getTime() + " [Error] PlugwiseDriver." + source + ": " + message;
    System.out.println(text);
  }

  /**
    Append error message to policy log.

    @param message  message to log
  */
  public static void logError(String message) {
    errorFound = true;				// note error found
    String text = getTime() + " [Error] PlugwiseDriver: " + message;
    System.err.println(text);
  }

  /**
    Append informational message to policy log.

    @param source	source of message
    @param message	message to log
  */
  public static void logNote(String source, String message) {
    String text =
      getTime() + " [Note] PlugwiseDriver." + source + ": " + message;
    System.out.println(text);
  }

  /**
    Append informational message to policy log.

    @param message  message to log
  */
  public static void logNote(String message) {
    String text = getTime() + " [Note] PlugwiseDriver: " + message;
    System.out.println(text);
  }

  /**
    Set PlugwiseDriver properties.

    @param context	bundle context
    @return		sensor mappings
    @exception		any exception
  */
  private void setProperties(BundleContext context) throws Exception {
    String osgiRoot = context.getBundle().getLocation();	// fallback
    String dataDirectory = context.getDataFile("").toString();
    if (dataDirectory != null) {
      int pos = dataDirectory.indexOf("osgi");
      if (pos != -1)
	osgiRoot = dataDirectory.substring(0, pos + 5);
    }
    String propertyFileName = osgiRoot + BUNDLE_PROPERTIES;
    Properties properties =  new Properties();
    try {
      File propertiesFile = new File(propertyFileName);
      InputStream propertyStream = new FileInputStream(propertiesFile);
      properties.load(propertyStream);
    }
    catch (IOException e) {
      logError("could not read properties file '" + propertyFileName + "'");
    }

    try {
      retryLimit =
	Integer.parseInt(properties.getProperty("plugwise.retry.limit"));
    }
    catch (NumberFormatException exception) {
      retryLimit = 3;
      logError("property 'plugwise.retry.limit' must be an integer");
    }
    if (retryLimit <= 0)
      logError("property 'plugwise.retry.limit' must be greater than 0");

    try {
      energyInterval =
	Integer.parseInt(properties.getProperty("plugwise.energy.interval"));
    }
    catch (NumberFormatException exception) {
      energyInterval = 60;
      logError("property 'plugwise.energy.interval' must be an integer");
    }
    if (energyInterval <= 0)
      logError("property 'plugwise.energy.interval' must be greater than 0");

    try {
      powerInterval =
	Integer.parseInt(properties.getProperty("plugwise.power.interval"));
    }
    catch (NumberFormatException exception) {
      powerInterval = 10;
      logError("property 'plugwise.power.interval' must be an integer");
    }
    if (powerInterval <= 0)
      logError("property 'plugwise.power.interval' must be greater than 0");

    startAddress = properties.getProperty("plugwise.address.start");
    portName = properties.getProperty("plugwise.port");
    portEntity = properties.getProperty("plugwise.entity");
    if (portName == null || portEntity == null || startAddress == null)
      logError(
	"properties 'plugwise.entity', 'plugwise.port' and " +
	"'plugwise.address.start' needed");

    Enumeration<?> propertyNames = properties.propertyNames();
    while (propertyNames.hasMoreElements()) {
      String property = ((String) propertyNames.nextElement()).trim();
      if (!property.startsWith("plugwise.")) {
	String value = properties.getProperty(property).trim();
	if (property.matches("^[0-9A-F]+\\+?$") &&
	    value.matches("^[^,]+,[^,]+$")) {
	  String sensorId = property.trim().toUpperCase();
	  int position = sensorId.indexOf('+');	// check for Circle+ suffix
	  if (position != -1) {			// Circle+ suffix found?
	    sensorId =				// get sensor id without '+'
	      sensorId.substring(0, position);
	    if (controllerAddress == null)	// Circle+ address unknown?
	      controllerAddress = sensorId;	// set Circle+ address
	    else
	      logError("duplicate Circle+ address '" + sensorId + "'");
	  }
	  String[] triggerValues = value.split("\\s*,\\s*");
	  String entityName = triggerValues[0].trim();
	  String entityInstance = triggerValues[1].trim();
	  DeviceParameters triggerParameters = new DeviceParameters(
	    entityName, entityInstance);
	  sensorMapping.put(sensorId, triggerParameters);
	  actionMapping.put(triggerParameters, sensorId);
	}
	else
	  logError("invalid mapping: " + property + " = " + value);
      }
    }
    if (controllerAddress == null)	// Circle+ address unknown?
      logError("Circle+ address not defined");
  }

  /**
    Start PlugwiseDriver bundle.

    @param context	bundle context
    @exception		any exception
  */
  public void start(BundleContext context) throws Exception {
     // set up event admin service
    trackerEventAdmin =
      new ServiceTracker(context, EventAdmin.class.getName(), null);
    trackerEventAdmin.open();
    EventAdmin eventAdminService =
      (EventAdmin) trackerEventAdmin.getService();

    // could use EventAdminPostEventService or MessageBrokerPostEventService
    postEventService = new EventAdminPostEventService(eventAdminService);

    // set up properties
    setProperties(context);
    if (!errorFound) {				// no error found?
      // set up serial port
      plugwiseTransceiver = new PlugwiseTransceiver(
	portName, startAddress, controllerAddress, retryLimit, energyInterval,
	powerInterval, sensorMapping, postEventService);

    // set up action handler
    PlugwiseAction actionHandler =
      new PlugwiseAction(context, plugwiseTransceiver, actionMapping);
    Hashtable<String, Object> dictionary = new Hashtable<String, Object>();
    dictionary.put(EventConstants.EVENT_TOPIC, DEVICE_TOPICS);
    actionRegistration = context.registerService(EventHandler.class.getName(),
      actionHandler, dictionary);

      // initialise serial port
      ServiceReference[] serviceReferences = context.getServiceReferences(
	"uk.ac.stir.cs.device.serial.SerialService", "(Port=" + portName + ")");
      if (serviceReferences != null && serviceReferences.length == 1) {
	  ServiceReference serviceReference = serviceReferences[0];
	  SerialService serialService =
	    (SerialService) context.getService(serviceReference);
	  serialPort = serialService.getPort();
	  if (!plugwiseTransceiver.initialise(portEntity, serialPort)) {
	    plugwiseTransceiver.close();
	    logError("could not initialise port - try restarting CommAccess");
	  }
	  else
	    logNote("started port " + portName);
      }
      else
	logError("port " + portName + " not found - start CommAccess");
    }
  }

  /**
    Stop PlugwiseDriver bundle.

    @param context	bundle context
    @exception		any exception
  */
  public void stop(BundleContext context) throws Exception {
    if (serialPort != null)
      plugwiseTransceiver.close();
    actionRegistration.unregister();
    logNote("stopped port " + portName);
  }

}
