package uk.ac.stir.cs.PlugwiseDriver;

import java.io.IOException;			// import input-output exception
import java.io.InputStream;			// import input stream
import java.io.OutputStream;			// import input stream

import java.util.Calendar;			// import calendar
import java.util.Enumeration;			// import enumeration
import java.util.Hashtable;			// import hash table
import java.util.Timer;				// import timer
import java.util.TimerTask;			// import timer task
import java.util.Vector;			// import vector

import javax.comm.*;				// import communications port

import org.osgi.service.event.EventConstants;	// import OSGI event constants

/**
  This class supports Plugwise Circles connected via a Stick. The Plugwise
  network should (perhaps) have first been set up using the Source. The
  following limitations or uncertainties apply:

  <ul>

    <li>
      Various aspects of the protocol and its format are not yet known (to me or
      others).
    </li>

    <li>
      Sequence numbers from the Circle+/Circle are not used. Instead commands
      are sent one at a time, waiting for the response.
    </li>

    <li>
      It is unclear whether it is sufficient to periodically set the Circle+
      clock, or whether all Circle clocks need to be set. This is presumably OK
      since the Plugwise documentation says that it is the Circle+ that has the
      clock. It appears that Circles synchronise clocks with the Circle+ on the
      hour. It appears that clocks follow UTC, and have to be read or written by
      adjusting local time.
    </li>

    <li>
      A Circle sometimes refuses to respond to a valid request for a power
      buffer, as if the current buffer were locked. This problem goes away after
      the next buffer is written on the following hour. This may be related to
      using the Source in between runs of this bundle. The Source successfully
      reads the buffer even though the bundle does not.
    </li>

  </ul>

  <p>
    Circles are periodically polled to perform the following tasks:
  </p>

  <ul>

    <li>
      The Circle+ clock is set every 24 hours to the PC clock, starting
      immediately after the bundle is started.
    </li>

    <li>
      The Circle energy consumption for the past hour is retrieved at 1 minute
      past the first hour after the bundle is started.
    </li>

    <li>
      The Circle instantaneous power consumption is retrieved at the given
      interval in minutes, starting at this interval after the bundle is
      started.
    </li>

  </ul>

  <p>
    The following kinds of Accent events are supported:
  </p>

  <ul>

    <li>
      <var>device_in(energy,lamp,lounge,14:00:00,70)</var> means 70 watt-hours
      have been consumed by the lounge lamp in the hour starting at 2PM. The
      total of all energy readings has the form
      <var>device_in(energy,,,14:00:00,130)</var>.
    </li>

    <li>
      <var>device_in(power,lamp,lounge,,110)</var> 110 watts are currently being
      consumed by the lounge lamp. The total of all power readings has the form
      <var>device_in(power,,,,650)</var>.
    </li>

    <li>
      <var>device_out(on,lamp,lounge)</var> switches the lounge lamp on, while
      "device_out(off,lamp,lounge)" switches it off. Any message type apart from
      <var>on</var> and <var>off</var> is an error. If the entity name/instance
      are not defined in the mapping given by the property files, the event is
      ignored.
    </li>

  </ul>

  <p>
    The protocol implementation was inspired by information from <a
    href="http://www.maartendamen.com/category/plugwise-unleashed/">Maarten
    Damen</a> and <a
    href="http://roheve.wordpress.com/2011/04/24/plugwise-protocol-analyse/">Roheve</a>.
    Inspiration was also drawn from code by <a
    href="https://github.com/gonium/libplugwise">Gonium</a> and by <a
    href="https://bitbucket.org/hadara/python-plugwise/wiki/Home">Sven
    Petai</a>. The CRC calculation was adapted from code by <a
    href="http://www.domoticaforum.eu/viewtopic.php?f=39&t=5803">Tiz</a>.
  </p>

  @author	Kenneth J. Turner
  @version	1.0: 17/05/11 (KJT)
*/
public class PlugwiseTransceiver {

  /* ****************************** Constants ******************************* */

  /** Plugwise Stick bit rate (bps) */
  private final static int BIT_RATE = 115200;

  /** Trigger message type for clock setting */
  private final static String CLOCK_TASK = "clock";

  /** CRC polynomial (x16 + x12 + x5 + 1) */
  private static int CRC_POLYNOMIAL = 0x1021;

  /** Whether to produce debug output for error situations */
  private static boolean ERROR_DEBUG = false;

  /** Trigger message type for energy input */
  private final static String ENERGY_TASK = "energy";

  /** Event topic for receiver input events */
  public final static String EVENT_TOPIC =
    "uk/ac/stir/cs/accent/device_in";

  /** Trigger message type for power input */
  private final static String POWER_TASK = "power";

  /* ************************* Protocol Constants *************************** */

  /** Circle+ general response code (hex */
  private final static String GENERAL_RESPONSE = "0000";

  /** Circle clock get request code (hex) */
  private final static String CLOCK_GET_REQUEST = "003E";

  /** Circle clock get response code (hex) */
  private final static String CLOCK_GET_RESPONSE = "003F";

  /** Circle clock set request code (hex) */
  private final static String CLOCK_SET_REQUEST = "0016";

  /** Circle+ clock set response code (hex) */
  private final static String CLOCK_SET_RESPONSE = GENERAL_RESPONSE;

  /** Circle calibration request code (hex) */
  private final static String DEVICE_CALIBRATION_REQUEST = "0026";

  /** Circle calibration response code (hex) */
  private final static String DEVICE_CALIBRATION_RESPONSE = "0027";

  /** Circle device information request code (hex) */
  private final static String DEVICE_INFORMATION_REQUEST = "0023";

  /** Circle device information response code (hex) */
  private final static String DEVICE_INFORMATION_RESPONSE = "0024";

  /** Circle power buffer request code (hex) */
  private final static String POWER_BUFFER_REQUEST = "0048";

  /** Circle power buffer response code (hex) */
  private final static String POWER_BUFFER_RESPONSE = "0049";

  /** Circle power change (i.e. on/off) request code (hex) */
  private final static String POWER_CHANGE_REQUEST = "0017";

  /** Circle+ power change (i.e. on/off) response code (hex)  */
  private final static String POWER_CHANGE_RESPONSE = GENERAL_RESPONSE;

  /** Circle power information request code (hex) */
  private final static String POWER_INFORMATION_REQUEST = "0012";

  /** Circle power information response code (hex) */
  private final static String POWER_INFORMATION_RESPONSE = "0013";

  /** Circle+ acknowledgement code (hex) */
  private final static String PROTOCOL_ACK = "00C1";

  /** Plugwise protocol header code (hex) */
  private final static String PROTOCOL_HEADER = "\u0005\u0005\u0003\u0003";

  /** Plugwise protocol trailer code (hex) */
  private final static String PROTOCOL_TRAILER = "\r\n";

  /** Amount by which to multiple pulse count to get watts */
  private final static float PULSE_FACTOR = 2.1324759f;

  /** Delay before expecting a protocol response (seconds) */
  private final static int RECEIVE_DELAY = 1;

  /** Delay (seconds) before assuming no further response data will arrive */
  private final static int RECEIVE_TIMEOUT = 2;

  /** Stick initialisation request code (hex) */
  private final static String STICK_INITIALISE_REQUEST = "000A";

  /** Stick initialisation response code (hex) */
  private final static String STICK_INITIALISE_RESPONSE = "0011";

  /* ****************************** Variables ******************************* */

  /** Default calibration if not available */
  private Calibration calibrationDefault =
    new Calibration(1.0f, 0.0f, 0.0f, 0.0f);

  /** Sensor mapping */
  private Hashtable<String, Calibration> calibrationMapping =
    new Hashtable<String, Calibration>();

  /** Clock offset for zone and daylight saving time (minutes) */
  private int clockOffset;

  /** Clock setting timer task */
  private PollTask clockTask = null;

  /** Circle+ address (second part only) */
  private String controllerAddress;

  /** CRC table */
  private int[] crcTable;

  /** Interval (minutes) between polls for recent energy consumption */
  private int energyInterval;

  /** Energy poll timer task */
  private PollTask energyTask = null;

  /** Plugwise input stream */
  private InputStream inputStream;

  /** Plugwise output stream */
  private OutputStream outputStream;

  /** Polling timer */
  private Timer pollTimer = null;

 /** Plugwise port entity */
  private String portEntity = null;

  /** Plugwise port name */
  private String portName;

  /** Post event service */
  private PostEventService postEventService;

  /** Interval (minutes) between polls for instantaneous power consumption */
  private int powerInterval;

  /** Power poll timer task */
  private PollTask powerTask = null;

  /** The number of attempts allowed at sending a message */
  private int retryLimit;

  /** Sensor mapping */
  private Hashtable<String, DeviceParameters> sensorMapping =
    new Hashtable<String, DeviceParameters>();

  /** Plugwise serial port */
  private SerialPort serialPort = null;

  /** First (and fixed) part of Plugwise addresses */
  private String startAddress;

  /* ******************************* Methods ******************************** */

  /**
    Constructor for a PlugwiseTransceiver.

    @param portName		serial port name
    @param startAddress		first part of Plugwise addresses
    @param controllerAddress	second part of Circle+ address
    @param retryLimit		retry limit
    @param energyInterval	energy poll interval (minutes)
    @param powerInterval	power poll interval (minutes)
    @param sensorMapping	sensor mapping
    @param postEventService	OSGi event service
   */
  public PlugwiseTransceiver(String portName, String startAddress,
   String controllerAddress, int retryLimit, int energyInterval, int
   powerInterval, Hashtable<String, DeviceParameters> sensorMapping,
   PostEventService postEventService) {
    this.portName = portName;
    this.startAddress = startAddress;
    this.controllerAddress = controllerAddress;
    this.energyInterval = energyInterval;
    this.powerInterval = powerInterval;
    this.retryLimit = retryLimit;
    this.sensorMapping = sensorMapping;
    this.postEventService = postEventService;
    crcTable = getCRCTable();
    Calendar calendar = Calendar.getInstance();	// get current date/time
    clockOffset =				// get clock offset (minutes)
      (calendar.get(Calendar.ZONE_OFFSET) + calendar.get(Calendar.DST_OFFSET)) /
	(60 * 1000);
  }

  /** Close the serial port streams and remove the listener. */
  public void close() {
    if (serialPort != null && inputStream != null && outputStream != null) {
      try {
	inputStream.close();
	outputStream.close();
      }
      catch (IOException exception) {
	Activator.logError(
	  "close", "I/O error on port '" + portName + "' - " + exception);
      }
      try {
	serialPort.removeEventListener();
      }
      catch (IllegalStateException exception) {
	Activator.logError("close", "port " + portName + " already closed");
      }
    }
    if (clockTask != null && energyTask != null && powerTask != null) {
      clockTask.cancel();			// cancel clock task
      energyTask.cancel();			// cancel energy task
      powerTask.cancel();			// cancel power task
    }
    if (pollTimer != null)
      pollTimer.cancel();				// cancel poll timer
  }

  /**
    Check received data, returning the payload if the response is valid (or null
    if not). Global "circleAddress" is set to the address of the responding
    Circle.

    @param command	command for which a response is expected
    @param buffer	string of characters
    @return		payload for valid response (null if invalid)
  */
  public String checkData(String command, String buffer) {
// System.err.println("buffer <" + buffer + ">");	// TEMP
// for (int i = 0; i < buffer.length(); i++)
//   System.err.print(String.format("%04X ", ((int) buffer.charAt(i))));
// System.err.println();
    String response = null;			// initialise response
    String code = GENERAL_RESPONSE;		// set Circle+ response code
    String header = PROTOCOL_HEADER + code;	// set response header
    int position1 = buffer.indexOf(header);	// check if response found
// System.err.println("code1 <" + code + "> position1 <" + position1 + ">");	// TEMP
    if (position1 != -1) {			// response header found?
      code =					// get acknowledgement code
	buffer.substring(position1 + 12, position1 + 16);
// System.err.println("code2 <" + code + ">");	// TEMP
      if (code.equals(PROTOCOL_ACK)) {		// Circle+ acknowledgement?
	code = responseCode(command);		// set command response code
	header = PROTOCOL_HEADER + code;	// set response header
	if (!code.equals(GENERAL_RESPONSE))	// not general response code?
	  position1 =				// check after Circle+ response
	    buffer.indexOf(header, position1 + 20);
// System.err.println("code3 <" + code + "> position1 <" + position1 + ">");	// TEMP
	if (position1 != -1) {			// response header found?
	  position1 += PROTOCOL_HEADER.length();// move past header to content
	  int position2 =			// get trailer start
	    buffer.indexOf(PROTOCOL_TRAILER, position1);
	  if (position2 != -1) {		// trailer found?
	    position2 -= 4;			// move to before CRC
	    String crcExpected =		// get expected CRC
	      getCRCString(buffer.substring(position1, position2));
	    String message =			// get message
	      buffer.substring(position1, position2);
	    position1 += code.length();		// move to payload start
	    String content =			// get payload
	      buffer.substring(position1, position2);
	    String crcActual =			// get actual CRC
	      buffer.substring(position2, position2 + 4);
	    if (crcExpected.equals(crcActual)) {// CRCs match?
	      response = content;		// store response
	      if (ERROR_DEBUG)			// debug
		Activator.logNote(		// log received message
		  "checkData", "receiving '" + formatBuffer(message) + "'");
	    }
	    else if (ERROR_DEBUG)		// CRC mismatch, debug?
	      Activator.logNote("checkData", "protocol response to " +
		commandName(command) + " ignored due to invalid checksum");
	  }
	  else if (ERROR_DEBUG)			// trailer not found, debug?
	    Activator.logError("checkData", "protocol response to " +
	      commandName(command) + "lacks trailer");
	}
	else if (ERROR_DEBUG)			// no response header, debug?
	  Activator.logError("checkData", "protocol response header for " +
	    commandName(command) + " not received");
      }
      else if (ERROR_DEBUG)			// no Circle+ ack, debug?
	Activator.logError("checkData", "protocol response to " +
	  commandName(command) + " is negative");
    }
    else if (ERROR_DEBUG)			// no response header, debug?
      Activator.logError("checkData", "protocol response header for " +
	commandName(command) + " not received");
    return(response);				// return response
  }

  /**
    Return a user-friendly name for a command code.

    @param command	command code
    @return		user-friendly name for command
  */
  public String commandName(String command) {
    String name = "?";				// initialise command name
    if (command.equals(POWER_BUFFER_REQUEST))	// power buffer request?
      name = "power buffer request";		// set name
    else if (command.equals(			 // calibration request?
	       DEVICE_CALIBRATION_REQUEST))
      name = "calibration request";		// set name
    else if (command.equals(			// device information request?
	       DEVICE_INFORMATION_REQUEST))
      name = "device information request";	// set name
    else if (command.equals(CLOCK_GET_REQUEST))	// clock get request?
      name = "clock get request";		// set name
    else if (command.equals(CLOCK_SET_REQUEST))	// clock set request?
      name = "clock set request";		// set name
    else if (command.equals(POWER_CHANGE_REQUEST)) // power change request?
      name = "power change request";		// set name
    else if (command.equals(			// power information request?
	       POWER_INFORMATION_REQUEST))
      name = "power information request";	// set name
    else if (command.equals(			// stick initialisation request?
	       STICK_INITIALISE_REQUEST))
      name = "stick initialisation request";	// set name
    return(name);				// return command name
  }

  /**
    Return a user-friendly name for a device given the Circle address.

    @param address	circle address (second part only)
    @return		user-friendly name for Circle
  */
  public String deviceName(String address) {
    String name = "unknown device";		// initialise device name
    if (address != null) {			// known address?
      DeviceParameters trigger =		// get trigger for address
	sensorMapping.get(address);
      if (trigger != null) {			// address found?
	name =					// get "entity in instance"
	  trigger.getEntityName() + " in " + trigger.getEntityInstance();
      }
    }
    else					// unknown address
      name = "Circle+";				// must be Circle+
    return(name);				// return user-friendly address
  }

  /**
    Convert a string of characters into a string of hex digits in groups of
    four.

    @param buffer	string of characters
    @return		string of hex digits
  */
  public String formatBuffer(String buffer) {
    String hexString = "";
    if (buffer != null) {
      for (int i = 0; i < buffer.length(); i++) {
	if (i > 0 && i % 4 == 0)
	  hexString += " ";
	hexString += buffer.charAt(i);
      }
    }
    return(hexString);
  }

  /**
    Return stored calibration for Circle. If the stored value is unknown or the
    default, try to retrieve it again.  Use the default calibration if it cannot
    be retrieved.

    @param circleAddress	circle address
  */
  public Calibration getCalibration(String circleAddress) {
    Calibration calibration =			// get calibration
      calibrationMapping.get(circleAddress);
    if (calibration == null ||			// undefined calibration or ...
	calibration.equals(calibrationDefault)) { // default calibration?
      calibration =				// get calibration
	requestCalibration(circleAddress);
      if (calibration == null)			// calibration not obtained?
	calibration = calibrationDefault;	// use default calibration
    }
    return(calibration);
  }

  /**
    Return CRC for a hext string as a hex string.

    @param buffer	send/receive buffer
    @return		CRC as hex string
  */
  public String getCRCString(String buffer) {
    int work = 0x0000;
    byte[] bytes = buffer.getBytes();
    // loop, calculating CRC for each byte of the string
    for (byte b : bytes)
      // xor next byte with high byte so far to look up CRC table
      // xor that with low byte so far and mask back to 16 bits
      work =
	(crcTable[(b ^ (work >>> 8)) & 0xff] ^ (work << 8)) & 0xffff;
    return(Integer.toHexString(work).toUpperCase());
  }

  /**
    Return the CRC table.

    @return	CRC table
  */
  private int[] getCRCTable() {
    int[] crcTable = new int[256];
    for (int i = 0; i < 256; i++) {
      int fcs = 0;
      int d = i << 8;
      for (int k = 0; k < 8; k++) {
	if (((fcs ^ d) & 0x8000) != 0)
	  fcs = (fcs << 1) ^ CRC_POLYNOMIAL;
	else
	  fcs = (fcs << 1);
	d <<= 1;
	fcs &= 0xffff;
      }
      crcTable[i] = fcs;
    }
    return(crcTable);
  }

  /**
    Return a float from the given hex digit string.

    @param buffer	receive buffer
    @param start	start of float
    @param length	length of float (8 hex digits)
    @return		converted float
  */
  private float getFloat(String buffer, int start, int length) {
    float number = 0.0f;
    if (length == 8) {
      int floatBits = getInt(buffer, start, length);
      number = Float.intBitsToFloat(floatBits);
    }
    else
      Activator.logError("getFloat", "float length for '" + length + "' not 8");
    return(number);
  }

  /**
    Return an integer from the given hex digit string.

    @param buffer	receive buffer
    @param start	start of integer
    @param length	length of integer (1 to 8 hex digits)
    @return		converted integer
  */
  private int getInt(String buffer, int start, int length) {
    int number = 0;
    if (1 <= length && length <= 8) {
      String numberString = buffer.substring(start, start + length);
      try {
	long numberLong =			// parse as long for 16 bits
	  Long.parseLong(numberString, 16);
	number = (int) (numberLong & 0XFFFFFFFF);
      }
      catch (NumberFormatException exception) {
	Activator.logError(
	  "getInt", "incorrect format for integer '" + numberString + "'");
      }
    }
    else
      Activator.logError(
	"getInt", "integer length for '" + length + "' not 1 to 8");
    return(number);
  }

  /**
    Return power consumption in watts for the given Circle calibration and pulse
    count measured over some period.

    @param calibration	Circle calibration settings
    @param pulses	pulse count over period
    @param period	period in seconds
    @return		corrected pulse count
  */
  public float getPower(Calibration calibration, int pulses,
   int period) {
    float power = 0.0f;				// assume no power consumed
    if (period > 0) {				// non-zero period?
      if (pulses != 0 &&			// meaningful pulse count?
	  pulses != 65535 && pulses != -1) {
	float gainA = calibration.getGainA();	// get Circle gains and offsets
	float gainB = calibration.getGainB();
	float offsetNoise = calibration.getOffsetNoise();
	float offsetTotal = calibration.getOffsetTotal();
	float pulsesAverage =			// get pulses per second
	  ((float) pulses) / period;
	float pulsesOffsetNoise =		// correct pulses for noise
	  pulsesAverage + offsetNoise;
	float pulsesCorrected =			// correct pulses for gains
	  pulsesOffsetNoise * pulsesOffsetNoise * gainB +
	  pulsesOffsetNoise * gainA + offsetTotal;
	power = PULSE_FACTOR * pulsesCorrected; // get power in watts
      }
    }
    else					// zero period
      Activator.logError(
	"getPower", "period '" + period + "' must be positive");
    return(power);
  }

  /**
    Return a time HH:MM:SS from the given minutes relative to midnight of the
    start of the current month.

    @param minutes	minutes relative to month start
    @return		converted time (HH:MM:SS)
  */
  private String getTime(int minutes) {
    minutes = minutes % (24 * 60);		// get minutes from day start
    int hour = minutes / 60;			// get hours
    int minute = minutes % 60;			// get minutes from hour start
    String time = String.format("%02d:%02d:00", hour, minute);
  return(time);
  }

  /**
    Return a time HH:MM:SS from the given hour, minute and second, adjusted by
    adding the given offset in minutes.

    @param hour		hour
    @param minute	minute
    @param second	second
    @param offset	offset for time zone/daylight saving (minutes)
    @return		adjusted time (HH:MM:SS)
  */
  private String getTime(int hour, int minute, int second, int offset) {
    Calendar calendar = Calendar.getInstance(); // get current date/time
    calendar.set(Calendar.HOUR_OF_DAY, hour);	// set hour
    calendar.set(Calendar.MINUTE, minute);	// set minute
    calendar.add(Calendar.MINUTE, offset);	// add clock offset in minutes
    hour = calendar.get(Calendar.HOUR_OF_DAY);	// get hour
    minute = calendar.get(Calendar.MINUTE);	// get minute
    String time = String.format("%02d:%02d:%02d", hour, minute, second);
    return(time);				// return time
  }

  /**
    Initialise the serial port used by the Plugwise receiver.

    @param portEntity	entity to own input events
    @param serialPort	serial port
    @return		true if initialisation of the serial port succeeds
   */
  public boolean initialise(String portEntity, SerialPort serialPort) {
    this.portEntity = portEntity;
    this.serialPort = serialPort;
    boolean result = false;

    try {
      inputStream = serialPort.getInputStream();
      outputStream = serialPort.getOutputStream();
      serialPort.setSerialPortParams(BIT_RATE,
	SerialPort.DATABITS_8,
	SerialPort.STOPBITS_1,
	SerialPort.PARITY_NONE);
      serialPort.enableReceiveTimeout(1000 * RECEIVE_TIMEOUT);
    }
    catch (IOException exception1) {
      Activator.logError(
	"initialise", "I/O error on port '" + portName + "' - " + exception1);
      serialPort.close();
      return(result);
    }
    catch (IllegalStateException exception2) {
      Activator.logError(
	"initialise", "port '" + portName +
	"' already closed - restart CommAccess");
      return(result);
    }
    catch (UnsupportedCommOperationException exception4) {
      Activator.logError(
	"initialise", "port '" + portName + "' unsupported operation");
      return(result);
    }

    serialPort.setDTR(false);
    serialPort.setRTS(false);
    result = initialisePort();
    return(result);
  }

  /**
    Initialise port and Stick.

    @return	true if initialisation port and Plugwise Stick succeeds
  */

  public boolean initialisePort() {
    boolean online = false;			// assume offline
    try {					// try to communicate
      if (inputStream.available() > 0) {	// left-over input?
	if (ERROR_DEBUG)			// debug?
	  Activator.logNote(
	    "initialisePort", "discarding old data from port '" + portName +
	    "'");
	while (inputStream.available() > 0)	// while left-over data
	  inputStream.read();			// read and ignore it
      }
      online = requestInitialisation();		// initialise Stick
      if (online)				// network online?
	startThreads();				// start calibration/polling
      else					// network offline
	Activator.logError("initialisePort", "Plugwise network offline");
    }
    catch (IOException exception) {
      Activator.logError("initialisePort", "I/O exception - " + exception);
    }
    return(online);				// return online result
  }

  /**
    Poll all Circles for the given task type, posting events for energy/power
    consumption:

    <ul>

      <li>The Circle clock is set to the current PC time.</li>

      <li>
	The Circle energy consumption for the past hour is retrieved. An energy
	event trigger is then sent with parameters such as
	"energy,lamp,lounge,14:00:00,70", meaning 70 watt-hours consumed by the
	lounge lamp in the hour leading up to 3PM. The total of all energy
	readings is then sent in the form "energy,,,14:00:00,130".
      </li>

      <li>
	The Circle instantaneous power consumption is retrieved. A power event
	trigger is then sent with parameters such as ""power,lamp,lounge,,110",
	meaning 110 watts are currently being consumed by the lounge lamp. The
	total of all power readings is then sent in the form "power,,,,250".
      </li>

    </ul>

    @param taskType	type of task (clock, energy, power)
  */
  public void pollCircles(String taskType) {
    if (taskType.equals(CLOCK_TASK)) {		// clock task?
      requestSetting(controllerAddress, null);	// set Circle+ clock
    }
    else if (taskType.equals(ENERGY_TASK)) {	// energy task?
      String messageType = taskType;		// set message type
      String messagePeriod = "";		// initialise message period
      String parameterValues = "";		// initialise parameter values
      int total = 0;				// initialise total energy
      Enumeration<String> circleAddresses =	// get Circle addresses
	sensorMapping.keys();
      while (circleAddresses.hasMoreElements()) { // go through Circles
	String circleAddress = circleAddresses.nextElement();
	DeviceParameters trigger =		// get trigger for address
	  sensorMapping.get(circleAddress);
	String entityName = trigger.getEntityName();
	String entityInstance = trigger.getEntityInstance();
	Energy energy = requestEnergy(circleAddress);
	if (energy != null) {			// energy reading retrieved?
	  messagePeriod = energy.getTime();	// get time of reading
	  int consumption = energy.getEnergy();	// get consumption
	  total += consumption;			// update total consumption
	  parameterValues =			// get consumption as string
	    Integer.toString(consumption);
	  postEvent(messageType, entityName, entityInstance, messagePeriod,
	    parameterValues);
	}
      }
      // the following assumes that the last period makes sense for the total
      parameterValues = Integer.toString(total);// get total consumption
      postEvent(messageType, "", "", messagePeriod, Integer.toString(total));
    }
    else if (taskType.equals(POWER_TASK)) {	// power task?
      String messageType = taskType;		// set message type
      String messagePeriod = "";		// initialise message period
      String parameterValues = "";		// initialise parameter values
      int total = 0;				// initialise total power
      Enumeration<String> circleAddresses =	// get Circle addresses
	sensorMapping.keys();
      while (circleAddresses.hasMoreElements()) { // go through Circles
	String circleAddress = circleAddresses.nextElement();
	DeviceParameters trigger =		// get trigger for address
	  sensorMapping.get(circleAddress);
	String entityName = trigger.getEntityName();
	String entityInstance = trigger.getEntityInstance();
	int consumption = requestPower(circleAddress); // get power
	if (consumption >= 0) {			// power reading retrieved?
	  messagePeriod = "";			// empty period means "now"
	  total += consumption;			// update total consumption
	  parameterValues =			// get consumption as string
	    Integer.toString(consumption);
	  postEvent(messageType, entityName, entityInstance, messagePeriod,
	    parameterValues);
	}
      }
      // the following assumes that the last period makes sense for the total
      parameterValues = Integer.toString(total);// get total consumption
      postEvent(messageType, "", "", messagePeriod, parameterValues);
    }
    else					// unknown task
      Activator.logError("pollCircles", "unknown task type '" + taskType + "'");
  }

  /**
    Post an event based on the trigger parameter.

    @param messageType		message type
    @param entityName		entity name
    @param entityInstance	entity instance
    @param messagePeriod	message period
    @param parameterValues	parameter values
   */
  public void postEvent(String messageType, String entityName,
   String entityInstance, String messagePeriod, String parameterValues) {
    // event.topics automatically added to event property
    Hashtable<String, Object> eventProperties = new Hashtable<String, Object>();

    eventProperties.put("arg1", messageType);
    eventProperties.put("arg2", entityName);
    eventProperties.put("arg3", entityInstance);
    eventProperties.put("arg4", messagePeriod);
    eventProperties.put("arg5", parameterValues);

    eventProperties.put("user", portEntity);
    String deviceParameters =
      messageType + "," + entityName + "," +
      entityInstance + "," + messagePeriod + "," +
      parameterValues;
    String deviceMessage = "trigger - " + deviceParameters;
    eventProperties.put(EventConstants.MESSAGE, deviceMessage);
    if (ERROR_DEBUG) {				// debug?
      String deviceIn = EVENT_TOPIC + "(" + deviceParameters + ")";
      Activator.logNote("postEvent", deviceIn);
    }
    postEventService.postEvent(EVENT_TOPIC, eventProperties);
  }

  /**
    Receive data in response to a command, returning the payload if the response
    is valid (or null if not).

    @param command	command for which a response is expected
    @return		payload for valid response (null if invalid)
  */
  public String receiveData(String command) {
    String response = null;			// initialise response
    try {
      Vector<Byte> bytes = new Vector<Byte>();	// initialise byte list
      int byteLength = 0;			// initialise byte count
      while (inputStream.available() != 0) {	// data available?
	byte b = (byte) inputStream.read();	// read bytes repeatedly
	byteLength++;				// increment byte count
	bytes.add(b);				// append byte value
      }
      if (byteLength != 0) {			// data read?
	StringBuffer buffer =			// initialise buffer
	  new StringBuffer(byteLength);
	for (int i = 0; i < byteLength; i++) {	// convert bytes to characters
	  byte b = bytes.get(i);		// get byte
	  buffer.append((char) b);		// append as character
	}
	response =				// get response (if response OK)
	  checkData(command, buffer.toString());
      }
      else if (ERROR_DEBUG)			// nothing read, debug?
	Activator.logError("receiveData", "protocol response to " +
	  commandName(command) + " not received");
    }
    catch (IOException exception) {
      Activator.logError("receiveData", "I/O exception reading response to " +
	commandName(command) + " - " + exception);
    }
    return(response);				// return response
  }

  /**
    Request Circle calibration, returning calibration settings from the
    response (null if calibration not retrieved).

    @param circleAddress	MAC address of Circle (second part only)
    @return			Circle calibration settings
  */
  public Calibration requestCalibration(String circleAddress) {
    Calibration calibration = null;	// initialise no calibration
    String response =				// get calibration response
      sendCommand(DEVICE_CALIBRATION_REQUEST, circleAddress, null);
    if (response != null) {			// response received?
      float gainA = getFloat(response, 20, 8);	// get gains and offsets
      float gainB = getFloat(response, 28, 8);
      float offsetNoise = getFloat(response, 44, 8);
      float offsetTotal = getFloat(response, 36, 8);
      calibration =				// create calibration
	new Calibration(gainA, gainB, offsetNoise, offsetTotal);
    }
    return(calibration);			// return calibration
  }

  /**
    Request a change in the power setting of a Circle, switching it on or off.

    @param circleAddress	circle address
    @param power		true/false to switch on/off
  */
  protected void requestChange(String circleAddress, boolean power) {
    String parameters = power ? "01" : "00";	// set power parameter
    String response =				// get change response
      sendCommand(POWER_CHANGE_REQUEST, circleAddress, parameters);
    if (response == null)			// no response received?
      if (ERROR_DEBUG)				// debug?
	Activator.logError("requestChange",
	  "could not change power state for " + deviceName(circleAddress));
  }

  /**
    Request the clock time, returning the current time HH:MM:SS (null
    if clock not retrieved).

    @param circleAddress	MAC address of Circle (second part only)
    @return			Circle current time (HH:MM:SS)
  */
  public String requestClock(String circleAddress) {
    String time = null;				// initialise no time
    String response =				// get information response
      sendCommand(CLOCK_GET_REQUEST, circleAddress, null);
    if (response != null) {			// response received?
      int hour = getInt(response, 20, 2);	// get hour
      int minute = getInt(response, 22, 2);	// get minute
      int second = getInt(response, 24, 2);	// get second
      // int weekDay = getInt(response, 26, 2);	// get day of week
      time =					// get time adjusted by offset
	getTime(hour, minute, second, clockOffset);
    }
    return(time);				// return time
  }

  /**
    Request Circle energy, returning the power consumption for the last hour
    (time of reading and energy in watt-hours, null if energy not obtained).

    @param circleAddress	MAC address of Circle (second part only)
    @return			energy consumption reading/energy for last hour
  */
  public Energy requestEnergy(String circleAddress) {
    Energy energy = null;			// initialise no energy reading
    int logAddress =				// get current log address
      requestInformation(circleAddress);
    logAddress -= 8;				// go to last filled buffer
    if (logAddress != 0) {			// log address obtained?
      String logAddressString =			// get log address in hex
	String.format("%08X", logAddress);
      Calibration calibration =			// get Circle calibration
	getCalibration(circleAddress);
      if (calibration != null) {		// calibration obtained?
	String response =			// get power buffer response
	  sendCommand(POWER_BUFFER_REQUEST, circleAddress, logAddressString);
	if (response != null) {			// response received?
	  // int year = getInt(response, 20, 2) + 2000; // get year
	  // int month = getInt(response, 22, 2); // get month
	  int minutes = getInt(response, 24, 4);// get minutes from month start
	  String time = getTime(minutes);	// get time from minutes
	  int pulses =				// get pulses for last buffer
	    getInt(response, 28, 8);
	  int energyConsumption =		// get power for last power
	    Math.round(getPower(calibration, pulses, 3600));
	  energy =
	    new Energy(time, energyConsumption);
	  if (ERROR_DEBUG)			// debug?
	    Activator.logNote("circle " + deviceName(circleAddress),
	      "time " + time + " energyConsumption " + energyConsumption);
	}
      }
    }
    return(energy);				// return energy reading
  }

  /**
    Request Circle information, returning current log address from the
    response (0 if device information not retrieved).

    @param circleAddress	MAC address of Circle (second part only)
    @return			Circle current log address
  */
  public int requestInformation(String circleAddress) {
    int logAddress = 0;				// initialise no log address
    String response =				// get information response
      sendCommand(DEVICE_INFORMATION_REQUEST, circleAddress, null);
    if (response != null)			// response received?
      logAddress = getInt(response, 28, 8);	// get last log address
    return(logAddress);				// return log address
  }

  /**
    Request Stick initialisation, returning network status from the response
    (false if status not obtained).

    @return		true/false if network is/is not online
  */
  public boolean requestInitialisation() {
    boolean networkStatus = false;		// initialise network offline
    String response =				// get initialisation response
      sendCommand(STICK_INITIALISE_REQUEST, null, null);
    if (response != null)			// response received?
      networkStatus =				// get network status
	getInt(response, 22, 2) == 1;
    return(networkStatus);			// return network status
  }

  /**
    Request Circle power, returning the 8-second instantaneous power
    consumption (watts, -1 if power not obtained).

    @param circleAddress	MAC address of Circle (second part only)
    @return			instantaneous power consumption (watts)
  */
  public int requestPower(String circleAddress) {
    float powerConsumption = -1.0f;		// initialise no power
    Calibration calibration =			// get Circle calibration
      getCalibration(circleAddress);
    if (calibration != null) {			// calibration obtained?
      String response =				// get power response
	sendCommand(POWER_INFORMATION_REQUEST, circleAddress, null);
      if (response != null) {			// response received?
	// use "getInt(response, 20, 4)" for 1-second power string
	int pulseCount =			// get 8-second pulse count
	  getInt(response, 24, 4);
	powerConsumption =			// get 8-second power
	  getPower(calibration, pulseCount, 8);
      }
    }
    return(Math.round(powerConsumption));	// return power consumption
  }

  /**
    Request setting of a Circle(+) clock to the current time and log address to
    a new value (use null to leave log buffers alone).

    @param circleAddress	circle address
    @param logAddress		log address (null to leave alone)
  */
  private void requestSetting(String circleAddress, String logAddress) {
    Calendar calendar = Calendar.getInstance();	// get current date/time
    calendar.add(Calendar.MINUTE, -clockOffset);// less clock offset in minutes
    int year = calendar.get(Calendar.YEAR);
    int month = calendar.get(Calendar.MONTH) + 1; // Jan supplied as 0 not 1
    int day = calendar.get(Calendar.DAY_OF_MONTH);
    int hour = calendar.get(Calendar.HOUR_OF_DAY);
    int minute = calendar.get(Calendar.MINUTE);
    int second = calendar.get(Calendar.SECOND);
    // get day number in week and minutes since the start of the month
    int weekDay =				// Java 1-7 = Plugwise 0-6
      calendar.get(Calendar.DAY_OF_WEEK) - 1;
    int minutes = 24 * 60 * (day - 1) + 60 * hour + minute;
    String date =				// set Plugwise date
      String.format("%02X%02X%04X", year - 2000, month, minutes);
    if (logAddress == null)			// no log address?
     logAddress = "FFFFFFFF";			// set no change to log buffers
    String time =				// set Plugwise time
      String.format("%02X%02X%02X%02X", hour, minute, second, weekDay);
    String parameters = date + logAddress + time;
    String response =				// get clock setting response
      sendCommand(CLOCK_SET_REQUEST, circleAddress, parameters);
    if (response == null)			// no response received?
      if (ERROR_DEBUG)				// debug?
	Activator.logError("requestSetting",
	  "could not set clock for " + deviceName(circleAddress));
  }

  /**
    Return the response code for the given command code.

    @param command	command code
    @return		response code
  */
  public String responseCode(String command) {
    String response = GENERAL_RESPONSE;		// initialise default code
    if (command.equals(POWER_BUFFER_REQUEST))
      response = POWER_BUFFER_RESPONSE;
    else if (command.equals(DEVICE_CALIBRATION_REQUEST))
      response = DEVICE_CALIBRATION_RESPONSE;
    else if (command.equals(CLOCK_GET_REQUEST))
      response = CLOCK_GET_RESPONSE;
    else if (command.equals(CLOCK_SET_REQUEST))
      response = CLOCK_SET_RESPONSE;
    else if (command.equals(DEVICE_INFORMATION_REQUEST))
      response = DEVICE_INFORMATION_RESPONSE;
    else if (command.equals(POWER_CHANGE_REQUEST))
      response = POWER_CHANGE_RESPONSE;
    else if (command.equals(POWER_INFORMATION_REQUEST))
      response = POWER_INFORMATION_RESPONSE;
    else if (command.equals(STICK_INITIALISE_REQUEST))
      response = STICK_INITIALISE_RESPONSE;
    return(response);
  }

  /**
    Send command to Stick, returning the payload if the response is valid (or
    null if not). This method is synchronised as it is called by independent
    threads, but each command/response pair must be handled separately.

    @param command	module command (null to omit)
    @param address	module address (second partly only, null to omit)
    @param parameters	module parameters (null to omit)
    @return		payload for valid response (null if invalid)
  */
  private synchronized String sendCommand(String command, String address,
   String parameters) {
    String response = null;			// initialise response
    String message = "";			// initialise message
    if (command != null)			// command not null?
      message += command;			// append it
    if (address != null)			// address not null?
      message += startAddress + address;	// append full address
    if (parameters != null)			// parameters not null
      message += parameters;			// append it
    if (ERROR_DEBUG)				// debug?
      Activator.logNote(			// log sent message
	"sendCommand", "sending '" + formatBuffer(message) + "'");
    String crc = getCRCString(message);		// get CRC for message
    String request =				// put request together
      PROTOCOL_HEADER + message + crc + PROTOCOL_TRAILER;
    byte[] bytes = stringToBytes(request);	// convert to hex byte string
    int retryCount = 0;				// initialise retry count
    while (response == null) {			// no valid response yet?
      try {					// try to write request
	retryCount++;				// increment retry count
	outputStream.write(bytes);		// write request
	wait(RECEIVE_DELAY);			// wait before trying read
	response = receiveData(command);	// get response payload
	if (response == null) {			// invalid response?
	  if (retryCount >= retryLimit) {	// at retry limit?
	    Activator.logError("sendCommand",
	      "retry limit reached with " + commandName(command) + " for " +
	      deviceName(address));
	    break;				// leave retry loop
	  }
	  else if (ERROR_DEBUG)			// not at retry limit, debug?
	    Activator.logNote("sendCommand",
	      "retrying " + commandName(command) + " for " +
	      deviceName(address));
	}
      }
      catch (IOException exception) {		// write failed?
	Activator.logError(
	  "sendCommand", "could not send message - " + exception);
      }
    }
    return(response);				// return response payload
  }

  /**
    Set up periodic polling for clock setting, energy consumption and power
    consumption.x

    <ul>

      <li>
	The Circle+ clock is set every 24 hours, starting immediately after this
	method is called.
      </li>

      <li>
	Energy is measured at the given interval, starting at 1 minute past the
	first hour after this method is called. This allows for Circles to
	update their power logs on the hour.
      </li>

      <li>
	Power is measured at the given interval in minutes, starting at this
	interval after this method is called.
      </li>

    </ul>
  */
  private void setPolling() {
    pollTimer = new Timer();			// create poll timer

    clockTask = new PollTask(CLOCK_TASK);	// create clock task
    long clockStart = 0;			// set clock initial wait (msec)
    long clockPeriod =				// get clock interval (msec)
      1000 * 24 * 60 * 60 * energyInterval;
    pollTimer.scheduleAtFixedRate(		// schedule clock polls
      clockTask, clockStart, clockPeriod);

    energyTask = new PollTask(ENERGY_TASK);	// create energy task
    Calendar energyStart = Calendar.getInstance(); // get current date/time
    energyStart.add(Calendar.HOUR, 1);		// start at next hour
    energyStart.set(Calendar.MINUTE, 1);	// poll at 1 minute past hour
    energyStart.set(Calendar.SECOND, 0);	// poll on the minute
    long energyPeriod =				// get energy interval (msec)
      1000 * 60 * energyInterval;
    pollTimer.scheduleAtFixedRate(		// schedule energy polls
      energyTask, energyStart.getTime(), energyPeriod);

    powerTask = new PollTask(POWER_TASK);	// create power task
    long powerStart = 1000 * 60 * powerInterval;// set power initial wait (msec)
    long powerPeriod = powerStart;		// use same delay between polls
    pollTimer.scheduleAtFixedRate(		// schedule power polls
      powerTask, powerStart, powerPeriod);
  }

  /**
    Using a thread to avoid bundle start delays, get calibration settings for
    each Circle and store them in global "calibrationMapping". Use the default
    calibration if it cannot be retrieved. Following calibration attempts,
    set up polling.
  */
  public void startThreads() {
    Thread calibrationThread = new Thread() {
      public void run() {
	Enumeration<String> circleAddresses =	// get circle addresses
	  sensorMapping.keys();
	while (circleAddresses.hasMoreElements()) { // go through calibrations
	  String circleAddress = circleAddresses.nextElement();
	  if (ERROR_DEBUG)			// debug?
	    Activator.logNote("startThreads",
	      "getting calibration for " + deviceName(circleAddress));
	  Calibration calibration =		// get calibration
	    requestCalibration(circleAddress);
	  if (calibration == null)		// calibration not obtained?
	    calibration = calibrationDefault;	// use default calibration
	  calibrationMapping.put(circleAddress, calibration);
	}
	setPolling();				// set up polling
      }
    };
    calibrationThread.start();
  }

  /**
    Convert a string of text to a byte array.

    @param text		character string
    @return		byte array
  */
  private byte[] stringToBytes(String text) {
    int count = text.length();
    byte[] bytes = new byte[count];
    for (int i = 0; i < count; i++) {
      char ch = text.charAt(i);
      byte value = (byte) (ch & 0XFF);
      bytes[i] = value;
    }
    return(bytes);
  }

  /**
    Wait for the given number of seconds.

    @param wait		delay in seconds
  */
  private void wait(int wait) {
    try {
      Thread.sleep(1000 * wait);
    }
    catch (InterruptedException exception) {
      // ignore exception
    }
  }

  /* *************************** PollTask Class **************************** */

  /**
    Handle polling of Circles for energy or power consumption readings.

    @author	Kenneth J. Turner
    @version	1.0: 12/05/11 (KJT)
  */
  public class PollTask extends TimerTask {

    /** Reading type (energy, power) */
    private String taskType;

    /**
      Poll Circles and post power events.

      @param taskType	type of reading (energy, power)
    */
    public PollTask(String taskType) {
      this.taskType = taskType;
    }

    /**
      Poll Circles and post consumption events.
    */
    public void run() {
      pollCircles(taskType);
    }

  }

}

