/******************************************************************
 *  (C) Copyright 1994; dit/upm
 *  Distributed under the conditions stated in the
 *  TOPO General Public License (see file LICENSE)
 ******************************************************************
 *  $Log$
 ******************************************************************/

/******************************************************************
 *  
 *  Santiago Pavon Gomez
 *
 *  30 Nov 1990
 *
 *  Management of LOTOS behaviour expressions
 *
 *  LOTOS has nullary operators (e.g stop), unitary operators
 *  (e.g action prefix) and binary operators (e.g. parallel).
 *  An exception is made for the choice expression.
 *  This interface considers the choice expression of n behaviours
 *  (n-1 choice operators B1[]B2[]...[]Bn) to be an n-ary operator
 *  ( only one operator ! ). 
 *  The access to an operand of all LOTOS operators is homogeneous.
 *
 *  Behaviours, alike LOTOS, always referres to trees.
 *  When behaviours are copied its attributes are shared unless
 *  otherwise stated.
 *
 *  COMPILATION FLAGS:
 *     SDEBUG : activate debugging checks
 *  
 *  TYPE DEFINITIONS:
 *     BehTyp : behaviour type 
 *  
 *  FUNCTION DEFINITIONS:
 *     LookTypeB          : return type of operator b 
 *     PutNameB           : put name to an operator
 *     NumArgB            : number of operands of an operator
 *     LookArgB           : look at argument <n> of an operator 
 *     GetArgB            : get argument <n> of an operator
 *     PutArgB            : put argument <n> to an operator
 *     LookNameB          : return the name of an operator b
 *     MakeB              : make an operator, given a type and a name
 *     AddArgB            : add next operand to an operator
 *     OwnersB            : number of sharers of a behaviour
 *     FreeB              : free a behaviour
 *     ShareB             : increment the copy counter of a behaviour
 *     UnshareB           : decrement the copy counter of a behaviour
 *     SharedB            : say whether an operator is being shared
 *     GetB               : subtract a copy of a behaviour
 *     GetUntilActB       : subtract a copy of a behaviour until actions
 *     CopyOperB          : copy an operator
 *     GetOperB           : subtract a copy of an operator
 *     CopyB              : copy a behaviour
 *     CopyUntilActB      : copy a behaviour
 *     UnshareAttrsB      : unshare the attributes of a behaviour
 *     UnshareAttrsUntilActB : unshare the attributes of a behaviour
 *                           until actions
 *     DrawB              : draw a behaviour
 *     DB                 : draw a behaviour in stdout (fast call)
 *     DBN                : draw a behaviour in stdout until depth (fast call)
 *     GlueB              : glue a behaviour into another
 *     GlueArgB           : glue an operand into its operator
 *     FreeArgB           : free an operand of a behavior
 *     ApplyProcChoiceB   : apply a procedure to a choice
 *     ApplyFuncChoiceB   : apply a function to a choice
 *     SubtractChoiceB    : subtract an operand from a choice
 *     Begin_VisitB       : begin a visit
 *     VisitedB           : say if an operator has been visited before
 *                          and mark it as visited
 *     UnVisitB           : unvisit an operator
 *     End_VisitB         : end of visit 
 *
 *  LOG:
 *
 ******************************************************************/

#ifndef babeh_H
#define babeh_H

#include "licell.h"

typedef PCellTyp BehTyp;


/* LookTypeB
 * Return the type of operator of the behaviour b.
 */
#define LookTypeB(b) (LookType(b))


/* PutNameB
 * Put the name d to the operator b.
 */
extern void PutNameB( /* b , d */ );
/*  BehTyp         b;
    DescriptorTyp  d;  */


/* NumArgB
 * Current number of arguments of the behaviour b.
 */
extern int NumArgB( /* b */ );
/*  BehTyp  b;  */


/* LookArgB
 * Look at the n-th argument of the behaviour b.
 */
extern BehTyp LookArgB( /* b , n */ );
/*  BehTyp  b; 
    int     n;  */


/* GetArgB
 * Get the n-th argument of the behaviour b.
 * The n-th argument will be NULL after calling this function.
 */
extern BehTyp GetArgB( /* b , n */ );
/*  BehTyp   b; 
    int      n;  */


/* PutArgB
 * Put the n-th argument to the behaviour b.
 * The n-th argument must be NULL.
 */
extern void PutArgB( /* b , a , n */ );
/*  BehTyp  b, a; 
    int      n;  */


/* LookNameB
 * Name of the behaviour operator b.
 * This descriptor can be an index in the table of gates, processes, etc ...
 */
extern DescriptorTyp LookNameB( /* b */ );
/*  BehTyp  b;  */


/* MakeB
 * Make a behaviour operator of type k with a name d . 
 * Its arguments can be added with AddArgB or PutArgB later.
 */
extern BehTyp MakeB( /* d, k */ );
/*  DescriptorTyp  d;
    CellTypeTyp    k;  */


/* AddArgB
 * Add an argument to the behaviour operator b.
 */
extern void AddArgB( /* b , a */ );
/*  BehTyp  b,a; */


/* OwnersB
 * Return the number of parents sharing a behaviour b.
 */
extern int OwnersB( /* b */ );
/* BehTyp b; */


/* FreeB
 * Free a behaviour b.
 */
extern void FreeB( /* b */ );
/* BehTyp b;  */    


/* ShareB
 * Increment the copy counter of the behaviour in one unit.
 */
extern BehTyp ShareB( /* b */ );
/* BehTyp b;  */    


/* UnshareB
 * Decrement the copy counter of a behaviour in one unit.
 */  
extern BehTyp UnshareB( /* b */ );
/*     BehTyp b; */


/* SharedB
 * Say if the cell is shared by more than one owners
 */
extern boolean SharedB(/* b */);
/*     BehTyp b; */


/* GetB
 * Subtract a copy of the behaviour b. i.e. copy & free old behaviour
 * Cells are copied and attributes are shared.
 */
extern BehTyp GetB( /* b */ );
/* BehTyp b;  */    


/* GetUntilActB
 * Subtract a copy of the behaviour b until actions are found.
 */
extern BehTyp GetUntilActB( /* b */ );
/*  BehTyp b;   */    


/* CopyOperB
 * Copy the operator b and share its arguments and attributes.
 */
extern BehTyp CopyOperB( /* b */ );
/* BehTyp b;  */    


/* GetOperB
 * Subtract a copy of the operator b.
 * The arguments are shared.
 */
extern BehTyp GetOperB( /* b */ );      
/* BehTyp b; */    


/* CopyB
 * Copy the cells and shares the attributes of the behaviour b
 */
extern BehTyp CopyB( /* b */ );
/* BehTyp b;  */    


/* CopyUntilActB
 * Copy the cells and share the attributes of the behaviour b
 * until actions are found.
 */
extern BehTyp  CopyUntilActB( /* b */ );
/*  BehTyp b;   */    


/* UnshareAttrsB
 * Unshare the attributes of a behaviour b until the end of 
 * the b or a shared subbehaviour is found.
 */
extern void UnshareAttrsB( /* b */ );
/*  BehTyp  t;  */


/* UnshareAttrsUntilActB
 * Take the behaviour b and unshare its attributes until an action is found.
 * b must have been unshared until actions.
 */     
extern void UnshareAttrsUntilActB( /* b */ );
/* BehTyp  b; */


/* DrawB
 * Print the behaviour b in internal format.
 * pstr is the function used to print strings.
 */
extern void DrawB( /* b, pstr */ );
/* BehTyp  b;
   void (*pstr)(); */


/* DB
 * Draw a behaviour using printTarget.
 */
extern void DB( /* b */ );
/* BehTyp b; */


/* DBN
 * Draw a behaviour until depth cells using printTarget.
 */
extern void DBN( /* b, depth */ );
/*     BehTyp b;
       int    depth; */


/* GlueB
 * Glue the behaviour b2 into b1.
 * After glueing b2 occupies the memory of b1, so that all pointers
 * to b1 point now to b2, and b1 does not exist any more.
 */
extern void GlueB( /* b1, b2 */ );
/* BehTyp b1,b2; */


/* GlueArgB
 * Glue the argument n of b into b.
 */
extern void GlueArgB( /* b, n */ );
/* BehTyp b; 
   int    n; */


/* FreeArgB
 * Free the argument n of the behaviour b.
 */
extern void FreeArgB( /* b, n */ );
/* BehTyp b; 
   int    n; */


/* ApplyProcChoiceB
 * Apply a procedure to all the branches of a behaviour choice operator.
 */
extern void ApplyProcChoiceB( /* b, p */ );
/*     BehTyp  b,
       void   *p(); */


/* ApplyFuncChoiceB
 * Apply a function to all the branches of a behaviour choice operator
 */
extern void ApplyFuncChoiceB( /* b, f */ );
/*     BehTyp b, (*f)(); */


/* SubtractChoiceB
 * Takes out the n-th operand from the behaviour choice b.
 * The alternativeC the operand hangs from is removed.
 */
extern BehTyp SubtractChoiceB( /* b, n */ );
/*       BehTyp b; 
         int    n; */


/******************************************************************
 *                                                                *
 *                   Visits to behaviours                         *
 *            Running behaviours avoiding revisiting              *
 *                                                                *
 *    NOTE : side effect : the copy counters will be changed      *
 *           temorarily during a visit.                           *
 *                                                                *
 ******************************************************************/

/* Begin_VisitB
 * Start a visit to a behaviour .
 */
extern void Begin_VisitB();


/* VisitedB
 * Declare a cell as "visited".
 * If it had already been visited return TRUE else FALSE.
 * Those cells with (|copy|>1) are stored and their copy fields get
 * changed of signus.
 * Therefore End_VisitB should be called to undo changes on the behaviour.
 */
extern boolean VisitedB( /* b */ );
/*       BehTyp b; */


/* UnVisitB
 * Declare a cell as "not visited".
 * Set its copy signus to positive.
 */
extern void UnVisitB( /* b */ );
/*       BehTyp b; */


/* End_VisitB
 * End a visit to a behaviour .
 * Release track of visited cells and restore altered copy fields.
 */
extern void End_VisitB();


#endif
