/******************************************************************
 *  (C) Copyright 1994; dit/upm
 *  Distributed under the conditions stated in the
 *  TOPO General Public License (see file LICENSE)
 ******************************************************************
 *  $Log$
 ******************************************************************/

/******************************************************************
 *  
 *  Santiago Pavon Gomez
 *
 *  17 Jan 1990
 *
 *  Management of LOTOS data type expressions
 *
 *  LOTOS data type expressions terms can be either variables or
 *  operations. Operations can have a list of operands.
 *  Variables can have one (hidden) value called Parameterized Value.
 *  That is the value the variable is symbolically standing for,
 *  during parameterised expansions.
 *  Data type subexpressions can be shared.
 *  
 *  COMPILATION FLAGS:
 *     SDEBUG : activate debugging
 *  
 *  TYPE DEFINITIONS:
 *     ExprTyp : expression type
 *  
 *  FUNCTION DEFINITIONS:
 *     InitE            : init this module
 *     StatE            : statistics of this module cell use
 *     StatisticE       : statistics on cell used in a expression
 *     SetPrintParamE   : set print mode for parameterized values
 *     PrintE           : print a expression
 *     ExprToStringE    : transform a expression into a string
 *     DrawE            : draw an expression
 *     PE               : print an expression in stdout
 *     DE               : draw an expression in stdout 
 *     IsVariableE      : say whether a expression term is a variable
 *     IsOperationE     : say whether a expression term is a operation
 *     LookNameE        : look at the name of the expression term
 *     LookTypeE        : look at the type of the expression term
 *     LookSortE        : look at the sort of the expression term
 *     ChangeNameE      : change the name of an expression term
 *     IsConstE         : say whether there is no variable in an expression 
 *     IsVoidE          : say whether an expression is null 
 *     EqualE           : expression equality
 *     MakeE            : make an expression, given a name and a type
 *     NumArgE          : number of operands of a term 
 *     LookArgE         : look at the n-th operand of a term
 *     GetArgE          : get the n-th operand of a term
 *     PutArgE          : put the n-th operand to a term
 *     AddArgE          : add an operand to a term
 *     Apply_Proc_ArgsE : apply a procedure to the operands of a term
 *     Apply_Func_ArgsE : apply a function to the operands of a term
 *     OwnersE          : number of this term sharers
 *     ShareE           : increment the sharing counter
 *     UnshareE         : decrement the sharing counter
 *     CopyE            : copy an expression
 *     GetE             : get an expression ( copy and free )
 *     GlueE            : glue an expression into another
 *     FreeE            : free an expression
 *     VarsInExpr       : list of vars used in an expression
 *     LookPVarE        : look at the parameterized value of a variable 
 *     GetPVarE         : get the parameterized value of a variable 
 *     PutPVarE         : put the parameterized value to a variable 
 *     CopyUntilVarE    : copy an expression until variables
 *     UnParVarE        : unparameterize a variable
 *     FreeUntilVarE    : free a expression until vars
 *     Begin_VisitE     : begin a visit to expressions
 *     VisitedE         : say if the current term has been visited before
 *     UnVisitE         : unvisit a term
 *     End_VisitE       : end of the visit
 *
 *  LOG:
 *     21/09/9. santiago. 
 *     New interface with sharing of expressions. 
 *
 ******************************************************************/


#ifndef baexpr_H
#define baexpr_H

#include "lilists.h"


#define OperationC 0
#define VariableC  1


/* 
 *   Expression type definition.
 *
 *   Expressions are built as trees, where each node has a name, 
 *   a list of operands and a counter of owners ( sharers of this cell ).
 *
 *   The name is an integer greater than 0 for operations, and less 
 *   than 0 for variables.
 *
 *   Nodes in the expressions are ExprTyp structs.
 *
 */
typedef struct exprNodeTyp { int      name;
                             int      copy;
                             ListTyp  sons;
                           } *ExprTyp;


/* InitE
 * Init this module
 */
extern void InitE();


/* StatE
 * Give statistics about the expression cells allocated and released.
 * nc  : number of expression cell queries originated by this module. 
 * rc  : expression cell releases.
 * mc  : max used expression cells
 * ntn : number of tnodes queries originated by this module. 
 * rtn : tnode releases.
 * mtn : max used tnodes
 */
extern void StatE(/* nc, rc, mc, ntn, rtn, mtn */);
/*    int *nc, *rc, *mc, *ntn, *rtn, *mtn; */


/* StatisticE
 * Statistic of node use in a single expression.
 *  nv,vv and ov are the virtual values of number of list nodes,
 *   variable and operation cells, i.e. without paying attention to the
 *   sharing feature.
 *  nr,vr and or are the real values, i.e. nodes and cells already counted
 *   in other expressions are not counted again here.
 */
extern void StatisticE( /* e, nv,vv,ov,nr,vr,or */ );
/* ExprTyp e;
   int     *nv, *vv, *ov, *nr, *vr, *or; */


#define PRINT_PARAM      1
#define PRINT_ONLY_PARAM 2
#define PRINT_NO_PARAM   3

/* SetPrintParamE
 * Set the printing mode for the parameterized values in the expressions for
 * all the following calls to PrintE or ExprToStringE.
 *
 *  Modes :                                                         Examples:
 *
 *  PRINT_PARAM      -> the parameterized value
 *                      is printed as a comment. (default)               x(*E*)
 *  PRINT_ONLY_PARAM -> only the parameterized value is printed.         E
 *  PRINT_NO_PARAM   -> no parameterized value is printed.               x
 *
 */
extern void SetPrintParamE( /* mode */ );
/*       boolean mode; */


/* PrintE              
 * Print the expression e.
 * pstr is the function used to print strings.
 */
extern void PrintE ( /* e, pstr */ );
/*  ExprTyp  e ;
    void     (*pstr)(); */


/* ExprToStringE              
 * Return a string that represents the expression e.
 */
extern char* ExprToStringE ( /* e */ );
/*  ExprTyp  e ;  */


/* DrawE
 * Draw an expression e.
 * pstr is the function used to print strings.
 */
extern void DrawE( /* e, pstr */ );
/* ExprTyp    e;
   void     (*pstr)(); */


/* PE
 * Print an expression in stdout
 */
extern void PE( /* e */ );
/* ExprTyp e; */


/* DE
 * Draw an expression in stdout
 */
extern void DE( /* e */ );
/* ExprTyp e; */


/* LookNameE
 * Return the name of the expression e.
 * This name is an index in the tables of operation and variables.
 * For variables the name is a negative value.
 */
extern int LookNameE( /* e */ );
/*  ExprTyp  e;  */


/* LookTypeE
 * Return the type of the expression e: VariableC or OperationC
 */
extern int LookTypeE( /* e */ );
/*  ExprTyp  e;  */


/* LookSortE
 * Return the sort of the expression ve.
 */
extern int LookSortE( /* ve */ );
/* ExprTyp  ve;  */


/* ChangeNameE
 * Change the name of an expression e
 */
extern void ChangeNameE( /* e, new_name */ );
/* ExprTyp  e; 
   int      new_name; */


/* IsVariableE
 * Return TRUE if ve is a variable.
 */
#define IsVariableE(ve) (LookTypeE(ve)==VariableC)


/* IsOperationE
 * Return TRUE if ve is not a variable.
 */
#define IsOperationE(ve) (LookTypeE(ve)==OperationC)



/* IsConstE
 * Return TRUE if there is no variable in the expression ve.
 */
extern boolean IsConstE( /* ve */ );
/* ExprTyp  ve;  */


/* IsVoidE
 * Return TRUE if the expression is void.
 */
extern boolean IsVoidE( /* e */ );
/*  ExprTyp  e;  */


/* EqualE
 * Return TRUE if e1 and e2 are equal, FALSE otherwise.
 */
extern boolean EqualE ( /* e1,e2 */ );
/* ExprTyp e1,e2; */


/* MakeE
 * Make an expression. If the expression is an operation then
 * the arguments can be added with AddArgE later.
 *  k indicates the expression type.
 *  d is the descriptor index in the table of operation or variables.
 */
extern ExprTyp MakeE( /* d, k */ );
/*  int  d;
    int  k;  */


/* NumArgE
 * Return the number of arguments of the operation e.
 */
extern int NumArgE( /* e */ );
/*  ExprTyp  e;  */


/* LookArgE
 * Look at the n-th argument of the operation e.
 */
extern ExprTyp LookArgE( /* e, n */ );
/*  ExprTyp  e; 
    int      n;  */


/* GetArgE
 * Get the n-th argument of the operation e.
 */
extern ExprTyp GetArgE( /* e, n */ );
/*  ExprTyp  e; 
    int      n;  */


/* PutArgE
 * Put a as the n-th argument of the operation e.
 */
extern void PutArgE( /* e , a , n */ );
/*  ExprTyp  e, a; 
    int      n;  */


/* AddArgE
 * Add the argument a to the operation e.
 */
extern void AddArgE( /* e , a */ );
/*  ExprTyp  e, a; */


/* Apply_Proc_ArgsE
 * Apply procedure p to all the arguments of the operation e.
 */
extern void Apply_Proc_ArgsE( /* e, p */ );
/* ExprTyp e;
   void    (*p)();  */


/* Apply_Func_ArgsE
 * Replace each argument a of e for f(a).
 */
extern void Apply_Func_ArgsE( /* e, f */ );
/* ExprTyp    e;
   ExprTyp    (*f)();  */


/* OwnersE
 * Number of entities sharing the expression e.
 */
extern int OwnersE( /* e */ );
/*  ExprTyp  e;  */


/* ShareE
 * Increment the sharing counter of the expression e in one unit.
 */
extern ExprTyp ShareE( /* e */ );
/*  ExprTyp  e;  */


/* UnshareE
 * Decrement the sharing counter of the expression e in one unit.
 */  
extern ExprTyp UnshareE( /* e */ );
/*  ExprTyp  e;  */


/* CopyE
 * Copy an expression.
 */
extern ExprTyp CopyE( /* e */ );
/*  ExprTyp  e;  */


/* GetE
 * Subtract a copy of the expression e ( copy e and free e ).
 */
extern ExprTyp GetE( /* e */ );
/* ExprTyp  e; */ 


/* GlueE
 * Expression e2 is placed in the memory used by e1.
 * The old e1 disappears.
 *  NOTE: e1 can be the argument of another expression but e2 cannot.
 */
extern void GlueE( /* e1,e2 */ );
/*  ExprTyp  e1,e2;  */


/* FreeE
 * Free the expression e.
 */
extern void FreeE( /* e */ );
/*  ExprTyp  e;  */


/* VarsInE
 * Return a list with the variables used in the expression e.
 */
extern ListTyp VarsInE( /* e */ );
/* ExprTyp e; */


/* LookPVarE
 * Return the parameterized value of the variable v.
 */
extern ExprTyp LookPVarE( /* v */ );
/* ExprTyp v;  */


/* GetPVarE
 * Get the parameterized value of the variable v.
 * This value will be null after this function.
 */
extern ExprTyp GetPVarE( /* v */ );
/*  ExprTyp  v; */


/* PutPVarE
 * Put the parameterized value p to the variable v.
 * The variable v can not already have a parameterized value.
 */
extern void PutPVarE( /* v , p */ );
/* ExprTyp  v,p;  */


/* CopyUntilVarE
 * Copy an expression until variables
 * (without copying the parameterized values).
 */
extern ExprTyp CopyUntilVarE(/* e */);
/*       ExprTyp  e;  */


/* UnParVarE
 * Unparameterize a variable. Example : UnParVarE(x(*succ(0)*)) => succ(0)
 * The parameterized value is glued onto the variable and the remainder
 * expression cell is disposed.
 */
extern void UnParVarE(/* v */);
/*       ExprTyp v; */


/* FreeUntilVarE
 * Free the expression e until variables.
 * The parameterized value is simply ignored.
 */
extern void FreeUntilVarE(/* e */);
/*       ExprTyp  e; */


/******************************************************************
 *                                                                *
 *                   Visits to expressions                        *
 *            Running expressions avoiding revisiting             *
 *                                                                *
 *    NOTE : side effect : the copy counters will be changed      *
 *           temorarily during a visit.                           *
 *                                                                *
 ******************************************************************/


/* Begin_VisitE
 * Start a visit to expressions.
 * This function must be invoked before exploring the expressions.
 */
extern void Begin_VisitE();


/* VisitedE
 * Declare a expression cell as "visited".
 * If it had already been visited return TRUE else FALSE.
 * Those cells with (|copy|>1) are stored and their copy fields get
 * changed of signus.
 * After visiting all the expressions, End_VisitE should be called to 
 * undo the changes on the expressions.
 */
extern boolean VisitedE( /* e */ );
/* ExprTyp e; */


/* UnVisitE
 * Declare a expression cell as "not visited".
 * Set its copy signus to positive.
 */
extern void UnVisitE( /* e */ );
/* ExprTyp e; */


/* End_VisitE
 * End the visit to expressions.
 * Release track of visited cells and restore altered copy fields.
 */
extern void End_VisitE();

#endif













