/***********************************
  (C) Copyright 1992-1993; dit/upm
  Distributed under the conditions stated in the
  TOPO General Public License (see file LICENSE)
  ***********************************
  $Log:  basynset.h,v $
  * Revision 1.2  92/05/20  15:44:09  lotos
  * data type included
  * 
  * Revision 1.1  92/01/14  13:12:50  lotos
  * Initial revision
  * 
  ***********************************/

/***********************************
  
  David Larrabeiti Lopez
  
  Dec 13, 1991
  
  Definition of synchronization gate sets
  and functions to manage them.
  Unlike normal gate sets , synchronizations gate set includes the
  idea of full set : || as the whole of declared gates.
  These sgs are used in the interleaved expansion.
  
  COMPILATION OPTIONS: The behaviour of this module can be modified
  by the following compilation flags:
  
  SDEBUG : activate debugging mode
  
  TYPE DEFINITIONS:
  SyncGateSetTyp, *PSyncGateSetTyp : synchronization gate sets
  
  FUNCTION DEFINITIONS:
  CreateSS : create an empty synchronization set ss
  FreeSS : free a ss
  CopySS : copy a ss
  DrawSS : draw a ss ( internal format )
  GateInSS : is this gate in the ss ?
  JunctionSS : ss union
  IntersSS : ss intersection
  DiffsSS : ss difference
  EqualSS : are these ss equal ?
  IsEmptySS : is this ss empty ? (*)
  InclSS : is this ss included in another ss ?
  AddSS : add a gate to a ss
  RemoveSS : remove a gate from a ss
  PrintSS : print a ss
  SS_to_Parallel : parallel cell equivalent to ss
  Parallel_to_SS : ss equivalent to parallel
  GateSet_to_SS : ss equivalent to gate set
  IsFullSyncSS : is this ss full ?
  IsPureInterleavingSS : is this ss empty ? (*)
  IsPartialSyncSS : is this ss not empty and not full ?
  FullSS : return a full ss.
  
  ************************************/


#ifndef basyncset_H
#define basyncset_H


#include "listdh.h"
#include "baattr.h"

typedef struct { DescriptorTyp syncKind;
                 GateSetTyp    gs;
               } SyncGateSetTyp, *PSyncGateSetTyp;


/* CreateSS
 * Create an empty SyncGateSet
 */
extern PSyncGateSetTyp CreateSS();


/* FreeSS
 * Frees the memory used by the SyncGateSet SS.
 */
extern void FreeSS( /*  */ );
/*  PSyncGateSetTyp  egs; */


/* CopySS
 * Copies the SyncGateSet egs.
 */
extern PSyncGateSetTyp CopySS( /* egs */ );
/* PSyncGateSetTyp egs; */


/* DrawSS
 * Draws the Sync Gate Set egs.
 * "pstr" is the function used to print strings.
 */
extern void DrawSS( /* egs,pstr */ );
/*  PSyncGateSetTyp  egs;
    void (*pstr)(); */


/* GateInSS
 * Check if an event belongs to a Sync gate set.
 * exit always belongs.
 * i never.
 * a gate sometimes.
 */
extern boolean GateInSS(/* g,egs */);
/*     BehTyp g;
       PSyncGateSetTyp egs; */


/* JunctionSS
 * Performs the union betwen two ext gate sets.
 */
extern PSyncGateSetTyp JunctionSS( /* egs1,egs2 */ );
/* PSyncGateSetTyp egs1,egs2; */


/* IntersSS
 * Performs the union between two ext gates sets
 */
extern PSyncGateSetTyp IntersSS( /*egs1,egs2*/ );
/* PSyncGateSetTyp egs1,egs2 */


/* DiffsSS
 * Perform the differents between two ext gates sets
 */
extern PSyncGateSetTyp DiffsSS( /*egs1,egs2 */);
/* PSyncGateSetTyp egs1,egs2; */


/* EqualSS
 * Return TRUE if the gates are equal
 */
extern boolean EqualSS( /*egs1,egs2 */);
/* PSyncGateSetTyp egs1,egs2;  */


/* IsEmptySS
 * Returns true if egs is a empty gate set, else false
 */
extern boolean IsEmptySS( /* egs */ );
/* PSyncGateSetTyp  egs; */


/* InclSS
 * Return TRUE if a gate set is included in another
 */
extern boolean InclSS(/*egs1,egs2*/);
/* PSyncGateSetTyp egs1,egs2;  */


/*ClearSS
 * Return 0
 */
extern PSyncGateSetTyp ClearSS( /*egs*/ );
/* PSyncGateSetTyp egs;  */


/* AddSS
 * Add a gate to ext gs
 */
extern PSyncGateSetTyp AddSS(/*egs,g*/);
/* PSyncGateSetTyp egs;
   DescriptorTyp g;    */


/* RemoveSS
 * Remove a gate of ext gate set
 */
extern PSyncGateSetTyp RemoveSS(/*egs,g*/);
/* PSyncGateSetTyp egs;
   DescriptorTyp g;    */


/* PrintSS
 * Prints the names of the gates given for a gate set
 * "pstr" is the function used to print strings.
 */
extern void PrintSS( /*egs,pstr*/ );
/*  PSyncGateSetTyp egs;
    void (*pstr)(); */


/* SS_to_Parallel
 * return the parallel cell referred to by an egs
 */
extern BehTyp SS_to_Parallel(/* egs */);
/*     PSyncGateSetTyp egs; */


/* Parallel_to_SS
 * return the extended gate set associated to a parallel cell.
 */
extern PSyncGateSetTyp Parallel_to_SS(/* b */);
/*     BehTyp b; */


/* GateSet_to_SS
 * return the extended gate set associated to a gate set gs.
 */
extern PSyncGateSetTyp GateSet_to_SS (/* gs */);
/*     GateSetTyp gs; */


/* IsFullSyncSS
 */
extern boolean IsFullSyncSS(/* egs */);
/*     PSyncGateSetTyp egs; */


/* IsPureInterleavingSS
 */
extern boolean IsPureInterleavingSS(/* egs */);
/*     PSyncGateSetTyp egs; */


/* IsPartialSyncSS
 */
extern boolean IsPartialSyncSS(/* egs */);
/*     PSyncGateSetTyp egs; */


/* FullSS
 * return a full ss.
 */
extern PSyncGateSetTyp FullSS();


/* InitSS
 * Init the SS module
 */
extern void InitSS();


/* StatSS
 * Number of asked  and released sgs nodes
 */
extern void StatSS(/* new, released */);
/*       int * new, *released; */


#endif
