/***********************************
  (C) Copyright 1992-1993; dit/upm
  Distributed under the conditions stated in the
  TOPO General Public License (see file LICENSE)
  ***********************************/

/***********************************
 * 
 * David Larrabeiti Lopez
 * 
 * 22 Jan 1991
 * 
 * Module to compare syntactically two behaviours
 * (or two attribute values), except for variables
 * renaming . A variable table keeps track of the
 * necessarily one-to-one correspondence between
 * the variables of both behaviours ( or attribute
 * values ), plus some additional information on
 * which of the variables are being defined.
 * 
 * Use :
 
 * boolean x1_equal_x2;

 * Init_VPTable();
 * x1_equal_x2 = Comp_X( x1, x2 );
 * 
 * COMPILATION OPTIONS: The behaviour of this module can be modified
 * by the following compilation flags:
 * 
 * TYPE DEFINITIONS:
 * 
 * FUNCTION DEFINITIONS:
 *   Comp_Expr      : compares two expressions
 *   Comp_Offer     : compares two offers
 *   Comp_SortOL    : compares the sorts of two offers lists
 *   Comp_OLG       : compares two gate offers lists
 *   Comp_OLE       : compares two exit offers lists
 *   Comp_EL        : compares two expressions lists
 *   Comp_VL        : compares two variables lists
 *   Comp_VAL_let   : compares two variable assignments lists of let
 *   Comp_VAL_plet  : compares two variable assignments lists of plet
 *   Comp_Beh       : compares two behaviours
 *   Init_VPTable   : initialize and reset VP table to start a new comparison
 *   Print_VP       : prints the VP table into a file
 *   Store_VP       : Looks up & store variables in the VP table
 *                    There can be parameterized values.
 *   StoreD_VP      : Looks up & store variables descriptors in the VP table.
 *                    There are not parameterized values.
 *   Mk_VPTable_VL1 : Makes a variable list with the first components 
 *                    of the VP table that remain undefined.
 *   Mk_VPTable_VL2 : Makes a variable list with the second components 
 *                    of the VP table that remain undefined.
 *   
 *************************************/


#ifndef excomp_H
#define excomp_H


/* Comp_Expr
 * Compares two Expressions. Returns TRUE iff the comparison succeeds.
 * This happens when e1 and e2 are equal except for one-to-one variable
 * renaming . This renaming  updates the varpair table and
 * must be coherent with that table.
 */
extern boolean Comp_Expr(/* e1, e2, def */);
/*     ExprTyp e1,e2;
       boolean def;   */


/* Comp_Offer
 * Compares two Offers. Returns TRUE iif the comparison succeeds.
 */
extern boolean Comp_Offer(/* o1, o2 */);
/*     OfferListTyp o1,o2;*/


/* Comp_SortOL
 * Compares the sorts of the expressions of two OfferLists
 * Returns TRUE iff the comparison succeeds.
 */
extern boolean Comp_SortOL ( /* ol1, ol2 */);
/* OfferListTyp ol1,ol2; */


/* Comp_OLG
 * Compares two OfferLists in two gates.
 * Returns TRUE iif the comparison succeeds.
 */
extern boolean Comp_OLG(/* ol1, ol2 */);
/*     OfferListTyp ol1,ol2;*/


/* Comp_OLE
 * Compares two OfferLists in two exit cells.
 * Returns TRUE iif the comparison succeeds.
 * The offers of kind QUESTION are ignored.
 */
extern boolean Comp_OLE(/* ol1, ol2 */);
/*     OfferListTyp ol1,ol2;*/


/* Comp_EL
 * Compares two ExprLists. Returns TRUE iif the comparison succeeds.
 */
extern boolean Comp_EL(/* el1, el2 */);
/*     OfferListTyp el1,el2;*/


/* Comp_VL
 * Compares two VarLists. Returns TRUE iff the comparison succeeds.
 */
extern boolean Comp_VL(/* vl1, vl2 */);
/*     ExprListTyp vl1,vl2;*/


/* Comp_VAL_let
 * Compares two VarAssignLists of let. 
 * Returns TRUE iff the comparison succeeds. 
 */
extern boolean Comp_VAL_let(/* val1, val2 */);
/*     VarAssignListTyp val1,val2;*/


/* Comp_VAL_plet
 * Compares two VarAssignLists of plet.
 * Returns TRUE iff the comparison succeeds.
 */
extern boolean Comp_VAL_plet(/* val1, val2 */);
/*     VarAssignListTyp val1,val2;*/


/* Comp_Beh
 * Compares two behaviour trees and the attributes in their cells:
 *           PA, VALA, OLA, ELA, GLA, GSA & RFLA.
 * Returns TRUE only if the comparison succeeds.This happens when these behs
 * are equal except for one-to-one variable renaming.
 * The comparison does not stop until all the free variables of both behaviours
 * have been captured.
 */
extern boolean Comp_Beh(/* b1,b2 */);
/*     BehTyp b1,b2;*/



/******************************************************************
 *                                                                *
 *                Table of Pairs of Variables                     *
 *                                                                *
 ******************************************************************/

/*  
 * The following section manages a table of variable pairs
 * which keeps track of a possible var renaming
 * while comparing two behaviours with Comp_Beh.
 * Therefore, it is checked the one-to-one relationship
 * between the names of the variables being stored.
 * If the comparison succeeds then the list of vars renaming
 * is returned if so asked for.
 * 
 *  Sequence of Operation:
 *
 *    Init_VPTable();
 *    (* begin comparison *)
 *    Store_VP(x,y,TRUE)...
 *    (* end comparison   *)
 *    if ( the_comparison_succeeds )
 *      {
 *        vl1 = Mk_VPTable_VL1();
 *        vl2 = Mk_VPTable_VL2();
 *       }
 *    (* release table *)       
 *    Init_VPTable();
 *
 *  This functions are normally only used by the comparison functions.
 *
 *
 */


/* InitVP
 * Init this module
 */
extern void InitVP();


/* StatVP
 * Number of asked  and released vp nodes
 */
extern void StatVP(/* new, released */);
/*       int * new, *released; */


/* Init_VPTable()
 * Init a VP table
 */
extern void Init_VPTable();


/* Print_VP
 * Prints the VP table.
 * "pstr" is the function used to print strings.
 */
extern void Print_VP( /* pstr */ );
/* void (*pstr)(); */


/* Store_VP
 * Looks up (v1,v2,def) in the table and :
 * if found update def ( only in the sense FALSE->TRUE ) and return TRUE.
 * if (v1,v3,*) or (v3,v2,*) is found ,the comparison fails => return FALSE.
 * else , (v1,v2,def) is appended to the table and return TRUE.
 */
extern boolean Store_VP( /* v1,v2,def */ );
/* ExprTyp v1,v2;
   boolean def; */


/* StoreD_VP
 * Like Store_VP but v1 and v2 are descriptorTyp, and
 * there are not parameterized values.
 */
extern boolean StoreD_VP( /* v1,v2,def */ );
/* DescriptorTyp v1,v2;
   boolean def; */


/* Mk_VPTable_VL1
 * Makes a varlist with the first components of the pairs of variables that 
 * remain undefined in the varpairtable.
 */
extern ExprListTyp Mk_VPTable_VL1();


/* Mk_VPTable_VL2
 * Makes a varlist with the second components of the pairs of variables that 
 * remain undefined in the varpairtable.
 */
extern ExprListTyp Mk_VPTable_VL2();


#endif






